/****************************************************************************/
/* Copyright 1992 MBARI                                                     */
/****************************************************************************/
/* $Header$		    */
/* Summary  : Program to extract CTD data from Seabird dump		    */
/* Filename : getctd.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision$							    */
/* Created  : 05/13/91							    */
/****************************************************************************/
/* Modification History:						    */
/* 13may92 rah - created from extract.c					    */
/* $Log$
*/
/****************************************************************************/

#include <stdio.h>			/* Standard I/O			    */
#include <stdlib.h>			/* For exit()			    */
#include <mbari/types.h>		/* MBARI type definitions	    */
#include <mbari/const.h>		/* MBARI constants		    */
#include <decode.h>			/* OASIS controller definitions	    */
#include <sys/types.h>			/* Unix types			    */
#include <dirent.h>			/* For searching directory	    */
#include <time.h>			/* Time				    */
#include <string.h>			/* strncpy(), etc		    */
#include <memory.h>			/* memset(), etc		    */
#include <stdarg.h>			/* variable argument lists	    */

#define NAMESIZE	256		/* Space allocated for file names   */
#define BASE_DIR	"/oasis"
#define HDR_ENTRIES	200		/* Max number of header entries	    */
#define BUFSIZE		512		/* Size of general purpose buffer   */

typedef struct				/************************************/
{					/* Struct contains CTD times from hdr*/
    time_t	ct_sttime;		/* Start time in time_t format	    */
    Int		ct_start;		/* First sample in this set	    */
    Int		ct_end;			/* Last sample in this set	    */
    Int		ct_interval;		/* Interval between samples	    */
} CTDTimes;				/************************************/


/********************************/
/*	External Functions	*/
/********************************/

Extern Status	read_ctd_cal( char *name, CTDCal *ccp );
Extern Status	decode_ctd( Byte *dp, Int len, CTDDecode *cdp, CTDCal *ccp );


/********************************/
/*	External Data		*/
/********************************/

Extern char	*optarg;		/* Option argument from getopt()    */
Extern Int	optind;			/* Option index from getopt()	    */
Extern Int	opterr;			/* getopt() error flag		    */
Extern long	timezone;


/********************************/
/*	Global Data		*/
/********************************/

Global char	buff[BUFSIZE];		/* Data buffer			    */
Global Byte	dbuff[BUFSIZE/2];	/* Decoded data buffer		    */
Global CTDTimes	hdr[HDR_ENTRIES];	/* Buffer for input header entries  */
Global Int	hdr_ents;		/* Number of header entries	    */
Global time_t	ttime;			/* Time of data collect in time_t fmt*/
Global double	dtime;			/* Time of data collect in real format*/
Global Int	idate;			/* Integer year/day as yyddd	    */
Global CTDCal	ctdcal;			/* Buffer for CTD calibration	    */
Global CTDDecode ctd;			/* Buffer for decoded CTD data	    */
Global Int	ctd_sernum;		/* CTD serial number from filename.nnn*/
Global Int	toffset = 0;		/* Offset to add in seconds	    */
Global Int	sample_num = 0;		/* First sample number		    */

/* File names		*/
Global char	cal_file[NAMESIZE] = "/oasis/cfg/ctd.099";
Global char	hdr_file[NAMESIZE] = "/oasis/cfg/m1/ctd.hdr";
Global char	indata_file[NAMESIZE];
Global char	inhdr_file[NAMESIZE];
Global char	outdata_file[NAMESIZE];

/* File descriptors	*/
Global FILE	*outfp;


/********************************/
/*	Forward Declarations	*/
/********************************/

Int	getargs( Int argc, char **argv );
Void	usage( char *name );
FILE	*open_input_file( char *name );
time_t	sample_time( Int num );
Void	print_ctd( char *fmt, ... );


/************************************************************************/
/* Function    : main							*/
/* Purpose     : Main routine						*/
/* Inputs      : argc, argv						*/
/* Output      : none							*/
/************************************************************************/
	Void
main( Int argc, char **argv )
{
    Int		argnum;				/* argv index		  */
    char	*name;				/* Input file base name   */
    FILE	*infp;				/* Input file descriptor  */
    Int		sample;				/* Sample number	  */
    time_t	samp_tm;			/* Sample time		  */
    struct tm	*tp;				/* Time struct ptr	  */
    int		i, j;

    argnum = getargs( argc, argv );		/* Get command-line args  */

    if ( (argnum == ERROR) || (argnum >= argc) )
    {						/* If error in arguments  */
	usage( argv[0] );			/*   print usage message  */
	exit( 1 );				/*   and exit		  */
    }

    while ( argnum < argc )
    {
	name = argv[argnum++];

	if ( (hdr_ents = read_hdr_file(name)) <= 0 )
	{
	    printf( "Cannot read %s\n", inhdr_file );
	    continue;
	}

	if ( (infp = open_input_file(name)) == (FILE *)NULL )
	{
	    printf( "No CTD data file of type %s.nnn\n", name );
	    continue;
	}

	sprintf( cal_file, "%s/cfg/ctd.%03d", BASE_DIR, ctd_sernum );
	if ( read_ctd_cal(cal_file, &ctdcal) == ERROR )
	{
	    printf( "Can't open CTD calibration file %s\n", cal_file );
	    continue;
	}
	else
	    ctdcal.cc_size -= 2;		/* Downloaded data has no */
						/*  sample number	  */
	for ( sample = sample_num;  (fgets(buff, BUFSIZE, infp) != NULL); )
	{
	    for ( i = 0; i < strlen(buff)/2; i++ )
	    {
		if ( sscanf(&buff[i+i], "%2x", &j) == 1 )
		    dbuff[i] = (Byte)j;
	        else
		    break;
	    }
	    if ( decode_ctd(dbuff, i, &ctd, &ctdcal) == OK )
		if ( (samp_tm = sample_time(sample)) != (time_t)(-1) )
		{		    
		    tp = gmtime( &samp_tm );
		    dtime = tp->tm_yday + 1.0 + (double)((3600 * tp->tm_hour) 
				+ (60 * tp->tm_min) + tp->tm_sec) / 86400.0;
		    idate = (1000 * tp->tm_year) + tp->tm_yday + 1;
		    print_ctd( "%9.5f %d %6.3f %6.3f %6.3f %6.4f %6.4f %6.4f\n",
			       dtime, sample, ctd.ctd_press,
			       ctd.ctd_temp, ctd.ctd_cond, ctd.ctd_sal,
			       ctd.ctd_trans, ctd.ctd_fluor );
		    sample++;
		}
	}
	fclose( infp );
    }

    exit( 0 );

} /* main() */


/************************************************************************/
/* Function    : usage							*/
/* Purpose     : Print usage message					*/
/* Inputs      : Name of program					*/
/* Output      : None							*/
/************************************************************************/
	Void
usage( char *name )
{
    printf("Usage: %s [-c cal_file] [-h hdr_file] [-o offset] [-s sample num] filename(s)\n",
	   name);
    printf("where\tcal_file is the CTD calibration file\n");
    printf("\thdr_file is the output file header,\n");
    printf("\toffset is time in seconds to add to sample time,\n");
    printf("\tsample num is the number of the first sample, and\n");
    printf("\tfilename(s) are the base names of the file to convert\n");
    printf("\n%s will look for filename.hdr and filename.nn\n", name);
    printf("and will create output files in the current directory\n");
    printf("The default value of cal_file is /oasis/cfg/ctd.nnn,\n");
    printf("where nnn is taken from filename.nnn above\n");
    printf("The default hdr_file is /oasis/cfg/m1/ctd.hdr\n");
    printf("\nFor example, assume your current directory has files");
    printf(" 92057.099 and 92057.hdr\n");
    printf("The command:\n");
    printf("\t\t%s 92057\n", name);
    printf("will use calibration file /oasis/cfg/ctd.099\n");
    printf("to decode the data and send the output to standard out\n\n");

} /* usage() */

/************************************************************************/
/* Function    : getargs						*/
/* Purpose     : Process command-line arguments				*/
/* Inputs      : argc, argv						*/
/* Output      : argv index of first non-option argument, or ERROR	*/
/************************************************************************/
	Int
getargs( Int argc, char **argv )
{
    Int		i;
    char	*file;
    FILE	*fd;
    struct tm	*tp;

    opterr = 0;					/* Disable error msgs	*/
    while ( (i = getopt(argc, argv, "c:h:o:s:")) != EOF )
	switch( i )
	{
	  case 'c':
	    strncpy(cal_file, argv[optind], NAMESIZE);
	    break;

	  case 'h':
	    strncpy(hdr_file, argv[optind], NAMESIZE);
	    break;

	  case 'o':
	    toffset = atoi(optarg);
	    break;

	  case 's':
	    sample_num = atoi(optarg);
	    break;

	  default:
	    return( ERROR );
	}

    return( optind );

} /* getargs() */


/************************************************************************/
/* Function    : read_hdr_file						*/
/* Purpose     : Read header file from CTD				*/
/* Inputs      : Base name of header file				*/
/* Output      : Number of entries read, or ERROR			*/
/* Side Effects: inhdr_file[], hdr[]					*/
/************************************************************************/
	Int
read_hdr_file( char *name )
{
    FILE	*hfp;
    Int		i, rtn;
    struct tm	t;
    time_t	tm;

    memset( hdr, 0, sizeof(hdr) );

    sprintf( inhdr_file, "%s.hdr", name );
    if ( (hfp = fopen(inhdr_file, "r")) == (FILE *)NULL )
	return( ERROR );

    t.tm_year = 92;				/* Assume 1992		*/
    sscanf( name, "%2d", &t.tm_year );		/* Get year from file name*/
    if ( t.tm_year < 90 )			/* If before 1990, 	*/
	t.tm_year += 100;			/*  make it 21st century*/

    for ( i = 0; (i < HDR_ENTRIES) && 
	 	 (fgets((char *)buff, BUFSIZE, hfp) != NULL); )
    {
	rtn = sscanf( (char *)buff, 
		     "%d/%d %d:%d:%d samples %d to %d interval = %d",
		    &t.tm_mon, &t.tm_mday, &t.tm_hour, &t.tm_min, &t.tm_sec,
		    &hdr[i].ct_start, &hdr[i].ct_end, &hdr[i].ct_interval );
	if ( rtn >= 8 )
	{
	    t.tm_mon--;				/* Unix month is 0-11	*/
	    t.tm_wday = t.tm_yday = 0;		/* mktime ignores these	*/
	    t.tm_isdst = 0;			/* Force no dst		*/

	    if ( (tm = mktime(&t)) != (time_t)(-1) )	/* Get sample time*/
		hdr[i++].ct_sttime = tm - timezone + toffset;
	}						/* Convert to GMT */
    }

    fclose( hfp );
    return( i );

} /* read_hdr_file() */


/************************************************************************/
/* Function    : open_input_file					*/
/* Purpose     : Find and open input file named "name.nnn"		*/
/* Inputs      : Base name of input file				*/
/* Output      : Integer value of nnn					*/
/* Side Effects: Opens input file, fills in indata_file[] & ctd_sernum	*/
/************************************************************************/
	FILE *
open_input_file( char *name )
{
    DIR			*dp;
    struct dirent	*dep;
    FILE		*fp;

    if ( (dp = opendir(".")) == (DIR *)NULL )
    {
	printf( "Cannot open current directory\n" );
	return( FALSE );
    }

    while ( (dep = readdir(dp)) != (struct dirent *)NULL )
    {
	sprintf((char *)buff, "%s.%%d", name);
	if ( (sscanf(dep->d_name, (char *)buff, &ctd_sernum) > 0) &&
	     ((fp = fopen(dep->d_name, "r")) != (FILE *)NULL) )
	{
	    strcpy( indata_file, dep->d_name );
	    closedir( dp );
	    return( fp );
	}
    }

    closedir( dp );
    return( (FILE *)NULL );

} /* open_input_file() */


/************************************************************************/
/* Function    : sample_time						*/
/* Purpose     : Return time of sample as a time_t			*/
/* Inputs      : Sample number						*/
/* Output      : time_t sample time					*/
/************************************************************************/
	time_t
sample_time( Int num )
{
    Int		i, min;

    for ( i = min = 0; i < hdr_ents; i++ )
    {
	if ( num >= hdr[i].ct_start )
	{
	    if ( num <= hdr[i].ct_end )
	    {
		if ( hdr[i].ct_sttime == (time_t)(-1) )
		    return( (time_t)(-1) );
		else
		    return( hdr[i].ct_sttime + 
			   ((num - hdr[i].ct_start + 1) * hdr[i].ct_interval) );
	    }
	    else if ( hdr[i].ct_end > hdr[min].ct_end )
		min = i;
	}
    }

    return( hdr[min].ct_sttime + 
	   ((num - hdr[min].ct_start + 1) * hdr[min].ct_interval) );

} /* sample_time() */


/************************************************************************/
/* Function    : print_ctd						*/
/* Purpose     : Print to CTD output file				*/
/* Inputs      : Format, arguments					*/
/* Output      : None							*/
/************************************************************************/
	Void
print_ctd( char *fmt, ... )
{
    va_list	ap;			/* Argument pointer		*/

    va_start( ap, fmt );
    vfprintf( stdout, fmt, ap );

} /* print_ctd() */


/************************************************************************/
/* Function    : getMotword						*/
/* Purpose     : Get a word in Motorola format from data stream		*/
/* Inputs      : Ptr to data stream					*/
/* Outputs     : Word							*/
/************************************************************************/
	Nat16
getMotword( Byte *p )
{
    Nat32	rtn;

    rtn = (Nat16)(*p << 8);
    rtn += (Nat16)(p[1]);
    return( rtn );

} /* getMotword() */
