/****************************************************************************/
/* Copyright 1992 MBARI                                                     */
/****************************************************************************/
/* $Header: getoasis.c,v 1.5 94/01/17 11:07:42 hebo Exp $		    */
/* Summary  : Program to download OASIS data				    */
/* Filename : getoasis.c						    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 1.5 $							    */
/* Created  : 02/22/92							    */
/****************************************************************************/
/* Modification History:						    */
/* 22feb92 rah - created, from gps.c					    */
/* $Log:	getoasis.c,v $
 * Revision 1.5  94/01/17  11:07:42  11:07:42  hebo (Bob Herlien)
 * Misc changes
 * 
 * Revision 1.4  92/06/22  14:48:11  14:48:11  hebo (Bob Herlien)
 * Fixed timeout bug in wait_for_prompt() that caused early abort.
 * 
 * Revision 1.3  92/06/22  10:20:22  10:20:22  hebo (Bob Herlien)
 * Fixed bug in tnc_cmd_mode.  Added error message if release_port() fails.
 * 
 * Revision 1.2  92/05/29  09:13:14  09:13:14  hebo (Bob Herlien)
 * Changed restore parameters to delete "M OFF", add "EC ON"
 * Deleted #define _HPUX_SOURCE.  It's now defined in Makefile.
 * 
 * Revision 1.1  92/03/16  15:44:43  15:44:43  hebo (Bob Herlien)
 * First working version
 * 
 * Revision 1.0  92/03/12  13:59:22  13:59:22  hebo (Bob Herlien)
 * Initial revision
*/
/****************************************************************************/

#include <stdio.h>			/* Standard I/O			    */
#include <stdlib.h>			/* For exit()			    */
#include <sys/types.h>			/* For FD_ISSET, etc		    */
#include <mbari/types.h>		/* MBARI type definitions	    */
#include <mbari/const.h>		/* MBARI constants		    */
#include <time.h>			/* For timeout			    */
#include <termio.h>			/* For terminal I/O stuff	    */
#include <fcntl.h>			/* For open() stuff		    */
#include <signal.h>			/* For signal()			    */
#include <sys/ioctl.h>			/* For ioctl()			    */
#include <sys/param.h>			/* For NOFILE			    */
#include <ctype.h>			/* For toupper()		    */
#include <string.h>			/* For strchr()			    */
#include "decode.h"			/* For LogStruct		    */

#define TNC_DEV	  "tnc"			/* Default radio TNC device file    */
#define BUFSIZE	  1023			/* Size of line buffer		    */
#define TNC_ADDR  "mbari-3"		/* Dflt TNC to connect to	    */
#define OPROMPT	  "OASIS>"		/* OASIS prompt			    */
#define CAN_NAME  "m1a"			/* Can name			    */

#define ARG_ERR		1		/* Bad command line arg exit() code */
#define PORT_ERR	2		/* Serial port error exit() code    */
#define CONNECT_ERR	3		/* Can't connect exit() code	    */
#define DOWNLOAD_ERR	4		/* Download error exit() code	    */
#define ABORT_ERR	5		/* Aborted exit() code		    */


/********************************/
/*	External Functions	*/
/********************************/
			/* From port.c			    */
Extern MBool	set_baud( struct termio *tp, Int baud );
Extern Int	get_port( char *name, Int mode, 
			  struct termio *setup, struct termio *save );
Extern Status	release_port( char *name, Int fd, struct termio *parms );

Extern Void	rcdInit( LogStruct *lp );
Extern MBool	readRcd( char *can, LogStruct *lp );
Extern MBool	getMissingRcd( LogStruct *lp, Nat32 lognum,
			       Nat32 last, RcdRange *rcd );

/********************************/
/*	External Data		*/
/********************************/

Extern char	*optarg;		/* Option argument from getopt()    */
Extern Int	optind;			/* Option index from getopt()	    */
Extern Int	opterr;			/* getopt() error flag		    */
Extern char	*rcdfile;		/* File of record numbers	    */


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal char	*tncfile = TNC_DEV;	/* Radio TNC device file	    */
MLocal char	*tncaddr = TNC_ADDR;	/* TNC address to connect to	    */
MLocal time_t	timeout = 660;		/* 11 minute timeout by default	    */
MLocal MBool	get_rcds = FALSE;	/* TRUE to get missing records	    */
MLocal Int32	tfd = ERROR;		/* TNC file descriptor		    */
MLocal MBool	echo = FALSE;		/* TRUE to echo tnc reads to stdout */
MLocal MBool	aborted = FALSE;	/* Set from signal handler	    */
MLocal char	rbuf[BUFSIZE+1];	/* Read line buffer		    */
MLocal Nat32	rcnt = 0;		/* # chars in rbuf		    */
MLocal LogStruct logs;			/* Which records have been read	    */
MLocal char	*can_name = CAN_NAME;	/* Can name			    */

MLocal struct termio tnc_term =		/* Termio characteristics for TNC   */
{   IGNPAR | IXON,			/* c_iflag			    */
    0,					/* c_oflag			    */
    B9600 | CS8 | CREAD | CLOCAL,	/* c_cflag			    */
    NOFLSH,				/* c_lflag			    */
    0,					/* c_line, HP has no line disciplines*/
    {0, 0, 0, 0, 0, 10, 0, 0}		/* control chars (1 second timeout) */
};

MLocal struct termio tnc_save;	 	/* Place to save terminal charac.   */

MLocal char *setup_tnc[] = { "NO ON\r", "EC OFF\r", "RET 15\r", "M OFF\r" };

MLocal char *restore_tnc[] = { "NO OFF\r", "EC ON\r" };


/********************************/
/*	Forward Declarations	*/
/********************************/

MBool		process_command_line ( Int argc, char **argv );
Void		use_msg( char *s );
Status		connect_tnc( Void );
Status		disconnect_tnc( Void );
Status		send_tnc_cmds( char *cmds[], Int num );
Status		get_missing_records( Void );
Status		get_data( Void );
Void		quit( Int sig );


/************************************************************************/
/* Function    : main							*/
/* Purpose     : Main routine						*/
/* Inputs      : argc, argv						*/
/* Outputs     : none							*/
/************************************************************************/
	Void
main( Int argc, char **argv )
{
    Int		rtn;			/* exit() code			*/
    
    printf("OASIS download program $Revision: 1.5 $\n");
    if ( !process_command_line(argc, argv) )
    {
	use_msg(argv[0]);
	exit( ARG_ERR );
    }

    signal(SIGTERM, quit);		/* Catch terminate signals	*/
    signal(SIGHUP, quit);
    signal(SIGINT, quit);
    signal(SIGQUIT, quit);

    if ( (tfd = get_port(tncfile, O_RDWR, &tnc_term, &tnc_save)) == ERROR )
    {
	printf( "Couln't acquire serial port %s\n", tncfile ); 
	exit( PORT_ERR );
    }

    if ( send_tnc_cmds(setup_tnc, sizeof(setup_tnc)/sizeof(char *)) == ERROR )
	printf( "Failed to setup TNC commands\n" );

    if ( connect_tnc() == OK )
    {
	rtn = wait_for_prompt();		/* Look for initial prompt*/
    
	if ( get_rcds && (rtn == OK) )
	    rtn = get_missing_records();

	if ( rtn == OK )
	    rtn = get_data();

	write( tfd, "quit\r", 5 );
	sleep( 5 );
    }
    else
    {
	printf( "Couldn't establish radio communication\n" );
	rtn = CONNECT_ERR;
    }

    disconnect_tnc();

    send_tnc_cmds( restore_tnc, sizeof(restore_tnc)/sizeof(char *) );
    sleep( 2 );
    printf("\n");

    if ( release_port(tncfile, tfd, &tnc_save) == ERROR )
	printf( "Error in releasing serial port %s\n", tncfile );

    exit( aborted ? ABORT_ERR : rtn );

} /* main() */


/************************************************************************/
/* Function : use_msg							*/
/* Purpose  : Print Usage Message					*/
/* Inputs   : Name of program						*/
/* Outputs  : None							*/
/************************************************************************/
	Void
use_msg( char *s )
{
    fprintf( stderr,
	    "Usage: %s [-a tnc_address] [-b baud] [-c can] [-p port] ", s );
    fprintf( stderr, "[-r rcd_file] [-t timeout]\n");

} /* use_msg() */


/************************************************************************/
/* Function : process_command_line					*/
/* Purpose  : Read the arguments from the command line			*/
/* Inputs   : argc, argv from main() routine				*/
/* Outputs  : TRUE if arguments OK, else FALSE				*/
/************************************************************************/
	MBool
process_command_line ( Int argc, char **argv )
{
    Int		i, baud;

    while ( (i = getopt(argc, argv, "a:b:c:p:r:t:")) != EOF )
	switch( i )
	{
	  case 'a':
	    tncaddr = optarg;
	    break;

	  case 'b':
	    if ( !set_baud(&tnc_term, atoi(optarg)) )
		return( FALSE );
	    break;

	  case 'c':
	    can_name = optarg;
	    get_rcds = TRUE;
	    break;

	  case 'p':
	    tncfile = optarg;
	    break;

	  case 'r':
	    rcdfile = optarg;
	    get_rcds = TRUE;
	    break;

	  case 't':
	    timeout = 60 * atoi(optarg);
	    break;

	  default:
	    return( FALSE );
	}

    return( TRUE );

} /* process_command_line() */


/****************************************************************************/
/* Function    : find_str						    */
/* Purpose     : Find target string embedded in source string, case insens  */
/* Inputs      : Source string ptr, Target string ptr			    */
/* Outputs     : NULL if strings don't match, else ptr to next char of source*/
/****************************************************************************/
	char *
find_str( char *src, char *tgt )
{
    Reg char	*p, *p1, *q;

    for ( p1 = src; *p1; p1++ )
    {
	for( p = p1, q = tgt; ; p++, q++ )
	{
	    if ( *q == '\0' )
		return( p );
	    if ( *p == '\0' )
		return( NULL );
	    if ( toupper(*p) != toupper(*q) )
		break;
	}
    }

    return( NULL );

} /* find_str() */


/************************************************************************/
/* Function    : gets_tmout						*/
/* Purpose     : Read one line from a file descriptor with timeout	*/
/* Inputs      : File descriptor, buffer, length, timeout (seconds)	*/
/* Outputs     : Number of characters read, or ERROR			*/
/* Comment     : Needs a buffer of length len+1 (for '\0')		*/
/************************************************************************/
	Int
gets_tmout( Int fd, char *buf, Int len, Int tmout )
{
    time_t	start, curtime;
    Int		i, rtn;
    char	c;

    time( &start );				/* Get start time	*/
    for ( rtn = 0; rtn < len; )
    {
	if ( (i = read(fd, &buf[rtn], 1)) < 0 ) /* Read one character	*/
	    break;
	if ( i > 0 )				/* If got character,	*/
	{
	    c = buf[rtn++];			/* increment char count	*/
	    if ( (c == '\r') || (c == '\n') )	/* look for end of line */
		break;
	}
	time( &curtime );			/* Get time		*/
	if ( curtime > start + tmout )		/* If timed out, return */
	    break;
    }

    buf[rtn] = '\0';				/* Insert EOS		*/
    return( rtn );				/* Return # chars	*/

} /* gets_tmout() */


/************************************************************************/
/* Function    : tnc_cmd_mode						*/
/* Purpose     : Send ^C to TNC to go into command mode			*/
/* Inputs      : None							*/
/* Outputs     : OK or ERROR						*/
/************************************************************************/
	Status
tnc_cmd_mode( Void )
{
    Int32	i;

    if ( write(tfd, "\003\r", 2) != 2 )		/* Send ctrl-C to TNC	*/
	return( ERROR );

    for ( i = 0; i < 10; i++ )			/* Wait for "cmd:" prompt*/
    {
	if ( gets_tmout(tfd, rbuf, 4, 10) <= 0 )
	    return( ERROR );
	if ( find_str(rbuf, "cmd:") != NULL )
	    return( OK );
    }

} /* tnc_cmd_mode() */


/************************************************************************/
/* Function    : send_tnc_cmds						*/
/* Purpose     : Send commands to the TNC				*/
/* Inputs      : Ptr to cmds, number of cmds				*/
/* Outputs     : OK or ERROR						*/
/************************************************************************/
	Status
send_tnc_cmds( char *cmds[], Int num )
{
    Int		i;
    Status	rtn;

    rtn = tnc_cmd_mode();			/* Go into command mode	*/

    for ( i = 0; i < num; i++ )			/* Send commands to TNC	*/
    {
	write(tfd, cmds[i], strlen(cmds[i]));
	gets_tmout( tfd, rbuf, BUFSIZE, 1 );	/* Discard any replys	*/
    }

    return( OK );

} /* send_tnc_cmds() */


/************************************************************************/
/* Function    : tnc_status						*/
/* Purpose     : Get STATus message from TNC				*/
/* Inputs      : None							*/
/* Outputs     : Status of TNC, or ERROR				*/
/* Comment     : Leaves TNC in command mode				*/
/************************************************************************/
	Int
tnc_status( Void )
{
    char	*p;
    Int		res;

    write( tfd, "STAT\r", 5);			/* Ask for status	*/
    res = -1;					/* Initialize result	*/

    while ( gets_tmout(tfd, rbuf, BUFSIZE, 10) != ERROR )
	if ( (p = find_str(rbuf, "STAT")) != NULL )
	    if ( (p = strchr(p, '$')) != NULL )
		if ( sscanf(p+1, "%x", &res) == 1 )
		    break;

    gets_tmout( tfd, rbuf, BUFSIZE, 10 );	/* Clear tnc buffer	*/

    return( res );

} /* tnc_status() */


/************************************************************************/
/* Function    : disconnect_tnc						*/
/* Purpose     : Break TNC connection					*/
/* Inputs      : None							*/
/* Outputs     : OK or ERROR						*/
/************************************************************************/
	Status
disconnect_tnc( Void )
{
    Int		i;
    Status	rtn;

    rtn = tnc_cmd_mode();			/* Go into command mode	*/

    if ( write(tfd, "D\r", 2) != 2 )		/* Make sure we're 	*/
	return( ERROR );			/*  disconnected	*/

    for ( i = 10; i; i-- )
    {
	if ( tnc_status() == 0 )
	    return( rtn );
	sleep( 2 );
    }

    return( ERROR );

} /* disconnect_tnc() */


/************************************************************************/
/* Function    : connect_tnc						*/
/* Purpose     : Put TNC into converse mode				*/
/* Inputs      : None							*/
/* Outputs     : OK or ERROR						*/
/************************************************************************/
	Status
connect_tnc( Void )
{
    time_t	start, curtime;

    disconnect_tnc();				/* Make sure disconnected*/
    time( &start );				/* Get time		*/

    while ( time(&curtime) < start + timeout )	/* Loop until connect	*/
    {						/*  or timeout		*/
	if ( aborted )				/* If program aborted,	*/
	    return( ERROR );			/*  return ERROR	*/

	switch( tnc_status() )			/* Check if we're connected*/
	{
	  case 0:				/* If no conn in progress*/
	    sprintf(rbuf, "C %s\r", tncaddr);	/* Send connect message	*/
	    if ( write(tfd, rbuf, strlen(rbuf)) != strlen(rbuf) )
		return( ERROR );
	    break;

	  case 1:				/* Connection in progress*/
	  case 3:				/* Just wait		 */
	    break;

	  case 4:				/* Connected		*/
	  case 6:
	    if (write(tfd, "K\r", 2) == 2)	/* Go to converse mode	*/
		return( OK );			/*  and return OK	*/
	    else
		return( ERROR );
	}
	sleep( 1 );				/* Wait 2 seconds each loop*/
    }

    return( ERROR );				/* Timed out.  Error	*/

} /* connect_tnc() */


/************************************************************************/
/* Function    : wait_for_prompt					*/
/* Purpose     : Wait for OASIS prompt or "DISCONNECTED" message	*/
/* Inputs      : None							*/
/* Outputs     : OK or ERROR						*/
/* Comments    : Error if "DISCONNECTED" or no data for timeout seconds	*/
/************************************************************************/
	Status
wait_for_prompt( Void )
{
    time_t	datatime, curtime;
    Int		i, j;
    char	c;

    time( &datatime );				/* Get start time	*/
    j = 0;

    while ( !aborted && (time(&curtime) <= datatime + timeout) )
    {
	if ( (i = read(tfd, &c, 1)) > 0 ) 	/* Read one character	*/
	{					/* If got character,	*/
	    rbuf[j++] = c;			/* Store it		*/
	    rbuf[j] = '\0';			/* Mark end of string	*/
	    if ( c != '\r' )
		putchar( c );			/* Echo it to stdout	*/
	    datatime = curtime;			/* Save time of last data*/
	    if ( find_str(rbuf, OPROMPT) != NULL )
	    {					/* Look for prompt	*/
		while ( read(tfd, &c, 1) > 0 )	/* If got it, throw away*/
		    ;				/*   extra chars (spaces)*/
		return( OK );			/*   and return OK	*/
	    }
	    if ( find_str(rbuf, "DISCONNECTED") != NULL )
		return( DOWNLOAD_ERR );		/* Look for DISCON msg	*/
	    if ( (c == '\r') || (c == '\n') || (j > BUFSIZE) )
		j = 0;
	} /* if */
    } /* while */

    return( DOWNLOAD_ERR );

} /* wait_for_prompt() */


/************************************************************************/
/* Function    : get_data						*/
/* Purpose     : Get data from mooring, copy to stdout			*/
/* Inputs      : None							*/
/* Outputs     : Return code for exit()					*/
/************************************************************************/
	Status
get_data( Void )
{
    if ( write(tfd, "logs\r", 5) != 5 )		/* Send "logs" command	*/
	return( DOWNLOAD_ERR );

    if ( wait_for_prompt() == ERROR )		/* Wait for end of "logs"*/
	return( DOWNLOAD_ERR );
    
    printf("\n");				/* Echo newline to output*/
    if ( write(tfd, "getdata\r", 8) != 8 )	/* Send "getdata" command*/
	return( DOWNLOAD_ERR );

    if ( wait_for_prompt() == ERROR )		/* Wait for end of data	 */
	return( DOWNLOAD_ERR );

    return( OK );

} /* get_data() */


/************************************************************************/
/* Function    : get_missing_records					*/
/* Purpose     : Look for and download missing data records		*/
/* Inputs      : None							*/
/* Outputs     : Return code for exit()					*/
/************************************************************************/
	Status
get_missing_records( Void )
{
    Reg Nat32	lognum, rcd;
    RcdRange	range;
    MBool	changedToBin;

    rcdInit( &logs );				/* Init LogStructs	*/

    if ( !readRcd(can_name, &logs) )		/* Read record file	*/
	return( OK );

    changedToBin = FALSE;

    for ( lognum = logs.firstLog; lognum < logs.nLogs; lognum++ )
	for (rcd = 0;
	     getMissingRcd(&logs, lognum, rcd, &range); )
	{
	    if ( !changedToBin )
	    {
		printf("\n");
		if ( write(tfd, "parm ascii 0\r", 13) != 13 )
		    return( DOWNLOAD_ERR );
		if ( wait_for_prompt() == ERROR )
		    return( DOWNLOAD_ERR );
		changedToBin = TRUE;
	    }
    
	    printf("\n");			/* Echo newline to output*/
	    sprintf(rbuf, "getlog %d %d %d\r", lognum,
		    range.min, range.max - range.min +1);
	    if ( write(tfd, rbuf, strlen(rbuf)) != strlen(rbuf) )
		return( DOWNLOAD_ERR );
	    if ( wait_for_prompt() == ERROR )
		return( DOWNLOAD_ERR );
	    rcd = range.max + 1;
	}

    return( OK );

} /* get_missing_records() */


/************************************************************************/
/* Function    : tnc_read						*/
/* Purpose     : SIGIO handler						*/
/* Inputs      : Signal number received					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
tnc_read( Void )
{
    Int32	nbytes;
    char	c;

    if ( (nbytes = read(tfd, &c, 1)) > 0 )
    {
	rbuf[rcnt++] = c;
	if ( echo && (c != '\r') )
	    putchar( c );
	
	if ( rcnt > BUFSIZE )
	    rcnt = 0;

	rbuf[rcnt] = '\0';
    }
    else if ( nbytes == -1 )
    {
	aborted = TRUE;
	perror( "Read error in SIGIO handler\n" );
    }
}

/************************************************************************/
/* Function    : quit							*/
/* Purpose     : SIGTERM handler					*/
/* Inputs      : Signal number received					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
quit( Int sig )
{
    aborted = TRUE;

} /* quit() */
