/************************************************************************/
/* Copyright 1991 MBARI							*/
/************************************************************************/
/* $Header: decode.h,v 2.5 94/12/15 10:59:37 hebo Exp $			*/
/* Summary  : Definitions for decoding OASIS data			*/
/* Filename : decode.h							*/
/* Author   : Robert Herlien (rah)					*/
/* Project  : OASIS Mooring						*/
/* $Revision: 2.5 $							*/
/* Created  : 02/06/91							*/
/************************************************************************/
/* Modification History:						*/
/* 06feb91 rah - created						*/
/* $Log:	decode.h,v $
 * Revision 2.5  94/12/15  10:59:37  10:59:37  hebo (Bob Herlien)
 * Accumulated minor changes, mainly due to move to tsunami
 * 
 * Revision 2.4  94/01/21  14:36:10  14:36:10  hebo (Bob Herlien)
 * Added support for date ranges in cfg file
 * 
 * Revision 2.3  94/01/17  11:07:36  11:07:36  hebo (Bob Herlien)
 * Misc changes
 * 
 * Revision 2.2  92/09/24  18:36:45  18:36:45  hebo (Bob Herlien)
 * Changed trfl to trans, fluor on CTD; due to transposing cables on some CTDs
 * 
 * Revision 2.0  92/08/31  15:35:45  15:35:45  hebo (Bob Herlien)
 * Auguest 1992 Deployment.  Changed to allow multiple sensors of same type.
 * 
 * Revision 1.1  92/06/15  09:08:03  09:08:03  hebo (Bob Herlien)
 * *** empty log message ***
 * 
 * Revision 1.0  92/03/04  17:19:03  17:19:03  hebo (Bob Herlien)
 * Initial revision
 * 
*/
/************************************************************************/

#define MAXLOGSIZE	1040		/* Max size of log record	    */
#define NUM_CTDS	3		/* Number of CTDs we can decode	    */
#define NUM_NO3S	2		/* Number of NO3s we can decode	    */
#define NUM_SPECS	4		/* Number of Spectros we can decode */
#define CTD_CAL		0		/* Surface CTD uses cal file 0 	    */
#define CTD2_CAL	1		/* Second CTD uses cal file 1 	    */
#define CTD3_CAL	2		/* Third CTD uses cal file 2 	    */
#define NO3_CAL		0		/* First NO3 uses cal file 0	    */
#define NO3_CAL2	1		/* Second NO3 uses cal file 1	    */
#define SPECTRO_CAL	0		/* Surface spectro uses cal file 0  */
#define SPECTRO2_CAL	1		/* 10m spectro uses cal file 1	    */
#define SPECTRO3_CAL	2		/* 20m spectro uses cal file 2	    */
#define SPECPRR_CAL	3		/* PRR-600 spectro uses cal file 3  */

					/************************************/
/*	Data Logging Record Types	*/
#define LOG_EMPTY	0		/* Unused - empty record	    */
#define ERRORF		0		/* Also used for error file ptr	    */
#define TIME		1		/* Time record      -  2 words	    */
#define ATLAS		2		/* atlas record     - 32 words	    */
#define OASIS_CAN	3		/* OASIS battery, temp, press: 4 wrds*/
#define PAR		4		/* PAR sensors       - 2 words	    */
#define CTD		5		/* CTD Data          - variable	    */
#define SPECTRO		6		/* Spectroradiometer - variable	    */
#define ADCP		7		/* Acoustic Doppler  - variable	    */
#define GPS		8		/* GPS				    */
#define MODEM		9		/* Acoustic Modem		    */
#define PCO2		10		/* pCO2 sensor			    */
#define CTD2		11		/* Second CTD			    */
#define CTD3		12		/* Third CTD			    */
#define SPECTRO2	13		/* Second Spectroradiometer	    */
#define SPECTRO3	14		/* Third Spectroradiometer	    */
#define NO3		15		/* Nitrate sensor		    */
#define NO32		16		/* Second Nitrate sensor	    */
#define SPEC_PRR	17		/* PRR-600 Spectroradiometer	    */
#define SATLANTIC	18		/* Satlantic Spectroradiometer	    */
#define GPS_TYPE2	19		/* GPS				    */
#define NRL		20		/* NRL satellite controller	    */
#define O2		21		/* Oxygen sensor		    */
#define FLUOR		22		/* Fluorometer			    */
#define TRANSMISS	23		/* Transmissometer		    */
#define NO3_DECODED	24		/* Nitrate sensor, decoded data	    */
#define NO3_DECODED2	25		/* Second Nitrate sensor, decoded   */
#define SENSORS		(NO3_DECODED2 + 1) /* Number of sensors		    */
					/************************************/
#define LOG_ERROR	128		/* Error record			    */
#define LOG_COMMENT	129		/* User comments		    */
#define LOG_BIN		130		/* Miscellaneous Binary data	    */
					/************************************/
	/* OASIS Error types as logged in OASIS data log		    */
#define RAM_ERR		0x8001		/* RAM not initialized		    */
#define LOG_ERR		0x8002		/* Bogus log memory		    */
#define CLOCK_ERR	0x8004		/* Real-time clock failure	    */
#define INT_ERR		0x8008		/* Received spurious interrupt	    */
#define RESTART_ERR	0x8010		/* Restarted with bad keepalive     */
#define COMM_ERR	0x8020		/* Restarted due to no comm in 24hrs*/
					/************************************/
	/* Return types from decode subroutines  ****************************/
#define UUERROR		(-2)		/* Error return from uuread()	    */
#define UUEND		(-3)		/* uugetline() found "end" statement*/
#define	SIZE_ERR	(-2)		/* Data length error		    */
#define CHKSUM_ERR	(-3)		/* Checksum error		    */
#define FMT_ERR		(-4)		/* Data format error		    */
	/* Miscellaneous		 ************************************/
typedef int		Status;		/* Status return type		    */
#define NULLF		(FILE *)0	/* Null file pointer		    */
#define ANALOG_CHANS	8		/* Number of analog channels	    */
#define PAR_CHAN	0		/* Analog channel number for PARs   */
#define OASIS_CHAN	2		/* Analog chan num for OASIS stuff  */
#define FLUOR_CHAN	0		/* Analog channel for fluorometer   */
#define TRANSMISS_CHAN	1		/* Analog channel for transmissometer*/
					/************************************/
					/* ATLAS types and defines	    */
#define TEMPS		10		/* Number temperature pods	    */
#define PRESS		2		/* Number pressure pods	 	    */
#define ATLAS_WORDS	18		/* Number words in orig ATLAS record*/
#define ATLAS_LONG	20		/* Number words in new ATLAS record */
					/************************************/
typedef struct				/************************************/
{					/* LogRecHdr - Logging Record Header*/
    Nat16	log_type;		/* Record type - see below	    */
    Nat16	log_nmbr;		/* Record number		    */
    Int16	log_len;		/* Number of bytes in log_data	    */
    Nat32	log_time;		/* Record time			    */
} LogRecHdr;				/************************************/

typedef struct				/************************************/
{					/* Calibr struct for single ATLAS chan*/
    Int32	pod_type;		/* Pod type			    */
    Int32	ident;			/* Pod ID number		    */
    double	a;			/* A calibration coefficient	    */
    double	b;			/* B calibration coefficient	    */
    double	c;			/* C calibration coefficient	    */
    double	d;			/* D calibration coefficient	    */
    double	e;			/* E calibration coefficient	    */
} ACal;					/************************************/

typedef struct				/************************************/
{					/* Struct for ATLAS calibration	    */
    ACal	ac_air;			/* Air temp calibration		    */
    ACal	ac_sst;			/* Sea surface temp calibration	    */
    ACal	ac_temp[TEMPS];		/* Temperature pod calibrations	    */
    ACal	ac_press[PRESS];	/* Pressure pod calibrations	    */
    ACal	ac_rh;			/* Relative humidity calibration    */
    ACal	ac_wind;		/* Wind calibration		    */
} AtlasCal;				/************************************/

typedef struct				/************************************/
{					/* Struct to define ATLAS data	    */
    Nat16	at_time;		/* Date/time			    */
    Nat16	at_air;			/* Air temp			    */
    Nat16	at_sst;			/* Sea surface temp		    */
    Nat16	at_temp[TEMPS];		/* Temperature pods		    */
    Nat16	at_press[PRESS];	/* Pressure Pods		    */
    Nat16	at_rh;			/* Relative humidity		    */
    Nat16	at_wind;		/* East-west, north-south wind vects*/
    Nat16	at_chksum;		/* 16-bit checksum		    */
} Atlas;				/************************************/

typedef struct				/************************************/
{					/* Struct to define new ATLAS data  */
    Nat16	at_time;		/* Date/time			    */
    Nat16	at_air;			/* Air temp			    */
    Nat16	at_sst;			/* Sea surface temp		    */
    Nat16	at_temp[TEMPS];		/* Temperature pods		    */
    Nat16	at_press[PRESS];	/* Pressure Pods		    */
    Nat16	at_rh;			/* Relative humidity		    */
    Nat16	at_winduv;		/* East-west, north-south wind vects*/
    Nat16	at_covane;		/* Raw Compass, Vane values	    */
    Nat16	at_windspeed;		/* Raw wind speed		    */
    Nat16	at_chksum;		/* 16-bit checksum		    */
} AtlasLong;				/************************************/

typedef struct				/************************************/
{					/* Struct for decoded ATLAS data    */
    double	atd_time;		/* Date/time			    */
    double	atd_air;		/* Air temp			    */
    double	atd_sst;		/* Sea surface temp		    */
    double	atd_temp[TEMPS];	/* Temperature pods		    */
    double	atd_press[PRESS];	/* Pressure Pods		    */
    double	atd_rh;			/* Relative humidity		    */
    double	atd_windspd;		/* Wind speed			    */
    double	atd_winddir;		/* Wind direction		    */
    double	atd_windu;		/* Calibrated U (east-west) vector  */
    double	atd_windv;		/* Calibrated V (north-south) vector*/
    double	atd_compass;		/* Raw compass output		    */
    double	atd_vane;		/* Raw wind vane output		    */
    double	atd_rawdir;		/* Raw wind direction (co + va + mag)*/
    double	atd_rawspeed;		/* Raw wind speed output	    */
} AtlasDecode;				/************************************/

#define UNITNAMELEN	32		/* Length of buffer for units name  */

typedef struct				/************************************/
{					/* Struct for Analog calibration    */
    double	a;			/* f(x) = (a*x + b) * c + d	    */
    double	b;			/*                  b calibr. value */
    double	c;			/*                  c calibr. value */
    double	d;			/*                  d calibr. value */
    char	units[UNITNAMELEN];	/* Name of units		    */
} Analog;				/************************************/

#define CTD_BYTES	11		/* Number bytes in CTD record	    */
#define CTD_BYTES_SHORT	 9		/* Number bytes in short CTD record */

typedef struct				/************************************/
{					/* Single CTD Calibration struct    */
    int		offset;			/* Byte offset in data stream	    */
    double	a;			/* A or A0 calibration coefficient  */
    double	b;			/* B or A1 calibration coefficient  */
    double	c;			/* C or A2 calibration coefficient  */
    double	d;			/* D calibration coefficient	    */
    double	mf;			/* M or F0 calibration coefficient  */
} CCal;					/************************************/

typedef struct				/************************************/
{					/* Struct for CTD calibration	    */
    int		cc_size;		/* Size of CTD data		    */
    CCal	cc_cond;		/* Conductivity calibration	    */
    CCal	cc_temp;		/* Temperature calibration	    */
    CCal	cc_press;		/* Pressure calibration		    */
    CCal	cc_trans;		/* Transmissometer		    */
    CCal	cc_fluor;		/* Fluorometer			    */
    CCal	cc_sample;		/* Sample number		    */
} CTDCal;				/************************************/

typedef struct				/************************************/
{					/* Struct for decoded CTD data      */
    Nat16	ctd_sample;		/* Sample number		    */
    double	ctd_press;		/* Pressure			    */
    double	ctd_temp;		/* Temperature			    */
    double	ctd_cond;		/* Conductivity			    */
    double	ctd_sal;		/* Salinity			    */
    double	ctd_trans;		/* Transmissometer voltage	    */
    double	ctd_fluor;		/* Fluorometer voltage		    */
} CTDDecode;				/************************************/

typedef enum
{ NO3_DRIFT1, NO3_M1, NO3_M2 } No3Format;

typedef struct				/************************************/
{					/* Struct for No3 calibration	    */
    No3Format	nc_format;		/* Format that data is in	    */
    char	nc_samplechar;		/* Character that denotes sample    */
    Int32	nc_maxCounts;		/* Maximum A/D count		    */
    Int32	nc_absMult;		/* Absorption multiplier	    */
    double	nc_Co;			/* Conductivity reference	    */
    double	nc_Cf;			/* Conductivity frequency reference */
    double	nc_To;			/* Temperature reference	    */
    double	nc_Tf;			/* Temperature frequency reference  */
    double	nc_pressure;		/* Pressure to use		    */
} No3Cal;				/************************************/

#define	NO3_TIMELEN	16

typedef struct				/************************************/
{					/* Struct for decoded No3 data      */
    char	no3_time[NO3_TIMELEN];	/* Time string			    */
    Int16	no3_sample;		/* Sample number		    */
    double	no3_conc;		/* NO3 concentration		    */
    double	no3_temp;		/* Temperature at NO3 analyzer	    */
    double	no3_salinity;		/* Salinity			    */
    double	no3_conductivity;	/* Conductivity			    */
} No3Decode;				/************************************/

					/************************************/
#define	SPEC_BANKS	6		/* Number of A/D banks in spectro   */
					/* or addresses in PRR-600	    */
#define SPEC_CHANS	20		/* Number of A/D channels per bank  */
#define CHAN_NAMELEN	8		/* Length of channel name	    */
#define UNIT_NAMELEN	20		/* Length of units name	 	    */
					/************************************/

typedef struct				/************************************/
{					/* Struct for cal of one spectro chan*/
    Int		type;			/* 0 = unused, 1 = down irrad,	    */
					/* 2 = analog, 3 = up radiance,	    */
					/* 4 = up irrad, 5 = quadratic	    */
					/* 6 = Satlantic (a is add'l mult)  */
    char	name[CHAN_NAMELEN];	/* Name of channel		    */
    double	a;			/* x^2 coef (type 5) or multiplier (6)*/
    double	b;			/* multiplier (PRR) or divisor (MER)*/
    double	c;			/* subtractive offset		    */
    char	units[UNIT_NAMELEN];	/* Units of measure		    */
} SpecChanCal;				/************************************/

typedef struct				/************************************/
{					/* Spectro Calibration struct	    */
    Int		spc_totchans;		/* Total number of channels	    */
    Int		spc_nchans[SPEC_BANKS];	/* Number of active chans per bank  */
    SpecChanCal spc_cal[SPEC_BANKS][SPEC_CHANS]; /* Channel calibrations    */
} SpecCal;				/************************************/

typedef struct				/************************************/
{					/* Struct 1 chan of decoded spectro */
    double	spc_mean;		/* Mean				    */
    double	spc_std;		/* Standard deviation		    */
    double	spc_min;		/* Minimum value		    */
    double	spc_max;		/* Maximum value		    */
} SpecChanDecode;			/************************************/

typedef struct				/************************************/
{					/* Struct for decoded spectro data  */
    Int		spc_year;		/* Year of collection		    */
    Int		spc_day;		/* Julian day			    */
    Nat32	spc_sec;		/* Seconds since midnight	    */
    SpecChanDecode spc_chan[SPEC_BANKS][SPEC_CHANS]; /* Channel output values*/
} SpecDecode;				/************************************/

typedef double	SpecPRRDecode[SPEC_BANKS][SPEC_CHANS];

#define MAX_SAT_CHANS	12		/* Max number of satlantic channels */

typedef struct				/************************************/
{					/* Satlantic Spectro calibration type*/
    char	sat_name[CHAN_NAMELEN];	/* Channel name			    */
    double	sat_factor;		/* Calibration factor F		    */
    double	sat_immers;		/* Immersion factor		    */
    Int		sat_offset;		/* Offset counts		    */
} SatlantChanCal;			/************************************/

typedef struct				/************************************/
{					/* Struct for of decoded spectro data*/
    Int		sat_unused_chans;	/* Number of channels to skip	    */
    Int		sat_chans;		/* Number of channels to decode	    */
    SatlantChanCal sat_cal[MAX_SAT_CHANS]; /* Channel calibrations	    */
} SatlanticCal;				/************************************/

typedef double	SatlanticDecode[MAX_SAT_CHANS];


/************************************************/
/* Declarations for log records found/missing	*/
/************************************************/

#define NLOGS		20
#define NRANGES		20

typedef struct				/************************************/
{					/* Range of found records	    */
    Nat32	min;			/* Beginning of range		    */
    Nat32	max;			/* End of range			    */
} RcdRange;				/************************************/

typedef struct				/************************************/
{					/* Structure defining one OASIS log */
    Int32	nRanges;		/* Number of ranges found	    */
    RcdRange	range[NRANGES];		/* Log record number ranges	    */
} OneLog;				/************************************/

typedef struct				/************************************/
{					/* Structure defining logs found    */
    Int32	firstLog;		/* First log found		    */
    Int32	nLogs;			/* Number of logs found		    */
    OneLog	logs[NLOGS];		/* Logs found			    */
} LogStruct;				/************************************/
