/****************************************************************************/
/* Copyright 1991 MBARI                                                     */
/****************************************************************************/
/* $Header: decode.c,v 2.5 94/12/15 10:58:59 hebo Exp $			    */
/* Summary  : Program to decode an OASIS binary file			    */
/* Filename : decode.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 2.5 $							    */
/* Created  : 12/05/91							    */
/****************************************************************************/
/* Modification History:						    */
/* 05dec91 rah - created						    */
/* $Log:	decode.c,v $
 * Revision 2.5  94/12/15  10:58:59  10:58:59  hebo (Bob Herlien)
 * Accumlated minor changes, mainly due to move to tsunami
 * 
 * Revision 2.4  94/01/21  14:35:34  14:35:34  hebo (Bob Herlien)
 * Added support for date ranges in cfg file
 * 
 * Revision 2.3  94/01/17  11:09:31  11:09:31  hebo (Bob Herlien)
 * Misc changes
 * 
 * Revision 2.2  93/09/17  10:59:04  10:59:04  hebo (Bob Herlien)
 * Changed method of finding config file based on name of data file.
 * 
 * Revision 2.1  92/09/04  13:38:18  13:38:18  hebo (Bob Herlien)
 * Read CTD pressure from calibration file if no CTD pressure sensor.
 * 
 * Revision 2.0  92/08/31  15:35:26  15:35:26  hebo (Bob Herlien)
 * Auguest 1992 Deployment.  Changed to allow multiple sensors of same type.
 * 
 * Revision 1.3  92/06/15  09:07:23  09:07:23  hebo (Bob Herlien)
 * *** empty log message ***
 * 
 * Revision 1.2  92/05/12  18:18:39  18:18:39  hebo (Bob Herlien)
 * Added spectroradiometer decoding (spec.c)
 * 
 * Revision 1.1  92/03/16  15:42:08  15:42:08  hebo (Bob Herlien)
 * Ignore leading blanks on "begin" line
 * 
 * Revision 1.0  92/02/25  10:46:58  10:46:58  hebo (Bob Herlien)
 * Initial revision
*/
/****************************************************************************/

#include <stdio.h>			/* Standard I/O			    */
#include <stdlib.h>			/* For exit()			    */
#include <mbari/types.h>		/* MBARI type definitions	    */
#include <mbari/const.h>		/* MBARI constants		    */
#include <decode.h>			/* OASIS controller definitions	    */
#include <time.h>			/* Time				    */
#include <memory.h>			/* for memcpy()			    */
#include <math.h>			/* for sqrt()			    */
#include <ctype.h>			/* for isspace()		    */
#include <string.h>			/* for strcmp()			    */

#define UUBUFSIZE	512		/* Size of uu decode buffer	    */
#define BUFSIZE		2048		/* Size of sample buffer	    */
#define NAMESIZE	256		/* Space allocated for file names   */
#define UUERROR		(-2)		/* Error return from uuread()	    */
#define ANALOG_CHANS	8		/* Number of analog channels	    */
#define PAR_CHAN	0		/* Analog channel number for PARs   */
#define OASIS_CHAN	2		/* Analog chan num for OASIS stuff  */
#define CFG_DIR		"/oasis/cfg"

#define DEC(c)	(((c) - ' ') & 0x3f)	/* uudecode macro		    */


/********************************/
/*	External Functions	*/
/********************************/
Extern char	*get_can_name( char *dataname );
Extern Status	read_cfg( char **cfgname );
Extern Status	read_atlas_cal( char *name );
Extern Status	decode_atlas( Byte *atdata, Int atlen, 
			      AtlasDecode *adp, AtlasCal *acp );
Extern Status	read_ctd_cal( char *name, CTDCal *ccp );
Extern Status	decode_ctd( Byte *ctddata, Int len, 
			    CTDDecode *cdp, CTDCal *ccp );
Extern Status	read_no3_cal( char *name, No3Cal *no3cp );
Extern Status	decode_no3( Byte *no3data, Int len, 
			    No3Decode *no3dp, No3Cal *no3cp );
Extern Status	read_spec_cal( char *name, SpecCal *scp );
Extern Status	decode_spectro( Byte *spdata, Int splen, 
			        SpecDecode *sdp, SpecCal *scp );
Extern Status	decode_prr( Byte *spdata, Int len,
			    SpecPRRDecode sd, SpecCal *scp );
Extern Status	decode_satlantic( Byte *satdata, Int len,
				  SatlanticDecode sdp, SatlanticCal *scp );


/********************************/
/*	External Data		*/
/********************************/

Extern char	*optarg;		/* Option argument from getopt()    */
Extern Int	optind;			/* Option index from getopt()	    */
Extern Int	opterr;			/* getopt() error flag		    */


/********************************/
/*	Global Data		*/
/********************************/

Global Int	itime;			/* Integer year/day as yyddd	    */
Global AtlasCal	atlas_cal;		/* ATLAS calibration		    */
Global Analog	analog[ANALOG_CHANS];	/* Analog calibration		    */
Global CTDCal 	ctd_cal[NUM_CTDS];	/* Structs to hold CTD calibrations */
Global No3Cal 	no3_cal[NUM_NO3S];	/* Structs to hold NO3 calibrations */
Global SpecCal  spec_cal[NUM_SPECS];	/* Structs to hold Spect calibrations*/
Global SatlanticCal  satlantic_cal;	/* Struct to hold Satlantic cal	    */
Global char	*rcdfile = "/oasis/raw/oasis.rcd";


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal Byte	buffer[BUFSIZE];	/* Decoded data buffer		    */
MLocal Byte	uubuf[UUBUFSIZE];	/* Buffer for raw uuencoded data    */
MLocal LogRecHdr hdr;			/* Logging record header	    */
MLocal time_t	oasis_time;		/* Time of OASIS data in time_t fmt */
MLocal AtlasDecode atlas;		/* Buffer for decoded ATLAS data    */
MLocal Boolean	dolocaltime = TRUE;	/* Output local (not GMT) time	    */
MLocal Boolean	donumber = FALSE;	/* Output log numbers		    */
MLocal char	tmp_buf[NAMESIZE];	/* Misc temporary buffer	    */
MLocal Boolean	decode_all = TRUE;	/* Decode all sensors		    */
MLocal Boolean	decode_sensor[SENSORS];	/* TRUE to decode particular sensor */
MLocal SpecDecode	spec;		/* Struct to hold decoded Spect data*/
MLocal SpecPRRDecode	specprr;	/* Decoded PRR Spectro data	    */
MLocal SatlanticDecode	satlantic;	/* Decoded Satlantic data	    */

/* File names		*/
MLocal char	*cfgp = NULL;		/* Ptr to name of OASIS config file */
MLocal char	atlas_file[NAMESIZE];	/* Name of ATLAS calibration file   */
MLocal char	ctd_file[NAMESIZE];	/* Name of CTD calibration file     */
MLocal char	spec_file[NAMESIZE];	/* Name of Spectro calibration file */

MLocal char	*sensor_names[] = 
{ "Empty", "Time", "ATLAS", "OASIS", "PAR", "CTD", "Spectro", "ADCP",
  "GPS", "Modem", "pCO2", "CTD2", "CTD3", "Spectro10", "Spectro20",
  "Nitrate", "Nitrate2", "SpecPRR", "Satlantic", "GPS", "NRL", "Oxygen",
  "Fluorometer", "Transmissometer" };


/********************************/
/*	External Functions	*/
/********************************/

Extern FILE	*uuopen( char *name );
Extern Int32	get_begin_line( FILE *fd );
Extern Int	uuread( Byte *buf, Int len, FILE *fd );
Extern Int	uugetline( Byte *buf, Int len, FILE *fd );


/********************************/
/*	Forward Declarations	*/
/********************************/

Boolean	process_command_line ( Int argc, char **argv );
Void	use_msg( char *s );
Void	decode_file( char *s );
Nat16	getIntelword( Byte *p );
Nat32	getIntellong( Byte *p );
Void	print_header( char *ident );
Void	printbytes( Byte *s, Int len );
Void	printwords( Byte *p, Int len );
Void	print_ascii( Int len, Boolean strip_nl );
Void	print_analog( Nat16 chan, Int nchans );
Void	sensor_error( Status err );
Status	print_ctd( Int len, CTDCal *ccp );
Status  print_gps( Int len );
Status  print_gps_type2( Int len );
Status  print_spectro( Int len, SpecCal *scp );
Status  print_spec_prr( Int len, SpecCal *scp );
Status  print_satlantic( Int len, SatlanticCal *scp );
Status	print_adcpv1( Void );


/************************************************************************/
/* Function    : main							*/
/* Purpose     : Main routine						*/
/* Inputs      : argc, argv						*/
/* Outputs     : none							*/
/************************************************************************/
	Void
main( Int argc, char **argv )
{
    Int		i;
    char	*filename, *cp;

    if ( !process_command_line(argc, argv) )
    {
	use_msg(argv[0]);
	exit( 1 );
    }

    for ( i = optind; i < argc; i++ )
    {
	filename = argv[i];
	get_can_name( filename );
	cp = cfgp;
	if ( read_cfg(&cp) != OK )
	    printf("Can't read configuration file \"%s\"\n", cp);
	else
	    decode_file( filename );
    }

    exit( 0 );

} /* main() */


/************************************************************************/
/* Function    : decode_file						*/
/* Purpose     : Decode one data file, print results to stdout		*/
/* Inputs      : File name						*/
/* Outputs     : none							*/
/************************************************************************/
	Void
decode_file( char *filename )
{
    Int		cc, len, len_got;
    FILE	*fd;
    struct tm	*tp;
    Boolean	did_err_msg;
    Status	rtn;

    if ( (fd = uuopen(filename)) == (FILE *)NULL )
	return;

    if( get_begin_line(fd) < 0 )
    {
	printf("No begin line in %s\n", filename);
	return;
    }

    did_err_msg = FALSE;

    while ( (cc = uugetline(buffer, sizeof(buffer), fd)) != EOF )
    {
	if ( cc == UUEND )
	{
	    if( get_begin_line(fd) < 0 )
		break;
	    else
		continue;
	}

	if ( cc == 0 )
	    continue;

	did_err_msg = FALSE;

	hdr.log_type = buffer[0];
	hdr.log_nmbr = getIntelword(&buffer[1]);
	hdr.log_len = getIntelword(&buffer[3]);
	hdr.log_time = getIntellong(&buffer[5]);
	oasis_time =(time_t)hdr.log_time;
	tp = gmtime( &oasis_time );
	itime = (1000 * tp->tm_year) + tp->tm_yday + 1;

	len_got = cc - 9;		/* compute amount of log data gotten*/
	memmove( buffer, buffer + 9, sizeof(buffer) - 9 );
					/* Move log data to start of buffer*/

	if ( (len = hdr.log_len) > sizeof(buffer) )
	{				/* Get size of log data		*/
	    len = sizeof(buffer);
	    printf("Record too long in %s.  Truncating.\n", filename);
	}

	if ( len_got < len )
	{
	    if ( (len_got += uuread(buffer+len_got, len-len_got, fd)) != len)
	    {
		if ( !did_err_msg )
		    printf("Bad record in %s\n", filename);
		did_err_msg = TRUE;
		continue;
	    }
	    did_err_msg = FALSE;
	}

	if ( !decode_all && 
	     !((hdr.log_type < SENSORS) && decode_sensor[hdr.log_type]) )
	    continue;

	if ( hdr.log_type < SENSORS )
	    print_header( sensor_names[hdr.log_type] );

	switch( hdr.log_type )
	{
	  case LOG_EMPTY:
	    printbytes(buffer, len);
	    break;

	  case TIME:
	    oasis_time = (time_t)getIntellong( buffer );
	    tp = gmtime( &oasis_time );
	    printf("%s", asctime(tp));
	    break;

	  case ATLAS:
	    if ( (rtn = decode_atlas(buffer, len, &atlas, &atlas_cal)) != OK )
	    {
		sensor_error( rtn );
		break;
	    }

	    printf("Time %9.5f  Air %6.3f  SST %6.3f\n",
		   atlas.atd_time, atlas.atd_air, atlas.atd_sst);
	    printf("         ");
	    for ( cc = 0; cc < TEMPS; cc++ )
		printf("%6.3f ", atlas.atd_temp[cc]);
	    printf("\n         Press %6.2f %6.2f  RH %5.2f", 
		   atlas.atd_press[0], atlas.atd_press[1], atlas.atd_rh);
	    printf("  Wind %6.3f m/s toward %5.1f\n", 
		   atlas.atd_windspd, atlas.atd_winddir);
	    printf("         Compass %3.0f  Vane %3.0f  Raw dirctn %3.0f  Raw speed %5.2f\n",
		   atlas.atd_compass, atlas.atd_vane, 
		   atlas.atd_rawdir, atlas.atd_rawspeed);
	    break;

	  case OASIS_CAN:
	    print_analog( OASIS_CHAN, len/2 );
	    break;

	  case PAR:
	    print_analog( PAR_CHAN, len/2 );
	    break;

	  case CTD:
	    print_ctd( len, &ctd_cal[CTD_CAL] );
	    break;

	  case CTD2:
	    print_ctd( len, &ctd_cal[CTD2_CAL] );
	    break;

	  case CTD3:
	    print_ctd( len, &ctd_cal[CTD3_CAL] );
	    break;

	  case SPECTRO:
	    print_spec( len, &spec_cal[SPECTRO_CAL] );
	    break;

	  case SPECTRO2:
	    print_spec( len, &spec_cal[SPECTRO2_CAL] );
	    break;

	  case SPECTRO3:
	    print_spec( len, &spec_cal[SPECTRO3_CAL] );
	    break;

	  case SPEC_PRR:
	    print_spec_prr( len, &spec_cal[SPECPRR_CAL] );
	    break;

	  case ADCP:
	    print_adcpv1();
	    printbytes( buffer, len );
	    break;

	  case GPS:
	    print_gps( len );
	    break;

	  case GPS_TYPE2:
	    print_gps_type2( len );
	    break;

	  case PCO2:
	    print_ascii(len, TRUE);
	    break;

	  case NO3:
	    print_ascii(len, FALSE);
	    print_nitrate_decoded( len, &no3_cal[NO3_CAL] );
	    break;

	  case NO32:
	    print_ascii(len, FALSE);
	    print_nitrate_decoded( len, &no3_cal[NO3_CAL2] );
	    break;

	  case SATLANTIC:
	    print_satlantic( len, &satlantic_cal );
	    break;

	  case NRL:
	    print_ascii(len, FALSE);
	    break;

	  case O2:
	    print_ascii(len, TRUE);
	    break;

	  case FLUOR:
	    print_analog( FLUOR_CHAN, len/2 );
	    break;

	  case TRANSMISS:
	    print_analog( TRANSMISS_CHAN, len/2 );
	    break;

	  case LOG_ERROR:
	    cc = getIntelword(buffer);
	    if ( cc & 0x8000 ) 
	    {
		cc &= 0x7fff;
		print_header("OASIS Error");
		if ( cc & RAM_ERR )
		    printf("RAM_init ");
		if ( cc & LOG_ERR )
		    printf("LOG_memory_bad ");
		if ( cc & CLOCK_ERR )
		    printf("Clock_failure ");
		if ( cc & INT_ERR )
		    printf("Spurious_interrupt ");
		if ( cc & RESTART_ERR )
		    printf("Restarted ");
		if ( cc & COMM_ERR )
		    printf("24hr_silence");
		printf("\n");
	    }
	    else
	    {
		if ( cc >= SENSORS )
		    printf("Unknown Error type\n");
		else
		{
		    printf( sensor_names[cc] );
		    print_header( " Error" );
		    printf("\n");
		}
	    }
	    break;

	  case LOG_COMMENT:
	    print_header("Comment");
	    for ( cc = 0; cc < len; cc++ )
		    putchar( buffer[cc] );
	    printf("\n");
	    break;

	  case LOG_BIN:
	    switch( cc = getIntelword(buffer) )
	    {
	      case SPECTRO:
	      case SPECTRO2:
	      case SPECTRO3:
		printf( sensor_names[cc] );
		print_header( " V cmd" );
		printbytes( &buffer[2], len - 2 );
		break;

	      default:
		print_header( "Binary Comment" );
		printbytes( buffer, len );
	    }
	    break;

	  default:
	    print_header("Unknown record type\n");
	}
    }

    fclose( fd );

} /* decode_file() */


/************************************************************************/
/* Function : use_msg							*/
/* Purpose  : Print Usage Message					*/
/* Inputs   : Name of program						*/
/* Outputs  : None							*/
/************************************************************************/
	Void
use_msg( char *s )
{
    fprintf( stderr,
	    "Usage: %s [-c cfg_file] [-i instrument] [-g] [-l] [-n]", s );

} /* use_msg() */


/************************************************************************/
/* Function : process_command_line					*/
/* Purpose  : Read the arguments from the command line			*/
/* Inputs   : argc, argv from main() routine				*/
/* Outputs  : TRUE if arguments OK, else FALSE				*/
/************************************************************************/
	Boolean
process_command_line ( Int argc, char **argv )
{
    Int		i, baud;

    for ( i = 0; i < SENSORS; i++ )
	decode_sensor[i] = FALSE;

    while ( (i = getopt(argc, argv, "c:gi:ln")) != EOF )
	switch( i )
	{
	  case 'c':
	    cfgp = optarg;
	    break;

	  case 'g':
	    dolocaltime = FALSE;
	    break;

	  case 'i':
	    for ( i = 0; i < SENSORS; i++ )
		if ( strcmp(optarg, sensor_names[i]) == 0 )
		{
		    decode_all = FALSE;
		    decode_sensor[i] = TRUE;
		}
	    break;

	  case 'l':
	    dolocaltime = TRUE;
	    break;

	  case 'n':
	    donumber = TRUE;
	    break;

	  default:
	    return( FALSE );
	}

    return( TRUE );

} /* process_command_line() */


/************************************************************************/
/* Function    : getIntelword						*/
/* Purpose     : Get a word in Intel format from data stream		*/
/* Inputs      : Ptr to data stream					*/
/* Outputs     : Word							*/
/************************************************************************/
	Nat16
getIntelword( Byte *p )
{
    Nat16	rtn;

    rtn = (Nat16)(*p);
    rtn += (Nat16)(p[1] << 8);
    return( rtn );

} /* getIntelword() */


/************************************************************************/
/* Function    : getIntellong						*/
/* Purpose     : Get a longword in Intel format from data stream	*/
/* Inputs      : Ptr to data stream					*/
/* Outputs     : Long							*/
/************************************************************************/
	Nat32
getIntellong( Byte *p )
{
    Nat32	rtn;

    rtn = (Nat32)getIntelword(p);
    rtn += ((Nat32)getIntelword(&p[2]) << 16);
    return( rtn );

} /* getIntellong() */


/************************************************************************/
/* Function    : getMotword						*/
/* Purpose     : Get a word in Motorola format from data stream		*/
/* Inputs      : Ptr to data stream					*/
/* Outputs     : Word							*/
/************************************************************************/
	Nat16
getMotword( Byte *p )
{
    Nat32	rtn;

    rtn = (Nat16)(*p << 8);
    rtn += (Nat16)(p[1]);
    return( rtn );

} /* getMotword() */


/************************************************************************/
/* Function    : print_header						*/
/* Purpose     : Print log header					*/
/* Inputs      : Identification string					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
print_header( char *ident )
{
    struct tm	*tp;

    if ( dolocaltime )
	tp = localtime( (time_t *)&hdr.log_time );
    else
	tp = gmtime( (time_t *)&hdr.log_time );
    printf( "%-8s ", ident );
    if ( donumber )
	printf("%4d ", hdr.log_nmbr);
    printf("%02d/%02d/%02d %02d:%02d:%02d ", tp->tm_year, tp->tm_mon+1,
	   tp->tm_mday, tp->tm_hour, tp->tm_min, tp->tm_sec);


} /* print_header() */


/************************************************************************/
/* Function    : printbytes						*/
/* Purpose     : Print data stream as a stream of bytes			*/
/* Inputs      : Byte ptr, length					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printbytes( Byte *s, Int len )
{
    Int		i, curlen;

    for ( i = 0; i < len;  )
    {
	printf("\n         ");
	curlen = len - i;
	if ( curlen > 32 )
	    curlen = 32;
	while ( curlen-- )
	    printf("%02x", s[i++]);
    }
    printf("\n");

} /* printbytes() */


/************************************************************************/
/* Function    : printwords						*/
/* Purpose     : Print data stream as a stream of words			*/
/* Inputs      : Byte ptr, length					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printwords( Byte *p, Int len )
{
    Int		i, curlen, wrd;

    for ( i = 0; i < len; i += 2 )
    {
	wrd = getIntelword( p + i );
	printf("%4d ", wrd);
    }

    printf("\n");

} /* printwords() */


/************************************************************************/
/* Function    : print_ascii						*/
/* Purpose     : Print data stream as ascii stream			*/
/* Inputs      : Length, Boolean to strip CR & LF			*/
/* Outputs     : None							*/
/************************************************************************/
	Void
print_ascii( Int len, Boolean strip_nl )
{
    char	*p;

    buffer[len] = '\0';

    if ( strip_nl )
    {
	while ( (p = strchr((char *)buffer, '\r')) != NULL )
	    *p = ' ';

	while ( (p = strchr((char *)buffer, '\n')) != NULL )
	    *p = ' ';
    }

    printf("%s\n", buffer);

} /* print_ascii() */


/************************************************************************/
/* Function    : print_analog						*/
/* Purpose     : Print data stream as analog channel data		*/
/* Inputs      : First channel number, number of channels		*/
/* Outputs     : None							*/
/************************************************************************/
	Void
print_analog( Nat16 chan, Int nchans )
{
    Int		i;
    Reg Analog	*ap;

    for ( i = 0; i < nchans; i++ )
    {
	ap = &analog[chan + i];
	printf("%6.2f %s  ", 
	       (((ap->a * (double)getIntelword(&buffer[2*i])) + ap->b) * ap->c)
	       + ap->d, ap->units);
    }

    printf("\n");

} /* print_analog() */


/************************************************************************/
/* Function    : print_gps						*/
/* Purpose     : Print buffer as GPS data				*/
/* Inputs      : Length of buffer					*/
/* Outputs     : OK or SIZE_ERR						*/
/************************************************************************/
	Status
print_gps( Int len )
{
    Int32	lat, lon;
    Nat32	var;
    Nat16	samples, status;
    char	lathem, lonhem;

    if ( len != 16 )
    {
	printf("Bad GPS record size, was %d, should be 16\n",len);
	return( SIZE_ERR );
    }

    lat = getIntellong(buffer);
    lon = getIntellong(&buffer[4]);
    var = getIntellong(&buffer[8]);
    samples = getIntelword(&buffer[12]);
    status = getIntelword(&buffer[14]);

    lathem = 'N';
    lonhem = 'W';

    if ( lat < 0 )
    {
	lat = -lat;
	lathem = 'S';
    }

    if ( lon < 0 )
    {
	lon = -lon;
	lonhem = 'W';
    }

    printf("%3d:%02d.%03d%c %3d:%02d.%03d%c StdDev %.2f Samp %d Stat %04x\n",
	   lat/60000, (lat/1000)%60, lat%1000, lathem, lon/60000,
	   (lon/1000)%60, lon%1000, lonhem, sqrt((double)var),
	   samples, status);

    return( OK );

} /* print_gps() */


/************************************************************************/
/* Function    : print_gps_type2					*/
/* Purpose     : Print buffer as GPS data				*/
/* Inputs      : Length of buffer					*/
/* Outputs     : OK or SIZE_ERR						*/
/************************************************************************/
	Status
print_gps_type2( Int len )
{
    Int32	lat, lon;
    Nat32	var;
    Nat16	samples, totTime, status;

    if ( len != 18 )
    {
	printf("Bad GPS record size, was %d, should be 18\n",len);
	return( SIZE_ERR );
    }

    lat = getIntellong(buffer);
    lon = getIntellong(&buffer[4]);
    var = getIntellong(&buffer[8]);
    samples = getIntelword(&buffer[12]);
    totTime = getIntelword(&buffer[14]);
    status = getIntelword(&buffer[16]);

    printf("%10.5f %10.5f StdDev %.2f Samp %d Time %d Stat %04x\n",
	   lat/10000000., lon/10000000., sqrt((double)var),
	   samples, totTime, status);

    return( OK );

} /* print_gps_type2() */


/************************************************************************/
/* Function    : print_ctd						*/
/* Purpose     : Print buffer as CTD data				*/
/* Inputs      : Length of buffer, CTD calibration struct		*/
/* Outputs     : OK or Error type
/************************************************************************/
	Status
print_ctd( Int len, CTDCal *ccp )
{
    CTDDecode	ctd;			/* Struct to hold decoded CTD data  */
    Status	rtn;			/* Return code			    */

    if ( (rtn = decode_ctd(buffer, len, &ctd, ccp)) != OK )
    {
	sensor_error( rtn );
	return( rtn );
    }

    printf("Num %d  Press %6.3f  Temp %6.3f  Cond %6.3f\n",
	   ctd.ctd_sample, ctd.ctd_press, ctd.ctd_temp, ctd.ctd_cond);

    printf("         Salinity %6.4f  Transmiss %6.4f  Fluor %6.4f\n", 
	   ctd.ctd_sal, ctd.ctd_trans, ctd.ctd_fluor);

    return( OK );

} /* print_ctd() */


/************************************************************************/
/* Function    : print_nitrate_decoded					*/
/* Purpose     : Decode and print decoded nitrate data			*/
/* Inputs      : Length, calibration struct				*/
/* Outputs     : OK or Error type					*/
/************************************************************************/
	Status
print_nitrate_decoded( Int len, No3Cal *no3cp )
{
    Reg Byte	*p, *q;			/* Scratch ptrs			    */
    No3Decode	no3;			/* Struct to hold decoded NO3 data  */
    Status	rtn;			/* Return code			    */

    buffer[len] = '\0';

    for ( p = buffer; p - buffer < len; p = q )
    {					/* Loop to decode one line at a time*/
	while ( (p - buffer < len) && ((*p == '\r') || (*p == '\n')) )
	    p++;

	if ( (q = (Byte *)strchr((char *)p, '\r')) == NULL )
	    q = (Byte *)strchr((char *)p, '\n');

	if ( (q != NULL) && (q - buffer < len) )
	    *q++ = '\0';
	else
	    q = p + strlen((char *)p) + 1;

	if ( (rtn = decode_no3(p, len, &no3, no3cp)) == OK )
	{
	    if ( no3cp->nc_format == NO3_DRIFT1 )
		printf("%-.14s  no3 %6.4f  Temp %6.3f  Salinity %6.4f  Cond %6.4f\n", 
			     no3.no3_time, no3.no3_conc, no3.no3_temp,
			     no3.no3_salinity, no3.no3_conductivity);
	    else
		printf("%-.14s  no3 %6.4f  Temp %6.3f\n", 
			     no3.no3_time, no3.no3_conc, no3.no3_temp);
	}
    }

} /* print_nitrate_decoded() */


/************************************************************************/
/* Function    : print_spec						*/
/* Purpose     : Print buffer as Spectroradiometer data			*/
/* Inputs      : Length of buffer, Spectro calibration struct		*/
/* Outputs     : OK or Error type					*/
/************************************************************************/
	Status
print_spec( Int len, SpecCal *scp )
{
    Int		i, j, n;		/* Scratch			    */
    Status	rtn;			/* Return code			    */

    if ( (rtn = decode_spectro(buffer, len, &spec, scp)) != OK )
    {
	sensor_error( rtn );
	return( rtn );
    }

    printf("Time %9.5f", spec.spc_day + (double)(spec.spc_sec)/86400.0);

    for ( i = 0, n = 100; i < SPEC_BANKS; i++ )
	for ( j = 0; j < scp->spc_nchans[i]; j++ )
	{
	    if ( n >= 64 )
	    {
		printf("\n         ");
		n = 10;
	    }
	    if ( scp->spc_cal[i][j].type > 0 )
		n += printf("%-5.5s %9.6f  ", scp->spc_cal[i][j].name,
			    spec.spc_chan[i][j].spc_mean);
	}

    printf("\n");

    return( OK );

} /* print_spec() */


/************************************************************************/
/* Function    : print_spec_prr						*/
/* Purpose     : Print buffer as PRR-600 Spectroradiometer data		*/
/* Inputs      : Length of buffer, Spectro calibration struct		*/
/* Outputs     : OK or Error type					*/
/************************************************************************/
	Status
print_spec_prr( Int len, SpecCal *scp )
{
    Int		bank, chan, cursor;	/* Bank, chan nums, cursor position*/
    Status	rtn;			/* Return code			*/

    if ( (rtn = decode_prr(buffer, len, specprr, scp)) != OK )
    {
	sensor_error( rtn );
	return( rtn );
    }

    for ( bank = 0, cursor = 100; bank < SPEC_BANKS; bank++ )
	for ( chan = 0; (chan < scp->spc_nchans[bank]); chan++ )
	{
	    if ( cursor >= 60 )
	    {
		printf("\n         ");
		cursor = 0;
	    }
	    if ( scp->spc_cal[bank][chan].type > 0 )
		cursor += printf("%-5.5s %9.5f  ",
				 scp->spc_cal[bank][chan].name,
				 specprr[bank][chan]);
	}

    printf("\n");

    return( OK );

} /* print_spec_prr() */


/************************************************************************/
/* Function    : print_satlantic					*/
/* Purpose     : Print buffer as Satlantic spectroradiometer data	*/
/* Inputs      : Length of buffer, Satlantic calibration struct		*/
/* Outputs     : OK or Error type					*/
/************************************************************************/
	Status
print_satlantic( Int len, SatlanticCal *scp )
{
    Int		chan, cursor;		/* Channel number, cursor position  */
    Status	rtn;			/* Return code			    */

    if ( (rtn = decode_satlantic(buffer, len, satlantic, scp)) != OK )
    {
	sensor_error( rtn );
	return( rtn );
    }

    for ( chan = 0, cursor = 80; chan < scp->sat_chans; chan++ )
    {
	if ( cursor >= 64 )
	{
	    printf("\n         ");
	    cursor = 10;
	}
	cursor += printf("%s %9.6f  ", scp->sat_cal[chan].sat_name,
			 satlantic[chan]);
    }

    printf("\n");

    return( OK );

} /* print_satlantic() */


/************************************************************************/
/* Function    : print_adcpv1						*/
/* Purpose     : Print V1 voltage of ADCP				*/
/* Inputs      : None							*/
/* Outputs     : OK							*/
/************************************************************************/
#define V1_OFS	45
#define V1_MULT	0.05

	Status
print_adcpv1( Void )
{
    printf( "  V1 %5.2f", V1_MULT * (double)((unsigned char)buffer[V1_OFS]));

    return( OK );

} /* print_adcpv1() */


/************************************************************************/
/* Function    : sensor_error						*/
/* Purpose     : Print error in decoding sensor data			*/
/* Inputs      : Error type						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
sensor_error( Status err )
{
    switch( err )
    {
      case ERROR:
	printf("Error\n");
	break;

      case SIZE_ERR:
	printf("Wrong size\n");
	break;

      case CHKSUM_ERR:
	printf("Checksum error\n");
	break;

      case FMT_ERR:
	printf("Bad data format\n");
	break;

      default:
	printf("Unknown error\n");
    }

} /* sensor_error() */
