/****************************************************************************/
/* Copyright 1992 MBARI                                                     */
/****************************************************************************/
/* Summary  : Program to compute calibration coefficients for ATLAS temp pods*/
/* Filename : calsh5.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* Version  : 1.0                                                           */
/* Created  : 08/05/92							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 05aug92 rah - translated calsh5.f Fortran pgm written by J. Holbrook, 1975*/
/****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>

/*
c     program to compute steinhart-hart equation coefficients by least  
c     squares fit to (r,t) data in (resistance,degrees centigrade). 
c 
c     programmed by jim holbrook, aug 75. 
c     n = number of data points 

c--modifed to run on either old-style temp boards or newer self
c--calibrating boards.  Will check for flag after sensor number
c--If ieng = blank or 0, will assume old style board.  
c--If ieng = 1, will assume new style board.
c--Should be able to mix and match boards...
c--This will replace CALSH4 and CALENG
c--jan 91, ljm
C 
c  Ported to HP-UX running on hp9000/850 - rah, 20mar91
*/

#define NTEMPS	20
#define NPODS	20

FILE	*datfp, *outfp, *tblfp, *coeffp;
char	buff[1024];
time_t	curtime;
struct tm *curtm;
char	idate[40];
int	ntherm, ntemp, ihx, iflg, i, j;
double	r[NTEMPS], t[NTEMPS], rr,tt,r1,r2,r3,r4,r5,r6,r7,r8,r9,r10;
double	ymax1, ymin1;
int	ns;
int	nc[NTEMPS], res[NPODS][NTEMPS];
int	teng[NPODS][NTEMPS], ieng[NTEMPS], neng[NTEMPS], line[NTEMPS];
double	a,b,c,aa,bb,cc,reng[NTEMPS];
double	tsum, tmin, tmax;
double	rs, tr, td, trr;
double	amax;


/************************************************************************/
/* Function    : open_file						*/
/* Purpose     : Open an input or output file				*/
/* Inputs      : File name, file suffix, open type			*/
/* Outputs     : FILE *							*/
/* Comments    : Exits with error message if failed			*/
/************************************************************************/
    FILE *
open_file( char *name, char *s, char *type )
{
    FILE	*fp;

    sprintf( buff, "%s.%s", name, s );
    if ( (fp = fopen(buff, type)) == NULL )
    {
	printf("Could not open %s\n", buff);
	exit( 0 );
    }

    return( fp );

} /* open_file() */


/************************************************************************/
/* Function    : main							*/
/* Purpose     : Main routine						*/
/* Inputs      : argc, argv						*/
/* Outputs     : None							*/
/************************************************************************/
    void
main( int argc, char **argv )
{
    if ( argc <= 1 )
    {
	printf("Usage: %s filename\n", argv[0]);
	exit( 0 );
    }

    time( &curtime );
    curtm = localtime( &curtime );
    strftime( idate, 40, "%x", curtm );
    datfp = open_file(argv[1], "dat", "r");
    outfp = open_file(argv[1], "out", "w");
    tblfp = open_file(argv[1], "tbl", "w");
    coeffp = open_file(argv[1], "coef", "w");
    fgets( buff, sizeof(buff), datfp );
    sscanf( buff, " %d %d %d", &ntherm, &ntemp, &ihx );

  label921:
    iflg = 0;

/* c--read in bath temps */
    for ( i = 0; i < ntemp; i++ )
	fscanf( datfp, " %lg", &t[i] );
    ymax1 = 0.0;
    ymin1=0.0;

/* c--for each sensor, calculate coeff */
    for ( ns = 0; ns < ntherm; ns++ )
    {
	ieng[ns] = 0;
	r1=r2=r3=r4=r5=r6=r7=r8=0.0;
	fgets( buff, sizeof(buff), datfp );
	sscanf( buff, " %d %d", &line[ns], &ieng[ns] );

	fprintf( outfp, "          Thermistor Calibration of - %5d\n\n",
		 line[ns] );
	fprintf( outfp, "          Steinhart-Hart Equation Fit\n\n" );

	for ( i = 0; i < ntemp; i++ )
	    fscanf( datfp, (ihx==1 ? " %x" : " %d"), &nc[i] );

	sscanf( buff, " %d %d", &line[ns], &ieng[ns] );

	if ( ieng[ns] == 0 )
	{
	    fprintf( outfp, "          Old-Style Board\n" );
	    fprintf( outfp, "          R = 7.68E+08/(2.525*N - 7680)\n\n\n" );
	}
	else
	{
	    iflg = 1;
	    for ( i = 0; i < ntemp; i++ )
		fscanf( datfp, " %x", &neng[i] );
	    fprintf( outfp, "          Self-Calibrating Board\n" );
	    fprintf( outfp, "             R = 3.072E+08/N\n\n\n" );
	}

	for ( i = 0; i < ntemp; i++ )
	{
	    if ( ihx != 2 )
	    {
		if ( ieng[ns] == 1 )
		{    
		    r[i] = 3.072e+08/nc[i];
		    reng[i] = 3.072e+08/neng[i];
		}
		else
			r[i] = 7.68e+08/(2.525*nc[i]-7680.);
	    }
	    else
		r[i]=nc[i];

	    rr = r[i]; 
	    tt = t[i];
	    rr = log10(rr);
	    tt = 1.0/(tt+273.15);
	    r1 = r1+rr;
	    r2 = r2+rr*rr;
	    r3 = r3+rr*rr*rr;
	    r4 = r4+rr*rr*rr*rr;
	    r5 = r5+rr*rr*rr*rr*rr*rr;
	    r6 = r6+tt;
	    r7 = r7+tt*rr;
	    r8 = r8+tt*rr*rr*rr;
	}

	r9 = ntemp;
	r10=r9*(r2*r5-r4*r4)-r1*(r1*r5-r4*r3) +r3*(r1*r4-r2*r3);

	if ( r10 == 0.0 )
	{
	    fprintf( outfp, "          det = 0, no coefficients\n" );
	    goto label921;
	}

	a = r6*(r2*r5-r4*r4)-r1*(r7*r5-r4*r8)+r3*(r7*r4-r2*r8);
	a = a/r10;
	b = r9*(r7*r5-r4*r8)-r6*(r1*r5-r4*r3)+r3*(r1*r8-r3*r7);
	b = b/r10;
	c = r9*(r2*r8-r7*r4)-r1*(r1*r8-r3*r7)+r6*(r1*r4-r2*r3);
	c = c/r10;
	aa = a;
	bb = b; 
	cc = c;

	fprintf( outfp, "          A = %14.7g\n", aa );
	fprintf( outfp, "          B = %14.7g\n", bb );
	fprintf( outfp, "          C = %14.7g\n\n\n", cc );

	fprintf( coeffp, "%5d %s %15.7g %15.7g, %15.7g\n",
		 line[ns], idate, aa, bb, cc );
	
	if ( ieng[ns] == 0 )
	    fprintf(outfp, "                         R          T        T(R)     RESIDUAL\n");
	else
	    fprintf(outfp, "                         R          T        T(R)     RESIDUAL   N(ENG)  T(ENG)\n");

	tsum = 0.0;
	tmin = 0.0;
	tmax = 0.0;

	for ( i = 0; i < ntemp; i++ )
	{
	    rs = log10(r[i]);
	    tr = a+b*rs+c*rs*rs*rs; 
	    tr = 1.0/tr-273.15;
	    td = t[i]-tr;
	    trr = tr;

	    if ( ieng[ns] != 0 )
	    {
		rs = log10(reng[i]);
		tr = a+b*rs+c*rs*rs*rs;
		teng[i][ns] = 1.0/tr-273.15;
		fprintf(outfp, 
			"   %5x %6d    %12.5f%10.4f%10.4f%10.4f   %5x%10.4f\n",
			nc[i], nc[i], r[i], t[i], trr, td, neng[i]);
	    }
	    else
		fprintf(outfp, "   %5x %6d    %12.5f%10.4f%10.4f%10.4f\n",
			nc[i], nc[i], r[i], t[i], trr, td);

	    if ( td < tmin )
		tmin = td;
	    if ( td > tmax )
		tmax = td;

	    tsum = tsum + td*td;
	    res[i][ns] = td;
	}

	fprintf(outfp, "%f %f\n\n\n", tmin, tmax);

	tsum = sqrt(tsum/(ntemp-3));
	amax = (tmax>fabs(tmin)) ? tmax : fabs(tmin);

	fprintf(outfp, "          Maximum Residual = %8.4f\n", amax);
	fprintf(outfp, "          Standard Error   = %8.4f\n", tsum);

/* c  Write out Linda's table of information for this sensor */

	if ( ns == 1 )
	{
	    fprintf(tblfp,"                                 ATLAS\n");
	    fprintf(tblfp,"                                 TEMPERATURE CALIBRATIONS\n");
	    fprintf(tblfp,"                                 ------------------------\n\n");
	    fprintf(tblfp,"    Date:%s              Counts/Resist/Temp Conv:\n\n");
	    fprintf(tblfp,"    Observers___________            R=\n\n");
	    fprintf(tblfp,"    Calibration Program:%10s  T=\n\n", argv[0]);
	    fprintf(tblfp,"    Temperature range_______________\n\n");
	    fprintf(tblfp,"    ________________________________________________\n\n");
	    fprintf(tblfp,"    Comments regarding calibration:\n\n\n\n\n\n\n\n");
	    fprintf(tblfp,"    ________________________________________________\n\n");
	    fprintf(tblfp,"                  A          B        C       Max      Std\n");
	    fprintf(tblfp,"    Sensor      *E-02      *E-03    *E-05    Resid    Error  Comments\n\n");
	    
	}
	fprintf(tblfp, "   %c 5d %11.7f%11.7f%11.7f%9.4f%9.4f\n",
		line[ns], (ieng[ns] == 1) ? 'S' : ' ', 
		.01*aa, .001*bb, .00001*cc, amax, tsum);

	if ( tmin < ymin1 )
	    ymin1 = tmin;
	if ( tmax > ymax1 )
	    ymax1 = tmax;
    }

/* c--write out eng cable data if available */

    if ( iflg == 0 )
	goto label996;

    fprintf(tblfp,
	    "------------------------------------------------------\n\n");
    fprintf(tblfp,
	    "                        Self-Calibrating Board Summary:\n\n");
    fprintf(tblfp,"    Sensor ");

    for ( i = 0; i < ntemp; i++ )
	fprintf(tblfp, "%6.2f", t[i]);
    fprintf( tblfp, "\n");

    for ( i = 0; i < ntherm; i++ )
	if ( ieng[i] == 1 )
	{
	    fprintf(tblfp,"     %5d ", line[i]);
	    for ( j = 0; j < ntemp; j++ )
		fprintf(tblfp, "%6.2f", teng[j][i]);
	    fprintf(tblfp, "\n");
	}

  label996:
    fclose(datfp);
    fclose(outfp);
    fclose(tblfp);
    fclose(coeffp);

} /* main() */
