/************************************************************************/
/* Copyright 1997 MBARI							*/
/************************************************************************/
/* $Header: /home/cvs/oasis3/src/operations/src/argfilter.c,v 1.1 2003/08/20 19:39:43 headley Exp $		    */
/* Summary  : Program to filter ARGOS messages for one PTT number	*/
/* Filename : argfilter.c						*/
/* Author   : Bob Herlien (rah)						*/
/* Project  : OASIS							*/
/* Version  : 1.0							*/
/* Created  : 01/07/97							*/
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/************************************************************************/
/* Modification History:						*/
/* 7jan97 rah - created							*/
/* $Log: argfilter.c,v $
/* Revision 1.1  2003/08/20 19:39:43  headley
/* no message
/*
 * Revision 3.4  2002/03/01  15:38:38  15:38:38  oasisa (Oasis users)
 * Support for Local receiver input, WIN32
 * 
 * Revision 3.3  2001/06/19  13:00:42  13:00:42  oasisa (Oasis users)
 * Periodic Update w/ changes to utils 
 * plus new utils 6/19/2001 (klh)
 * 
 * Revision 3.0  99/05/12  10:11:31  10:11:31  bobh (Bob Herlien)
 * Added tstring, misc changes
 * 
 * Revision 2.9  98/08/24  13:46:00  13:46:00  bobh (Bob Herlien)
 * Archiving sources after M2/M3 & Eqpac deployments of 1998
 * 
 * Revision 2.8  98/03/17  11:11:47  11:11:47  bobh (Bob Herlien)
 * Archiving sources prior to porting to DOS/Windows
 * 
 * Revision 1.1  97/10/27  09:53:19  09:53:19  bobh (Bob Herlien)
 * Initial revision
 * 
*/
/************************************************************************/

#include <stdio.h>			/* Unix standard I/O		*/
#include <stdlib.h>			/* for atoi() */
#include <string.h>			/* for strstr() */
#include <mbariTypes.h>		/* MBARI standard types		*/
#include <mbariConst.h>		/* MBARI standard constants	*/
#include <time.h>			/* Unix time functions		*/

#define DEFAULT_PTT_NUM 11909
#define MAX_TARGETS	16
#define BUFSIZE		1024

typedef enum				/********************************/
{					/* Type of input		*/
    SVC_ARGOS,				/* From Serivce Argos		*/
    RCVR_NORM,				/* From local receiver, normal mode*/
    RCVR_DIAG,				/* Local receiver, diagnostic mode*/
    ALPHA_OMEGA				/* Alpha-Omega receiver		*/
} InputMode;				/********************************/



/********************************/
/*	Module Local Data	*/
/********************************/

MLocal Int32		numTargetPtts = 0;
MLocal Int32		afterTime = 0;
MLocal Int32		beforeTime = 0;
MLocal InputMode	inputMode = SVC_ARGOS;
MLocal Int32		targetPtts[MAX_TARGETS];
#ifdef WIN32
Extern Int getopt( Int argc, char **argv, char *fmt );
#endif
Extern char	*optarg;		/* Option argument from getopt()    */
Extern Int	optind;			/* Option index from getopt()	    */


/********************************/
/*	Forward Declarations	*/
/********************************/

MBool	processCommandLine ( Int32 argc, char **argv );
MBool	isTargetPtt( Int32 pttNum );
MBool	isInDateRange( Int32 year, Int32 month, Int32 day );


Global Nat16	jdays[12] =		/* Days since Jan. 1 in non-leap yr */
{ 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 };


/************************************************************************/
/* Function    : julday							*/
/* Purpose     : Calculate Julian Day given year, month, day		*/
/* Inputs      : year (00 - 99), month (1 - 12), day (1 - 31)		*/
/* Outputs     : Julian day						*/
/* Comment     : Returns Julian day in range 1:366			*/
/************************************************************************/
	Int32
julday( Int32 yr, Int32 mo, Int32 day )
{
    Int32		leap;

    leap = (((yr % 4) == 0) && (mo > 1)) ? 1 : 0;
    return( jdays[mo-1] + day + leap );

} /* julday() */


/************************************************************************/
/* Function    : main							*/
/* Purpose     : Main routine						*/
/* Inputs      : argc, argv						*/
/* Outputs     : Integer, 0 for success, 1 for failure			*/
/************************************************************************/
	Int32
main( Int32 argc, char **argv )
{
    Reg Int32		currentPttNum = 0;
    Int32		pttNum, unused[12];
    Int32		year, month, day;
    char		c;
    char		buffer[BUFSIZE];

    if ( !processCommandLine(argc, argv) )
	return( 1 );

    if ( numTargetPtts <= 0 )
    {
	targetPtts[0] = DEFAULT_PTT_NUM;
	numTargetPtts = 1;
    }

    while ( gets(buffer) != NULL )
    {

	if ( strstr(buffer, "From") != NULL )
	    currentPttNum = year = month = day = 0;

	switch( inputMode )
	{
	  case RCVR_DIAG:
		  if ( sscanf(buffer, " %d-%d-%d %d:%d:%d ID %d",
			&month, &day, &year, &unused[3],
			&unused[4], &unused[5], &pttNum) == 7 )
		currentPttNum = pttNum;
	    break;

 	  case RCVR_NORM:
		  if ( sscanf(buffer, " %d-%d-%d %d:%d:%d ID %d",
			&month, &day, &year, &unused[0],
			&unused[1], &unused[2], &pttNum) == 7 )
		if ( year > 1900 )
		    year -= 1900;
		currentPttNum = pttNum;
	    break;

	  case SVC_ARGOS:
	    if ( sscanf(buffer, " %d-%d-%d %d:%d:%d %* %* %* %* %*",
			&unused[0], &unused[1], &unused[2], &unused[3],
			&unused[4], &unused[5]) >= 6 )
	    {
		year = unused[0];
		if ( year > 1900 )
		    year -= 1900;
		month = unused[1];
		day = unused[2];
	    }
			
	    else if (sscanf(buffer, " %d %d %d %d%*[	 ] %c", &unused[0],
			    &pttNum, &unused[1], &unused[2], &c) == 5 )
			currentPttNum = pttNum;
	    break;

	  case ALPHA_OMEGA:
	    if ( sscanf(buffer, " %d/%d/%d %d:%d:%d %d",
			&month, &day, &year, &unused[3],
			&unused[4], &unused[5], &pttNum) == 7 )
		currentPttNum = pttNum;
	    break;
	}/* end switch */

	if ( isTargetPtt(currentPttNum) && isInDateRange(year, month, day) )
	    puts( buffer );

    }/* end while */

	return(0);
} /* main() */


/************************************************************************/
/* Function : isTargetPtt						*/
/* Purpose  : Determine whether passed PTT number is one of the target PTTs*/
/* Inputs   : PTT number						*/
/* Outputs  : TRUE if target PTT, else FALSE				*/
/************************************************************************/
	MBool
isTargetPtt( Int32 pttNum )
{
    Reg Int32	i;

    for ( i = 0; i < numTargetPtts; i++ )
	if ( pttNum == targetPtts[i] )
	    return( TRUE );

    return( FALSE );

} /* isTargetPtt() */


/************************************************************************/
/* Function : isInDateRange						*/
/* Purpose  : Determine whether date is in range specified by user	*/
/* Inputs   : year, month, day of ARGOS message				*/
/* Outputs  : TRUE if in date range, else FALSE				*/
/************************************************************************/
	MBool
isInDateRange( Int32 year, Int32 month, Int32 day )
{
    Int32	jday;

    jday = (1000 * year) + julday(year, month, day);

    if ( beforeTime && (jday > beforeTime) )
	return( FALSE );

    if ( afterTime && (jday < afterTime) )
	    return( FALSE );

    return( TRUE );

} /* isInDateRange() */


/************************************************************************/
/* Function : use_msg							*/
/* Purpose  : Print Usage Message					*/
/* Inputs   : Name of program						*/
/* Outputs  : None							*/
/************************************************************************/
	Void
use_msg( char *s )
{
    fprintf(stderr, "Usage: %s [-n ptt_number] [-a julian day] [-b julian day] [-d] <infile >outfile\n",s);
    fprintf(stderr, "-n indicates a PTT number to include in filter\n");
    fprintf(stderr, "   multiple -n parameters may be included\n");
    fprintf(stderr, "   if none specified, default PTT number is %d\n",
	    DEFAULT_PTT_NUM);
    fprintf(stderr, "-a include messages that came on or after julian day\n");
    fprintf(stderr, "-b include messages that came before or on julian day\n");
    fprintf(stderr, "   julian day specified as yyddd or ddd; e.g. 98110 or 110\n");
    fprintf(stderr, "   98110 means the 110th day of 1998.  110 means the 110th day of current year\n");
    fprintf(stderr, "-d means input is from Telonics receiver in diagnostic mode\n");
    fprintf(stderr, "   (default input mode is as email from Service Argos)\n");
    fprintf(stderr, "-N means input is from Telonics receiver in normal mode\n");
    fprintf(stderr, "-o means input is from Alpha-Omega receiver\n");
    fprintf(stderr, "Note that I/O is from stdin and stdout\n");

} /* use_msg() */


/************************************************************************/
/* Function : processCommandLine					*/
/* Purpose  : Read the arguments from the command line			*/
/* Inputs   : argc, argv from main() routine				*/
/* Outputs  : TRUE if arguments OK, else FALSE				*/
/************************************************************************/
	MBool
processCommandLine ( Int32 argc, char **argv )
{
    Int32	c, i;
    time_t	curtime;
    struct tm	*tp;

    while ( (c = getopt(argc, argv, "a:b:dn:No")) != EOF )
	switch( c )
	{
	  case 'a':
	    afterTime = atoi(optarg);
	    if ( afterTime < 1000 )
	    {
		time( &curtime );
		tp = gmtime( &curtime );
		afterTime += 1000 * tp->tm_year;
	    }
	    break;

	  case 'b':
	    beforeTime = atoi(optarg);
	    if ( beforeTime < 1000 )
	    {
		time( &curtime );
		tp = gmtime( &curtime );
		beforeTime += 1000 * tp->tm_year;
	    }
	    break;

	  case 'd':
	    inputMode = RCVR_DIAG;
	    break;

	  case 'n':
	    if ( ((i = atoi(optarg)) != 0) && (numTargetPtts < MAX_TARGETS) )
		targetPtts[numTargetPtts++] = i;
	    break;

	  case 'N':
	    inputMode = RCVR_NORM;
	    break;

	  case 'o':
	    inputMode = ALPHA_OMEGA;
	    break;

	  default:
	    use_msg( argv[0] );
	    return( FALSE );
	}

    return( TRUE );

} /* processCommandLine() */
