/****************************************************************************/
/* Copyright 1999 MBARI                                                     */
/****************************************************************************/
/* $Header: relays.c,v 1.1 2001/06/19 13:04:36 oasisa Exp $		*/
/* Summary  : Program to calculate relay select bits on OASIS		    */
/* Filename : relays.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 1.1 $		*/
/* Created  : 03/30/99 							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 30mar99 rah - created						    */
/* $Log:	relays.c,v $
 * Revision 1.1  2001/06/19  13:04:36  13:04:36  oasisa (Oasis users)
 * Initial revision
 * 
*/
/****************************************************************************/

#include <stdio.h>			/* Standard I/O			    */
#include <mbariTypes.h>			/* MBARI type definitions	    */
#include <mbariConst.h>			/* MBARI constants		    */
#include <signal.h>
#include <ctype.h>

#define NUM_PORTS	3		/* Number physical serial ports	    */
#define NUM_RELAYS	16		/* Number serial relays for mux	    */

typedef enum
{
    PORTA = 0,
    PORTB,
    PORTC,
    UNUSED
} RelayPort;


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal Nat16	relaySelect[NUM_PORTS][2]; /* Select bits for OASIS	    */
MLocal RelayPort relayPorts[NUM_RELAYS];   /* Which port each relay controls*/
MLocal MBool	done = FALSE;


/************************************************************************/
/* Function    : quit							*/
/* Purpose     : SIGTERM handler					*/
/* Inputs      : Signal number received					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
quit( Int sig )
{
    done = TRUE;

} /* quit() */


/************************************************************************/
/* Function    : main							*/
/* Purpose     : Main routine						*/
/* Inputs      : argc, argv						*/
/* Outputs     : none							*/
/************************************************************************/
	Void
main( Int argc, char **argv )
{
    Reg Int32	i;
    Nat32	relay;
    Nat16	sel0, sel1;
    MBool	ok;
    char	portChr;
    char	inbuff[256];

    printf("OASIS serial relay setup program\n\n");

    signal(SIGTERM, quit);		/* Catch terminate signals	*/
    signal(SIGINT, quit);
#ifndef WIN32
    signal(SIGQUIT, quit);
#endif
    for ( i = 0; i < NUM_RELAYS; i++ )
	relayPorts[i] = UNUSED;

    for ( i = 0; i < NUM_PORTS; i++ )
    {
	relaySelect[i][0] = 0;
	relaySelect[i][1] = 0;
    }

    while ( !done )
    {
	ok = FALSE;
	fflush( stdin );
	printf("Relay number [0 - 15 or <return> to end]  ");
	if ( (fgets(inbuff, 255, stdin) == NULL) ||
	     (sscanf(inbuff, "%d", &relay) <=  0) )
	    done = TRUE;
	else if ( relay < NUM_RELAYS )
	{
	    printf("Port number (A, B, C)  ");
	    if ( fgets(inbuff, 255, stdin) == NULL )
		done = TRUE;
	    else if ( sscanf(inbuff, "%1s", &portChr) >= 1 )	
	    {
		i = toupper(portChr) - 'A';
		if ( i <= NUM_PORTS )
		{
		    ok = TRUE;
		    relayPorts[relay] = i;
		    if ( relay < 8 )
			relaySelect[i][0] |= (2 << (2*relay));
		    else
			relaySelect[i][1] |= (2 << (2*(relay-8)));
		}
	    }
	}
	if ( !ok && !done )
	    printf("I didn't understand that.  Try again.\n");
    }

    printf("\n");
    printf("Relay\tPort\tSelect0\tSelect1\n\n");

    for ( i = 0; i < NUM_RELAYS; i++ )
    {
	printf("%2d", i);/* Relay */
	if ( relayPorts[i] == UNUSED )
	    printf("\tUnused\n");
	else
	{
	    sel0 = relaySelect[relayPorts[i]][0];
	    sel1 = relaySelect[relayPorts[i]][1];

	    if ( i < 8 )
		sel0 ^= (3 << (2*i));
	    else
		sel1 ^= (3 << (2*(i-8)));

	    printf("\t%c\t0x%04x\t0x%04x\n",relayPorts[i] + 'A', sel0, sel1);
	}
    }

} /* main() */
