/****************************************************************************/
/* Copyright 1994 MBARI                                                     */
/****************************************************************************/
/* $Header: range.c,v 3.2 2002/03/01 15:38:50 oasisa Exp $			    */
/* Summary  : Routines to handle Log Record Ranges			    */
/* Filename : range.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 3.2 $							    */
/* Created  : 03/30/94							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 30mar94 rah - created
 * 11May00 Gerry Hatcher changed file paths for WINN
 */
/* $Log
*/
/****************************************************************************/

#include <stdio.h>			/* Standard I/O			    */
#include <mbariTypes.h>			/* MBARI type definitions	    */
#include <mbariConst.h>			/* MBARI constants		    */
#include <decode.h>			/* OASIS controller definitions	    */
#include <string.h>			/* for strcmp(), memmove	    */

#ifdef WIN32
#include <malloc.h>
#include <memory.h>
#endif

#define LBUFSIZE	512		/* Size of line buffer		    */


/********************************/
/*	Global Data		*/
/********************************/
#ifdef WIN32
Global char	*rcdfile = "/oasisnt/oasis/raw/oasis.rcd";
#else
Global char	*rcdfile = "/oasis/raw/oasis.rcd";
#endif


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal char	lbuf[LBUFSIZE];		/* Line buffer			    */


/************************************************************************/
/* Function    : rcdInit						*/
/* Purpose     : Initialize LogStruct					*/
/* Inputs      : LogStruct ptr, number of LogStructs			*/
/* Outputs     : None							*/
/************************************************************************/
	Void
rcdInit( LogStruct *lp )
{
    memset( (void *)lp, 0, sizeof(LogStruct) );

} /*rcdInit() */


/************************************************************************/
/* Function    : isCanRcd						*/
/* Purpose     : Check string to see if it's a record for a given can	*/
/* Inputs      : String ptr, Can name					*/
/* Outputs     : Log number (0-n) if can string matches, else ERROR	*/
/************************************************************************/
	Int32
isCanRcd( char *s, char *can )
{
    Reg Nat32	len;
    Nat32	lognum;
    
    len = strlen(can);
    if ( strncmp(s, can, len) != 0 )
	return( ERROR );

    if ( sscanf(s + len, ".%d", &lognum) == 1 )
	return( lognum );
    else
	return( ERROR );

} /* isCanRcd() */


/************************************************************************/
/* Function    : updateLogPtrs						*/
/* Purpose     : Update pointers in LogStruct to allow insertion of lognum*/
/* Inputs      : LogStruct ptr, Log number you want to allow space for	*/
/* Outputs     : None							*/
/************************************************************************/
	Void
updateLogPtrs( LogStruct *lp, Int32 lognum )
{
    Reg Int32	shftCnt;

    shftCnt = lognum - lp->firstLog - NLOGS + 1;

    if ( shftCnt >= NLOGS )
	lp->firstLog = lognum;

#ifdef WIN32
    else if ( shftCnt > 0 )
    {
	memcpy( (char *)&lp->logs[shftCnt], (char *)&lp->logs,
	       (NLOGS - shftCnt) * sizeof(OneLog) );

	lp->firstLog = lognum - NLOGS + 1;
	memset( (char *)&lp->logs[NLOGS-1], 0, sizeof(OneLog) );
    }
    
    if ( lognum >= lp->nLogs )
	lp->nLogs = lognum + 1;
#else
    else if ( shftCnt > 0 )
    {
	bcopy( (char *)&lp->logs[shftCnt], (char *)&lp->logs,
	       (NLOGS - shftCnt) * sizeof(OneLog) );

	lp->firstLog = lognum - NLOGS + 1;
	bzero( (char *)&lp->logs[NLOGS-1], sizeof(OneLog) );
    }
    
    if ( lognum >= lp->nLogs )
	lp->nLogs = lognum + 1;

#endif
} /* updateLogPtrs() */


/************************************************************************/
/* Function    : readRcd						*/
/* Purpose     : Read Record file, fill in array of LogStructs		*/
/* Inputs      : Can name, Ptr to array of LogStructs, num structs in array*/
/* Outputs     : TRUE if read file OK					*/
/************************************************************************/
	MBool
readRcd( char *can, LogStruct *lp )
{
    Reg FILE	*fp;
    Reg char	*p;
    Reg Nat32	range, i;
    Reg Int32	lognum, lognumOffset;
    Nat32	min, max;
    
    if ( (fp = fopen(rcdfile, "r")) == (FILE *)NULL )
    {
	fprintf(stderr, "Cannot open record file %s\n", rcdfile);
	return( FALSE );
    }    

    while( fgets(lbuf, sizeof(lbuf), fp) != NULL )
    {
	if ( ((p = strtok(lbuf, ":")) == NULL) ||
	     ((lognum = isCanRcd(p, can)) == ERROR) )
	    continue;

	updateLogPtrs( lp, lognum );

	lognumOffset = lognum - lp->firstLog;

	for (range = 0; ((p = strtok(NULL, ",")) != NULL) && (range < NRANGES);)
	{
	    i = sscanf( p, "%d - %d", &min, &max );
	    if ( i == 1 )
		max = min;
	    if ( i >= 1 )
	    {
		lp->logs[lognumOffset].range[range].min = min;
		lp->logs[lognumOffset].range[range].max = max;
		range++;
	    }
	}
	lp->logs[lognumOffset].nRanges = range;
    }

#ifdef WIN32
	fclose(fp);
#else
    close( fp );
#endif
    return( TRUE );

} /* readRcd() */


/***************************************************************************/
/* Function    : writeRcd						   */
/* Purpose     : Write Record file, replace can strings with LogStruct info*/
/* Inputs      : Can name, Ptr to array of LogStructs, num structs in array*/
/* Outputs     : None							   */
/***************************************************************************/
	Void
writeRcd( char *can, LogStruct *lp )
{
    Reg FILE		*fp, *nfp;
    Reg char		*p, *name;
    Reg Nat32		i;
    Reg Int32		lognum;
    Reg OneLog		*olp;

    if ( (fp = fopen(rcdfile, "r")) == (FILE *)NULL )
	fprintf(stderr, "Cannot open record file %s\n", rcdfile);

    strcpy( lbuf, rcdfile );
    if ( (p = strrchr(lbuf, '/')) != NULL )
	*p = '\0';
    else
	strcpy( lbuf, "." );

    if ( (name = tempnam(lbuf, NULL)) == NULL )
    {	
	fprintf(stderr, "Cannot get temp file name\n");
	return;
    }

    if ( (nfp = fopen(name, "w")) == (FILE *)NULL )
    {
	fprintf(stderr, "Cannot open temp file %s\n", name);
	return;
    }    
	
	/* Copy all rcd file that isn't	*/
	/*  covered by current can	*/
    if ( fp != (FILE *)NULL ){		
		while( fgets(lbuf, sizeof(lbuf), fp) != NULL )
			if ( (lognum = isCanRcd(lbuf, can)) == ERROR )
				fwrite( lbuf, 1, strlen(lbuf), nfp );
		fclose(fp); /*close file*/
	}

				/* Now print current can records to rcd file*/
    for ( lognum = lp->firstLog, olp = lp->logs;
	  lognum < lp->nLogs; lognum++, olp++ )
	if ( olp->nRanges > 0 )
	{
	    fprintf( nfp, "%s.%d: ", can, lognum );
	    for( i = 0; i < olp->nRanges; i++ )
	    {
		if ( i > 0 )
		    fprintf( nfp, "," );
		fprintf( nfp, "%d", olp->range[i].min );

		if ( olp->range[i].min < olp->range[i].max )
		    fprintf( nfp, "-%d", olp->range[i].max );
	    }
	    fprintf( nfp, "\n" );
	}

    if (fclose(nfp) == ERROR )
		fprintf( stderr, "Cannot close record file %s\n", name );
    else
    {


	if( remove( rcdfile ) == -1 )
      perror( "Could not delete rcdfile" );

	if( rename( name, rcdfile ) == -1 )
      perror( "Could not rename rcdfile" );
    }


    free( name );

} /* writeRcd() */


/************************************************************************/
/* Function    : gotRcd							*/
/* Purpose     : Note received record number in LogStruct		*/
/* Inputs      : LogStruct, record number				*/
/* Outputs     : None							*/
/************************************************************************/
	Void
gotRcd( LogStruct *lp, Int32 lognum, Nat32 rcd )
{
    Reg Nat32	i;
    Reg OneLog	*olp;

    if ( lognum < lp->firstLog )
	return;

    updateLogPtrs( lp, lognum );		/* Allow space for this log*/
    olp = &lp->logs[lognum - lp->firstLog];	/* Point to appropriate log*/

    for ( i = 0; i < olp->nRanges; i++ )
    {
	if ( (rcd >= olp->range[i].min) && (rcd <= olp->range[i].max) )
	    return;				/* If already logged, rtn*/

	if ( rcd == (olp->range[i].min - 1) )
	{					/* If at low end of range*/
	    olp->range[i].min = rcd;		/* just note and return  */
	    return;
	}

	if ( rcd < olp->range[i].min )
	{					/* If between existing ranges*/
	    if ( olp->nRanges >= NRANGES )	/* create new range	  */
		return;
	    memmove( (void *)&olp->range[i+1], (void *)&olp->range[i],
		     (olp->nRanges - i) * sizeof(RcdRange) );
	    olp->nRanges++;
	    olp->range[i].min = olp->range[i].max = rcd;
	    return;
	}

	if ( rcd == (olp->range[i].max + 1) )
	{					/* If at hi end of range   */
	    olp->range[i].max = rcd;		/* note and return	   */
	    if (((i + 1) < olp->nRanges) && (rcd >= (olp->range[i+1].min - 1)))
	    {					/* Collapse two ranges	   */
		olp->range[i].max = olp->range[i+1].max;
		olp->nRanges--;
		memmove( (void *)&olp->range[i+1], (void *)&olp->range[i+2],
			 (olp->nRanges - i) * sizeof(RcdRange) );
		memset( (void *)&olp->range[olp->nRanges], 0, 
		        sizeof(RcdRange) );
	    }
	    return;
	}
    }
				/* Rcd num is bigger than existing ranges */
    if ( olp->nRanges < NRANGES )
    {
	olp->range[olp->nRanges].min = olp->range[olp->nRanges].max = rcd;
	olp->nRanges++;
    }

} /* gotRcd() */


/************************************************************************/
/* Function    : getMissingRcd						*/
/* Purpose     : Find next set of missing records			*/
/* Inputs      : LogStruct, log and record number to search from,	*/
/*		 place to put missing record range			*/
/* Outputs     : TRUE if more missing records, FALSE if not		*/
/* Side Effect : If return == TRUE, *rcd is filled in with range of	*/
/*		 missing records					*/
/************************************************************************/
	MBool
getMissingRcd( LogStruct *lp, Int32 lognum, Nat32 last, RcdRange *rcd )
{
    Reg Nat32	i;
    Reg OneLog	*olp;

    if ( (lognum < lp->firstLog) || (lognum >= lp->firstLog + NLOGS) )
	return( FALSE );

    olp = &lp->logs[lognum - lp->firstLog];

    for ( i = 0; i < olp->nRanges; i++ )
    {
	if ( last < olp->range[i].min )
	{
	    rcd->min = last;
	    rcd->max = olp->range[i].min - 1;
	    return( TRUE );
	}

	if ( (i < (olp->nRanges - 1)) && (last <= olp->range[i].max) )
	{
	    rcd->min = olp->range[i].max + 1;
	    rcd->max = olp->range[i+1].min - 1;
	    return( TRUE );
	}
    }

    return( FALSE );

} /* getMissingRcd() */
