/************************************************************************/
/* Copyright 1996 MBARI							*/
/************************************************************************/
/* $Header: argos.h,v 3.4 96/06/18 15:24:03 bobh Exp $			*/
/* Summary  : Definitions for ARGOS messages				*/
/* Filename : argos.h							*/
/* Author   : Robert Herlien (rah)					*/
/* Project  : OASIS Mooring						*/
/* $Revision: 3.4 $							*/
/* Created  : 06/28/96							*/
/************************************************************************/
/* Modification History:						*/
/* 28jun96 rah, created							*/
/* $Log$
*/
/************************************************************************/

#ifndef INCargosh
#define INCargosh		1

#define	ARGOS_LEN		32	/* Length of ARGOS data		    */
#define	ARGOS_WORDS	(ARGOS_LEN/2)	/* Length of data in words	    */

#define	CHEM_SAMPLES		 8	/* 8 samples/day for CO2, NO3, etc  */
#define NO3_SIZE		12	/* 12 bits * 8 samples = 12 bytes   */
#define ASPEC_CHANS		 7	/* Number of channels of spectro data*/
#define AC9_1_CHANS		15	/* Num chans in first AC-9 buff	    */
#define AC9_2_CHANS		 3	/* Num chans in second AC-9 buff    */
#define SPEC_0M_ED		 0	/* Address tag for surface Spec Ed  */
#define SPEC_0M_LU		 1	/* Address tag for surface Spec Lu  */
#define SPEC_20M_ED		 2	/* Address tag for 20m Spec Ed	    */
#define SPEC_20M_LU		 3	/* Address tag for 20m Spec Lu	    */
#define SPEC_NOON_LU_CHAN	 0	/* Channel for noon surface Spec Lu */
#define SPEC_AM_LU_CHAN		 8	/* Channel for 1030am surface Spec Lu*/


/****************************************/
/* ARGOS Message Buffer Identifiers	*/
/****************************************/

#define CHEM		0		/* pCO2 and NO3 buffer		    */
#define SPEC_0M_NOON	1		/* Surface Spectro, noon sample	    */
#define SPEC_0M_AM	2		/* Surface Spectro, 1030 AM sample  */
#if (ARGOS_REV == 1)
#define SPEC_20M	3		/* 20 meter Spectro, noon sample    */
#define AC9_1		4		/* First part of AC-9 data	    */
#define AC9_2		5		/* Second part of AC-9 data	    */
#define MISC		6		/* Misc data buffer		    */
#else
#define MISC		3		/* Misc data buffer		    */
#define SPEC_20M	4		/* 20 meter Spectro, noon sample    */
#define AC9_1		5		/* First part of AC-9 data	    */
#define AC9_2		6		/* Second part of AC-9 data	    */
#endif


/****************************************/
/* Definitions for CRC-12 checksum	*/
/****************************************/

#define CRC_MSB		0x1000		/* MSB of CRC checksum word	    */
#define CRC12		0x180f		/* CRC-12 check polynomial	    */


/****************************************/
/* ARGOS Message Buffer Types		*/
/****************************************/

typedef struct				/************************************/
{					/* Chemical Buffer - pCO2 & NO3     */
    Byte	no3[NO3_SIZE];		/* Surface NO3 samples		    */
    Byte	temp[CHEM_SAMPLES];	/* Surface NO3 temperatures	    */
} No3Msg;				/************************************/

typedef struct				/************************************/
{					/* Chemical Buffer - pCO2 & NO3     */
    Word	ch_type;		/* Message type and checksum	    */
    Byte	ch_pco2[CHEM_SAMPLES];	/* 8 LSBs for each pCO2 sample	    */
    Word	ch_pco2_msbs;		/* 2 MSBs for each pCO2 sample	    */
    No3Msg	ch_no3;			/* Surface NO3 and temperatures	    */
} ChemMsg;				/************************************/

typedef struct				/************************************/
{					/* Spectral Data Buffer		    */
    Word	sp_type;		/* Message type and checksum	    */
    Nat16	sp_ed[ASPEC_CHANS];	/* Downwelling (incident) channels  */
    Nat16	sp_lu[ASPEC_CHANS];	/* Upwelling (reflected) channels   */
    Nat16	sp_fluor;		/* Fluorometer (or misc)	    */
} SpecMsg;				/************************************/

typedef struct				/************************************/
{					/* AC-9 Message Buffer 1	    */
    Word	ac_type;		/* Message type and checksum	    */
    Nat16	ac_data[AC9_1_CHANS];	/* First set of AC-9 data	    */
} Ac9Msg;				/************************************/

typedef struct				/************************************/
{					/* AC-9 Message Buffer 2	    */
    Word	ac_type;		/* Message type and checksum	    */
    Nat16	ac_data[AC9_2_CHANS];	/* Second set of AC-9 data	    */
    Nat16	ac_temp;		/* AC-9 Temperature		    */
    Nat16	ac_chl[CHEM_SAMPLES];	/* AC-9 Chlorophyll from 8 chem times*/
    Nat16	ac_pump;		/* Pump flow			    */
    Nat16	ac_fill[2];		/* No data			    */
} Ac92Msg;				/************************************/

typedef struct				/************************************/
{					/* Miscellaneous Message Buffer     */
    Word	ms_type;		/* Message type and checksum	    */
    Nat16	ms_pco2;		/* pCO2 calibration and sample time */
    Nat32	ms_oasis;		/* OASIS batt, temp, ARGOS batt	    */
    Nat16	ms_mcp10;		/* 10m MCP from 20m PRR		    */
    Nat16	ms_mcp30;		/* 30m MCP from 20m PRR		    */
    No3Msg	ms_no3;			/* 20m NO3 and temperatures	    */
} MiscMsg;				/************************************/

typedef struct				/************************************/
{					/* ARGOS message as array of bytes  */
    Byte	by_data[ARGOS_LEN];	/* Data				    */
} ByteMsg;				/************************************/

typedef struct				/************************************/
{					/* ARGOS message as array of words  */
    Nat16	wd_data[ARGOS_WORDS];	/* Data				    */
} WordMsg;				/************************************/

typedef union				/************************************/
{					/* Union of all ARGOS Message types */
    ChemMsg	chem;			/* Chemistry buffer		    */
    SpecMsg	spec0m;			/* Noon surface spectral buffer	    */
    SpecMsg	spec0m_1030;		/* 1030 AM surface spectral buffer  */
    SpecMsg	spec20m;		/* Noon 20 meter spectral buffer    */
    Ac9Msg	ac9;			/* First AC-9 buffer		    */
    Ac92Msg	ac92;			/* Second AC-9 buffer		    */
    MiscMsg	misc;			/* Miscellaneous buffer		    */
    ByteMsg	rawb;			/* Raw byte format		    */
    WordMsg	raww;			/* Raw word format		    */
} ArgosUnion;				/************************************/

typedef struct				/************************************/
{					/* Struct to hold incoming msgs	    */
    time_t	first_msg;		/* First time msg was sent	    */
    time_t	last_msg;		/* Last time msg was sent	    */
    Nat32	msg_cnt;		/* Number of times msg was sent	    */
    ArgosUnion	msg;			/* Message			    */
} ArgosInMsg;				/************************************/

#endif	/* INCargosh */
