/****************************************************************************/
/* Copyright 1992 MBARI                                                     */
/****************************************************************************/
/* $Header: ctd.c,v 2.8 98/03/17 11:11:40 bobh Exp $			    */
/* Summary  : CTD decode routines for decode.c, extract.c		    */
/* Filename : ctd.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 2.8 $							    */
/* Created  : 01/24/92							    */
/****************************************************************************/
/* Modification History:						    */
/* 24jan91 rah - created						    */
/* $Log:	ctd.c,v $
 * Revision 2.8  98/03/17  11:11:40  11:11:40  bobh (Bob Herlien)
 * Archiving sources prior to porting to DOS/Windows
 * 
 * Revision 2.5  94/12/15  10:59:40  10:59:40  hebo (Bob Herlien)
 * Accumulated minor changes, mainly due to move to tsunami
 * 
 * Revision 2.2  92/09/24  18:34:44  18:34:44  hebo (Bob Herlien)
 * Changed trfl to trans and fluor, use upper 12 bits if offset even, lower
 *   if odd.  This due to trans, fluor backward on some CTDs
 * 
 * Revision 2.1  92/09/04  13:35:12  13:35:12  hebo (Bob Herlien)
 * Read pressure from calibration file if CTD has no pressure sensor.
 * 
 * Revision 2.0  92/08/31  15:34:16  15:34:16  hebo (Bob Herlien)
 * August 1992 Deployment.  Changed to allow multiple sensors of same type.
 * 
 * Revision 1.0  92/02/25  10:46:57  10:46:57  hebo (Bob Herlien)
 * Initial revision
 * 
 * Revision 1.1  92/02/25  10:45:31  10:45:31  hebo (Bob Herlien)
 * Initial revision
 * 
*/
/****************************************************************************/

#include <stdio.h>			/* Standard I/O			    */
#include <mbariTypes.h>			/* MBARI type definitions	    */
#include <mbariConst.h>			/* MBARI constants		    */
#include <decode.h>			/* OASIS controller definitions	    */
#include <time.h>			/* Time				    */
#include <ctype.h>			/* for toupper()		    */
#include <string.h>			/* for strcmp			    */
#include <math.h>


/********************************/
/*	External Function	*/
/********************************/

Extern Nat16	getMotword( Byte *p );


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal char	buff[256];		/* Scratch buffer		    */


/************************************************************************/
/* Function    : init_one_cal						*/
/* Purpose     : Initialize one CTD calibration value			*/
/* Inputs      : Ptr to Cal to init					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
init_one_cal( CCal *p )
{
    p->offset = 0;
    p->a = (double)0.0;
    p->b = (double)0.0;
    p->c = (double)0.0;
    p->d = (double)0.0;
    p->mf = (double)0.0;

} /* init_one_cal() */


/************************************************************************/
/* Function    : read_ctd_cal						*/
/* Purpose     : Read CTD calibration file				*/
/* Inputs      : Name of file, place to put calibrations		*/
/* Outputs     : OK or ERROR						*/
/************************************************************************/
	Status
read_ctd_cal( char *name, CTDCal *ccp )
{
    FILE	*fp;
    Int		ncals;

    ccp->cc_size = CTD_BYTES;
    init_one_cal(&ccp->cc_cond);
    init_one_cal(&ccp->cc_temp);
    init_one_cal(&ccp->cc_press);
    init_one_cal(&ccp->cc_trans);
    init_one_cal(&ccp->cc_fluor);
    init_one_cal(&ccp->cc_sample);

    if ( (fp = fopen(name, "rb")) == (FILE *)NULL )
	return( ERROR );

    ncals = 0;

    while ( fscanf(fp, " %s", buff) != EOF )
    {
	if ( strcmp(buff, "length") == 0 )
	{
	    if ( fscanf(fp, " %d", &ccp->cc_size) == 1 )
		ncals++;
	}
	if ( strcmp(buff, "cond") == 0 )
	{
	    if ( fscanf(fp, " %d %lg %lg %lg %lg %lg", &ccp->cc_cond.offset,
			&ccp->cc_cond.a, &ccp->cc_cond.b, &ccp->cc_cond.c, 
			&ccp->cc_cond.d, &ccp->cc_cond.mf) >= 5 )
		ncals++;
	}
	else if ( strcmp(buff, "temp") == 0 )
	{
	    if ( fscanf(fp, " %d %lg %lg %lg %lg %lg", &ccp->cc_temp.offset,
			&ccp->cc_temp.a, &ccp->cc_temp.b, &ccp->cc_temp.c,
			 &ccp->cc_temp.d, &ccp->cc_temp.mf) == 6 )
		ncals++;
	}
	else if ( strcmp(buff, "press") == 0 )
	{
	    if ( fscanf(fp, " %d %lg %lg %lg", &ccp->cc_press.offset,
			&ccp->cc_press.a, &ccp->cc_press.b, 
			&ccp->cc_press.c) >= 2 )
		ncals++;
	}
	else if ( strcmp(buff, "trans") == 0 )
	{
	    if ( fscanf(fp, " %d", &ccp->cc_trans.offset) == 1 )
		ncals++;
	}
	else if ( strcmp(buff, "fluor") == 0 )
	{
	    if ( fscanf(fp, " %d", &ccp->cc_fluor.offset) == 1 )
		ncals++;
	}
	else if ( strcmp(buff, "sample") == 0 )
	{
	    if ( fscanf(fp, " %d", &ccp->cc_sample.offset) == 1 )
		ncals++;
	}
    }

    fclose( fp );
    return( ncals >= 5 ? OK : ERROR );

} /* read_ctd_cal() */


/************************************************************************/
/* Function    : ctd_temperature					*/
/* Purpose     : Calculate CTD temperature				*/
/* Inputs      : Temp count, Calibration struct				*/
/* Outputs     : Temperature in degrees Celsius				*/
/************************************************************************/
	double
ctd_temperature( Nat16 t, CTDCal *ccp )
{
    double	freq;				/* val converted to freq    */
    double	x;				/* Intermediate value	    */

    freq = (double)t / 19.0 + 2100.0;		/* Convert to frequency     */
    x = log( ccp->cc_temp.mf / freq );		/* Calc ln(f0/f)	    */
    return( 1.0 / (ccp->cc_temp.a + (ccp->cc_temp.b * x)
		   + (ccp->cc_temp.c * x * x) + (ccp->cc_temp.d * x * x * x))
	   - 273.15 );

} /* ctd_temperature() */


/************************************************************************/
/* Function    : ctd_pressure						*/
/* Purpose     : Calculate CTD pressure					*/
/* Inputs      : Pressure count, Calibration struct			*/
/* Outputs     : Pressure in decibars					*/
/************************************************************************/
#define DECIBAR		0.6894759	/* PSI to decibar conversion const */

	double
ctd_pressure( Nat16 p, CTDCal *ccp )
{
    Nat32	p1;

    p1 = (p > 4095) ? p - 65536 : p;
    return( DECIBAR * (ccp->cc_press.a + (ccp->cc_press.b * (double)p1)
	   + (ccp->cc_press.c * (double)p1 * (double)p1) - 14.7) );

} /* ctd_pressure() */


/************************************************************************/
/* Function    : ctd_conduct						*/
/* Purpose     : Calculate CTD conductivity				*/
/* Inputs      : Conductivity count, Temp, Pressure (decibars), calibr	*/
/* Outputs     : Conductivity in Siemans/meter				*/
/************************************************************************/
#define CONP		9.57e-08	/* "Magic" conductivity constant    */

	double
ctd_conduct( Nat16 c, double t, double p, CTDCal *ccp )
{
    double	freq;				/* conductivity frequency   */
    double	x;				/* Intermediate value	    */

    freq = sqrt(2100.0 * (double)c + 6.25e06) / 1000.0;
    x = 0.1 * ((ccp->cc_cond.a * pow(freq,ccp->cc_cond.mf))
	       + (ccp->cc_cond.b * freq * freq) + ccp->cc_cond.c 
	       + (ccp->cc_cond.d * t));
    return( x / (1.0 - CONP * p) );

} /* ctd_conduct() */


/************************************************************************/
/* Function    : ctd_salinity						*/
/* Purpose     : Calculate CTD salinity					*/
/* Inputs      : Conductivity, temperature, pressure			*/
/* Outputs     : Salinity as a double precision real number		*/
/* Comments    : Lifted (almost) directly from Seabird's cnv.c file	*/
/************************************************************************/
#define A1 2.070e-5			/* salinity constants */
#define A2 -6.370e-10
#define A3 3.989e-15
#define B1 3.426e-2
#define B2 4.464e-4
#define B3 4.215e-1
#define B4 -3.107e-3
#define C0 6.766097e-1
#define C1 2.00564e-2
#define C2 1.104259e-4
#define C3 -6.9698e-7
#define C4 1.0031e-9

double a[6] = {	/* constants for salinity calculation */
	0.0080,
	-0.1692,
	25.3851,
	14.0941,
	-7.0261,
	2.7081
};

double b[6]={	/* constants for salinity calculation */
	0.0005,
	-0.0056,
	-0.0066,
	-0.0375,
	0.0636,
	-0.0144
};

	double
ctd_salinity( double c, double t, double p )
{
    double	rt,rp,temp,sum1,sum2,result;
    Int		i;

    if ( c <= 0.0 )
	result = 0.0;
    else
    {
	c *= 10.0;		/* convert Siemens/meter to mmhos/cm */
	c /= 42.914;
	rp = 1+(p*(A1+p*(A2+p*A3)))/(1+B1*t+B2*t*t+B3*c+B4*c*t);
	rt = c /(rp*(C0+(t*(C1+t*(C2+t*(C3+t*C4))))));
	sum1 = sum2 = 0.0;
	for(i=0;i<6;i++)
	{
	    temp = pow(rt,(double)i/2.0);
	    sum1 = sum1 + a[i] * temp;
	    sum2 = sum2 + b[i] * temp;
	}
	result = sum1+sum2*(t-15.0)/(1+0.0162*(t-15.0));
    }
    return( result );

} /* ctd_salinity() */


/************************************************************************/
/* Function    : ctd_transfluor						*/
/* Purpose     : Calculate transmissometer or fluorometer voltage	*/
/* Inputs      : Ptr to data buffer, ptr to calibration struct		*/
/* Outputs     : Voltage of transmissometer or fluorometer		*/
/* Comments    : Transmissometer and fluorometer get packed together	*/
/*		 into a 3 byte value (12 bits per voltage).  It looks	*/
/*		 OK coming out in hex from the CTD, because it's 3 ASCII*/
/*		 digits per value.  But OASIS decodes the ASCII into hex,*/
/*		 with the result that 3 bytes stores 2 values.		*/
/*		 The offset parameter of the calibration file doesn't	*/
/*		 really tell you whether to use the first 12 bits of the*/
/*		 word (i >> 4) or the last 12 bits (i & 0xfff).  But we	*/
/*		 fake it because we know that an even offset means the	*/
/*		 first 12 bits, and an odd offset means the last 12 bits*/
/************************************************************************/
	double
ctd_transfluor( Byte *dp, CCal *ccp )
{
    Reg Nat16	i;

    i = getMotword(dp + ccp->offset);

    if ( ccp->offset & 1 )
	i &= 0xfff;
    else
	i >>= 4;

    return( (double)i / 819.0 );

} /* ctd_transfluor() */


/************************************************************************/
/* Function    : decode_ctd						*/
/* Purpose     : Decode CTD information					*/
/* Inputs      : Pointer to CTD data, length, ptr to return struct	*/
/* Outputs     : OK or SIZE_ERR						*/
/************************************************************************/
	Status
decode_ctd( Byte *dp, Int len, CTDDecode *cdp, CTDCal *ccp )
{
    if ( len != ccp->cc_size )			/* Check # words, rtn if bad*/
	return( SIZE_ERR );

    if ( ccp->cc_sample.offset >= 0 )
	cdp->ctd_sample = getMotword(dp + ccp->cc_sample.offset);
    else
	cdp->ctd_sample = 0;

    cdp->ctd_temp = ctd_temperature(getMotword(dp + ccp->cc_temp.offset), ccp);
						/* Get temperature	    */
    if ( ccp->cc_press.offset >= 0 )
	cdp->ctd_press = 
	    ctd_pressure( getMotword(dp + ccp->cc_press.offset), ccp );
    else
	cdp->ctd_press = ccp->cc_press.a;

    cdp->ctd_cond = ctd_conduct( getMotword(dp + ccp->cc_cond.offset), 
				cdp->ctd_temp, cdp->ctd_press, ccp );
						/* Get conductivity	    */
    cdp->ctd_sal = ctd_salinity(cdp->ctd_cond, cdp->ctd_temp, cdp->ctd_press);
						/* Get salinity		    */

    cdp->ctd_fluor = ctd_transfluor( dp, &ccp->cc_fluor );
    cdp->ctd_trans = ctd_transfluor( dp, &ccp->cc_trans );

    return( OK );

} /* decode_ctd() */
