/****************************************************************************/
/* Copyright 1995 MBARI                                                     */
/****************************************************************************/
/* $Header: no3.c,v 3.1 2001/06/19 13:17:25 oasisa Exp $			    */
/* Summary  : NO3 decode routines for decode.c, extract.c		    */
/* Filename : no3.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 3.1 $							    */
/* Created  : 04/18/95							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 18apr95 rah - created						    */
/* $Log:	no3.c,v $
 * Revision 3.1  2001/06/19  13:17:25  13:17:25  oasisa (Oasis users)
 * Periodic Update 6/19/2001 (klh)
 * 
 * Revision 3.0  99/05/12  10:11:26  10:11:26  bobh (Bob Herlien)
 * Added tstring, misc changes
 * 
 * Revision 2.9  98/08/24  13:45:54  13:45:54  bobh (Bob Herlien)
 * Archiving sources after M2/M3 & Eqpac deployments of 1998
 * 
 * Revision 2.8  98/03/17  11:11:40  11:11:40  bobh (Bob Herlien)
 * Archiving sources prior to porting to DOS/Windows
 * 
 * Revision 2.6  96/05/30  15:07:56  15:07:56  bobh (Bob Herlien)
 * Update for version in use during 1995-6 deployment
 * 
*/
/****************************************************************************/

#include <stdio.h>			/* Standard I/O			    */
#include <mbariTypes.h>			/* MBARI type definitions	    */
#include <mbariConst.h>			/* MBARI constants		    */
#include <decode.h>			/* OASIS controller definitions	    */
#include <time.h>			/* Time				    */
#include <string.h>			/* for strcmp()			    */
#include <math.h>


/********************************/
/*	External Function	*/
/********************************/

Extern double	ctd_salinity( double c, double t, double p );


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal char	buff[256];		/* Scratch buffer		    */


/************************************************************************/
/* Function    : init_one_cal						*/
/* Purpose     : Initialize one CTD calibration value			*/
/* Inputs      : Ptr to Cal to init					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
init_no3_cal( No3Cal *p )
{
    p->nc_format = NO3_DRIFT1;
    p->nc_samplechar = 'S';
    p->nc_maxCounts = 3000;
    p->nc_absMult = (Int32)(20.0/0.345);
    p->nc_Co = (double)(-0.1);
    p->nc_Cf = (double)72.;
    p->nc_To = (double)(-2.2);
    p->nc_Tf = (double)34.;
    p->nc_pressure = (double)(1.0);

} /* init_no3_cal() */


/************************************************************************/
/* Function    : read_no3_cal						*/
/* Purpose     : Read NO3 calibration file				*/
/* Inputs      : Name of file, place to put calibrations		*/
/* Outputs     : OK or ERROR						*/
/************************************************************************/
	Status
read_no3_cal( char *name, No3Cal *no3cp )
{
    FILE	*fp;

    init_no3_cal( no3cp );

    if ( (fp = fopen(name, "rb")) == (FILE *)NULL )
	return( ERROR );

    while ( fscanf(fp, " %s", buff) != EOF )
    {
	if ( buff[0] == '#' )
	    ;					/* Skip comments	*/

	else if ( strcasecmp(buff, "samplechar") == 0 )
	    fscanf( fp, " %c", &no3cp->nc_samplechar );

	else if ( strcasecmp(buff, "maxCounts") == 0 )
	    fscanf( fp, " %d", &no3cp->nc_maxCounts );

	else if ( strcasecmp(buff, "absMult") == 0 )
	    fscanf( fp, " %d", &no3cp->nc_absMult );

	else if ( strcasecmp(buff, "pressure") == 0 )
	    fscanf( fp, " %lg", &no3cp->nc_pressure );

	else if ( strcasecmp(buff, "Co") == 0 )
	    fscanf( fp, " %lg", &no3cp->nc_Co );

	else if ( strcasecmp(buff, "Cf") == 0 )
	    fscanf( fp, " %lg", &no3cp->nc_Cf );

	else if ( strcasecmp(buff, "To") == 0 )
	    fscanf( fp, " %lg", &no3cp->nc_To );

	else if ( strcasecmp(buff, "Tf") == 0 )
	    fscanf( fp, " %lg", &no3cp->nc_Tf );

	else if ( strcasecmp(buff, "format") == 0 )
	{
	    fscanf( fp, " %s", buff );

	    if ( strcasecmp(buff, "Drift1") == 0 )
		no3cp->nc_format = NO3_DRIFT1;
	    else if ( strcasecmp(buff, "M1") == 0 )
		no3cp->nc_format = NO3_M1;
	    else if ( strcasecmp(buff, "M2") == 0 )
		no3cp->nc_format = NO3_M2;
	}

	fgets(buff, sizeof(buff), fp);		/* Skip remainder of line */
    }

    fclose( fp );
    return( OK );

} /* read_no3_cal() */


/************************************************************************/
/* Function    : decode_no3						*/
/* Purpose     : Decode NO3 information					*/
/* Inputs      : Pointer to CTD data, length, ptr to return struct	*/
/* Outputs     : OK or FMT_ERR						*/
/************************************************************************/
	Status
decode_no3( Byte *no3data, Int len, No3Decode *no3dp, No3Cal *no3cp )
{
    Reg Byte	*p;
    Int32	nValues;
    Int32	no3Count;
    Int32	Vf, Vo, Vt, Vc;
    Int32	unusedInt;
    double	unusedFloat;
    double	Ts, Tz, Cs, Cz;
    double	countRatio;
    char	indicator;
    char	unused[64];

    for ( p = no3data; *p && (p - no3data) < len; p++ )
	if ( *p == ',' )		/* Convert commas to spaces for scanf*/
	    *p = ' ';

    no3Count = 0;
    no3dp->no3_conc = 0.0;
    no3dp->no3_temp = 0.0;
    no3dp->no3_salinity = 0.0;
    no3dp->no3_conductivity = 0.0;

    switch( no3cp->nc_format )
    {
      case NO3_DRIFT1:
	nValues = sscanf((char *)no3data, 
			 " %c %d %s %12s %d %d %lg %d %d %d %d",
			 &indicator, &no3dp->no3_sample, unused,
			 &no3dp->no3_time, &no3Count, &unusedInt,
			 &unusedFloat, &Vf, &Vo, &Vt, &Vc);

	if ( indicator != no3cp->nc_samplechar )
	    return( FMT_ERR );

	if ( nValues >= 10 )
	{
	    Ts = (no3cp->nc_Tf - no3cp->nc_To) / (Vf - Vo);
	    Tz = no3cp->nc_Tf - (Ts * Vf);
	    no3dp->no3_temp = (Ts * Vt) + Tz;
	}

	if ( nValues >= 11 )
	{
	    Cs = (no3cp->nc_Cf - no3cp->nc_Co) / (Vf - Vo);
	    Cz = no3cp->nc_Cf - (Cs * Vf);
	    no3dp->no3_conductivity = ((Cs * Vc) + Cz) / 10.0;
					/* /10 converts to Siemens/meter  */
	    no3dp->no3_salinity =
		ctd_salinity( no3dp->no3_conductivity, no3dp->no3_temp,
			      no3cp->nc_pressure );
	}
	break;

      case NO3_M1:
	nValues = sscanf((char *)no3data, " %c %d %s %12s %d %d %lg %lg",
			 &indicator, &no3dp->no3_sample, unused, 
			 &no3dp->no3_time, &no3Count, &unusedInt, &unusedFloat,
			 &no3dp->no3_temp);
	break;

      case NO3_M2:
	nValues = sscanf((char *)no3data, " %d %s %12s %d %d %lg",
			 &no3dp->no3_sample, unused, &no3dp->no3_time, 
			 &no3Count, &unusedInt, &no3dp->no3_temp);
	break;

      default:
	return( FMT_ERR );
    }

    if ( no3Count > 0 )
    {
	countRatio = (double)no3Count / (double)(no3cp->nc_maxCounts);
	no3dp->no3_conc =  no3cp->nc_absMult * (-log10(countRatio));
    }
    else
	return( SIZE_ERR );

    return( OK );

} /* decode_no3() */
