/****************************************************************************/
/* Copyright 1992 MBARI                                                     */
/****************************************************************************/
/* $Header: m1.c,v 2.3 2001/06/19 13:16:22 oasisa Exp $			    */
/* Summary  : Program to print meteorological readings from OASIS M1 mooring*/
/* Filename : m1.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 2.3 $							    */
/* Created  : 09/24/92							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 10feb92 rah - created						    */
/* $Log:	m1.c,v $
 * Revision 2.3  2001/06/19  13:16:22  13:16:22  oasisa (Oasis users)
 * Periodic Update 6/19/2001 (klh)
 * 
 * Revision 1.1  94/01/17  11:09:55  11:09:55  hebo
 * Misc changes
 * 
 * Revision 1.0  92/09/24  10:29:53  10:29:53  hebo (Bob Herlien)
 * Initial revision
 * 
*/
/****************************************************************************/

#include <stdio.h>			/* Standard I/O			    */
#include <stdlib.h>			/* For exit()			    */
#include <mbariTypes.h>			/* MBARI type definitions	    */
#include <mbariConst.h>			/* MBARI constants		    */
#include <decode.h>			/* OASIS controller definitions	    */
#include <time.h>			/* Time				    */
#include <math.h>			/* For fmod()			    */
#include <ctype.h>

#define NAMESIZE	256		/* Space allocated for file names   */
#define MLINELEN	 95		/* Approx length of one METSYS record*/
#define CLINELEN	 60		/* Approx length of one CTD record  */
#define GLINELEN	 52		/* Approx length of one GPS record  */
#define LINEBUFLEN	512		/* Size of line buffer		    */
#define NRECS		150		/* Max # records displayed	    */
#define DFLT_RECS	20		/* Default # records displayed	    */
#define NFIELDS		15		/* # fields in METSYS record	    */
#define YEAR0		70		/* Unix Epoch year (1970)	    */
#define SECS_PER_DAY	86400		/* Seconds per day		    */
#define START_WINDOW	0.007		/* 10 minute window at start	    */

typedef struct				/************************************/
{					/* Struct for decoded Metsys data   */
    double	met_time;		/* OASIS Date/time		    */
    Int32	met_format;		/* Format code			    */
    Int32	met_jday;		/* Julian day from metsys	    */
    Int32	met_hour;		/* Hour of day reported by metsys   */
    Int32	met_min;		/* Minute of hour reported by metsys*/
    double	met_battery;		/* Metsys battery voltage	    */
    double	met_temp;		/* Air Temperature		    */
    double	met_humid;		/* Humidity			    */
    double	met_barometer;		/* Barometric pressure minus 800    */
    double	met_windspeed;		/* Wind speed in knots		    */
    double	met_winddir;		/* Wind direction from		    */
    double	met_windu;		/* Wind U vector		    */
    double	met_windv;		/* Wind V vector		    */
    double	met_buoy_head;		/* Buoy heading			    */
    double	met_windvane;		/* Wind vane offset from heading    */
} MetDecode;				/************************************/

typedef struct
{
    double	ctd_time;
    CTDDecode	ctdd;
} CTDDec;

typedef struct
{
    double	gps_time;
    double	gps_lat;
    double	gps_long;
    double	gps_std;
    int		gps_samples;
    int		gps_status;
} GPSDec;


/********************************/
/*	Global Data		*/
/********************************/

Global MetDecode metrecs[NRECS];	/* METSYS records		    */
Global CTDDec	ctdrecs[NRECS];		/* CTD records			    */
Global GPSDec	gpsrecs[NRECS];		/* GPS records			    */
Global char	metsys_file[NAMESIZE];	/* Name of MetSys data file	    */
Global char	ctd_file[NAMESIZE];	/* Name of CTD data file	    */
Global char	gps_file[NAMESIZE];	/* Name of GPS data file	    */
Global char	*metsys_dir = "/oasis/m1/metsys";
Global char	*ctd_dir = "/oasis/m1/ctd";
Global char	*gps_dir = "/oasis/m1/gps";
Global char	buffer[LINEBUFLEN];

Global char	*months[12] =		/* Month names			    */
{"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"};

Global Nat16	jdays[12] =		/* Days since Jan. 1 in non-leap yr */
{ 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 };


/************************************************************************/
/* Function    : julday							*/
/* Purpose     : Calculate Julian Day given year, month, day		*/
/* Inputs      : year (00 - 99), month (0 - 11), day (1 - 31)		*/
/* Outputs     : Julian day						*/
/* Comment     : Returns Julian day in range 1:366			*/
/************************************************************************/
	Int32
julday( Int32 yr, Int32 mo, Int32 day )
{
    Int32		leap;

    leap = (((yr % 4) == 0) && (mo > 1)) ? 1 : 0;
    return( jdays[mo] + day + leap );

} /* julday() */


/************************************************************************/
/* Function    : ep_days						*/
/* Purpose     : Calculate number days since Unix epoch			*/
/* Inputs      : years since 1970, Unix-type julian day (0:365)		*/
/* Outputs     : Days since 1/1/1970					*/
/* Comment     : The (year + 1)/4 term accounts for leap years, the	*/
/*		 first of which was 1972 & should be added starting '73	*/
/************************************************************************/
	Int32
ep_days( year, yday )
     int	year, yday;
{
    return( (365 * year) + ((year + 1)/4) + yday );

} /* ep_days() */


/************************************************************************/
/* function    : main							*/
/* purpose     : Main routine						*/
/* inputs      : argc, argv						*/
/* outputs     : none							*/
/************************************************************************/
	Void
main( int argc, char **argv )
{
    time_t	curtime, basetime, mtm, ctm, gtm;
    struct tm	*tp;
    Int		i, j, k, itime, mrecs, crecs, grecs, year, mo, yday, mday, rtn;
    FILE	*mfp, *cfp, *gfp;
    double	starttime;
    Int		lat, lng;

    time( &curtime );
    tp = gmtime( &curtime );
    year = tp->tm_year;
    itime = (1000 * year) + tp->tm_yday + 1;
    mrecs = DFLT_RECS;
    mfp = cfp = NULLF;

    for ( i = 1; i < argc; i++ )
    {
	if ( *argv[i] == '-' )
	    switch( argv[i][1] )
	    {
	      case 'd':
		if ( isdigit(argv[i][2]) )
		    itime = atoi( &argv[i][2] );
		else
		    itime = atoi( argv[++i] );
		if ( itime < 1000 )
		    itime += year * 1000;
		break;
	    
	      case 'f':
		i++;
		metsys_dir = argv[i];
		break;

	      case 'n':
		if ( isdigit(argv[i][2]) )
		    mrecs = atoi( &argv[i][2] );
		else
		    mrecs = atoi( argv[++i] );
		if ( mrecs > NRECS )
		    mrecs = NRECS;
		break;

	      default:
		printf("Unknown command line switch \"%s\"\n", argv[i]);
	    }
    }

    sprintf( metsys_file, "%s/%05d", metsys_dir, itime );
    sprintf( ctd_file, "%s/%05d", ctd_dir, itime );
    sprintf( gps_file, "%s/%05d", gps_dir, itime );
    mfp = fopen( metsys_file, "r" );
    cfp = fopen( ctd_file, "r" );
    gfp = fopen( gps_file, "r" );

    crecs = mrecs;
    grecs = mrecs/3 + 1;
    starttime = 0.0;

    if ( mfp == NULLF )
    {
	printf( "Could not open METSYS file %s/%05d\n", metsys_dir, itime );
	mrecs = 0;
    }

    if ( cfp == NULLF )
    {
	printf( "Could not open CTD file %s/%05d\n", ctd_dir, itime );
	crecs = 0;
    }

    if ( gfp == NULLF )
    {
	printf( "Could not open GPS file %s/%05d\n", gps_dir, itime );
	grecs = 0;
    }

    if ( mfp != NULLF )
    {
	fseek( mfp, (long)(mrecs+1) * MLINELEN * (-1), SEEK_END );
	fgets(buffer, LINEBUFLEN, mfp);
	for ( i = 0; fgets(buffer, LINEBUFLEN, mfp) != NULL; )
	{
	    if ( buffer[0] == '#' )
		continue;
	    rtn = sscanf(buffer, " %lg %d %d %d %d %lg %lg %lg %lg %lg %lg %lg %lg %lg %lg",
		   &metrecs[i].met_time, &metrecs[i].met_format,
		   &metrecs[i].met_jday, &metrecs[i].met_hour,
		   &metrecs[i].met_min, &metrecs[i].met_battery,
		   &metrecs[i].met_temp, &metrecs[i].met_humid,
		   &metrecs[i].met_barometer, &metrecs[i].met_windspeed,
		   &metrecs[i].met_winddir, &metrecs[i].met_windu,
		   &metrecs[i].met_windv, &metrecs[i].met_buoy_head,
		   &metrecs[i].met_windvane);
	    if ( rtn >= NFIELDS )
		i++;
	}
	fclose( mfp );
	mrecs = i;
    }

    if ( mrecs <= 0 )
	printf( "No MetSys data in file %s\n", metsys_file );
    else
	starttime = metrecs[0].met_time;

    if ( cfp != NULLF )
    {
	for ( i = 0; fgets(buffer, LINEBUFLEN, cfp) != NULL; )
	{
	    if ( buffer[0] == '#' )
		continue;
	    if ( sscanf(buffer, " %lg %d %lg %lg %lg %lg %lg %lg",
			&ctdrecs[i].ctd_time, &ctdrecs[i].ctdd.ctd_sample,
			&ctdrecs[i].ctdd.ctd_press, &ctdrecs[i].ctdd.ctd_temp,
			&ctdrecs[i].ctdd.ctd_cond, &ctdrecs[i].ctdd.ctd_sal,
			&ctdrecs[i].ctdd.ctd_trans, &ctdrecs[i].ctdd.ctd_fluor)
		>= 8 )
		if ( ctdrecs[i].ctd_time > starttime - START_WINDOW )
		    i++;
	}
	fclose( cfp );
	crecs = i;
    }

    if ( crecs <= 0 )
	printf( "No CTD data in file %s\n", ctd_file );

    if ( gfp != NULLF )
    {
	for ( i = 0; fgets(buffer, LINEBUFLEN, gfp) != NULL; )
	{
	    if ( buffer[0] == '#' )
		continue;
	    if ( sscanf(buffer, " %lg %lg %lg %lg %d %d", 
			&gpsrecs[i].gps_time, &gpsrecs[i].gps_lat, 
			&gpsrecs[i].gps_long, &gpsrecs[i].gps_std,
			&gpsrecs[i].gps_samples, &gpsrecs[i].gps_status) == 6 )
		if ( gpsrecs[i].gps_time > starttime - START_WINDOW )
		    i++;
	}
	fclose( gfp );
	grecs = i;
    }

    if ( grecs <= 0 )
	printf( "No GPS data in file %s\n", gps_file );

    yday = itime % 1000;
    for ( mo = 0; (yday > julday(year,mo+1,0)) && (mo < 11); mo++ )
	;				/* Loop to find month		*/

    mday = yday - julday(year,mo,0);	/* Find day of month		*/
		/* Get time_t for 00:00.  The +30 is for rounding minutes*/
    basetime = ep_days(year - YEAR0, yday) * SECS_PER_DAY + 30;

    printf("\nLatest M1 Meteorological and CTD data from %s %d, %d (Julian day %d)\n\n", 
	   months[mo], mday, year + 1900, yday );
    printf("Local  Air    Rel    Wind  Barometer Surface Salinity      Position\n");
    printf("Time   Temp  Humid kts from   mbar     Temp    o/oo\n");

    for ( i = j = k = 0; (i < mrecs); )
    {
	mtm = basetime + (SECS_PER_DAY * fmod(metrecs[i].met_time, 1.0));
	if ( j < crecs )
	    ctm = basetime + (SECS_PER_DAY * fmod(ctdrecs[j].ctd_time, 1.0));
	else
	    ctm = (time_t)0x7fffffff;
	if ( k < grecs )
	    gtm = basetime + (SECS_PER_DAY * fmod(gpsrecs[k].gps_time, 1.0));
	else
	    gtm = (time_t)0x7fffffff;
	tp = localtime(&mtm);

	if ( abs(ctm - mtm) <= 300 )
	{
	    printf("%2d:%02d ", tp->tm_hour, tp->tm_min);
	    printf("%5.1f %4.0f  %5.1f %3.0f   %4.0f", 
		   metrecs[i].met_temp, metrecs[i].met_humid,
		   metrecs[i].met_windspeed, metrecs[i].met_winddir,
		   metrecs[i].met_barometer);
	    i++;
	    printf("  %7.3f %8.4f", 
		   ctdrecs[j].ctdd.ctd_temp, ctdrecs[j].ctdd.ctd_sal );
	    j++;
	}
	else if ( mtm <= ctm )
	{
	    printf("%2d:%02d ", tp->tm_hour, tp->tm_min);
	    printf("%5.1f %4.0f  %5.1f %3.0f   %4.0f             ", 
		   metrecs[i].met_temp, metrecs[i].met_humid,
		   metrecs[i].met_windspeed, metrecs[i].met_winddir,
		   metrecs[i].met_barometer);
	    i++;
	}
	else
	{
	    tp = localtime(&ctm);
	    printf("%2d:%02d                               ", 
		   tp->tm_hour, tp->tm_min);
	    printf("%7.3f %8.4f", 
		   ctdrecs[j].ctdd.ctd_temp, ctdrecs[j].ctdd.ctd_sal );
	    j++;
	}

	if ( gtm <= mtm + 300 )
	{
	    lat = labs((Int)gpsrecs[k].gps_lat);
	    lng = labs((Int)gpsrecs[k].gps_long);
	    printf( " %d:%06.3fN %d:%06.3fW", 
		   lat, (fabs(gpsrecs[k].gps_lat) - (double)lat) * 60.0,
		   lng, (fabs(gpsrecs[k].gps_long) - (double)lng) * 60.0 );
	    k++;
	}
	printf("\n");
    }

    exit( 0 );

} /* main() */
