/************************************************************************/
/* Copyright 1991 MBARI							*/
/************************************************************************/
/* $Header: decode.h,v 3.5 2002/02/13 14:44:57 oasisa Exp $			*/
/* Summary  : Definitions for decoding OASIS data			*/
/* Filename : decode.h							*/
/* Author   : Robert Herlien (rah)					*/
/* Project  : OASIS Mooring						*/
/* $Revision: 3.5 $							*/
/* Created  : 02/06/91							*/
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/************************************************************************/
/* Modification History:						*/
/* 06feb91 rah - created						*/
/* $Log:	decode.h,v $
 * Revision 3.5  2002/02/13  14:44:57  14:44:57  oasisa (Oasis users)
 * Support for serial shutter
 * 
 * Revision 3.4  2001/08/29  09:26:14  09:26:14  oasisa (Oasis users)
 * hobi HR chan separation
 * NT base dir change
 * 
 * Revision 3.3  2001/08/03  14:05:22  14:05:22  oasisa (Oasis users)
 * add support for emeter, sbe43_o2, sbe47_ct
 * 
 * Revision 3.2  2001/07/11  18:59:59  18:59:59  oasisa (Oasis users)
 * Added emeter support
 * 
 * Revision 3.1  2001/06/19  13:19:05  13:19:05  oasisa (Oasis users)
 * Periodic Update 6/19/2001 (klh)
 * 
 * Revision 3.0  99/05/12  10:11:21  10:11:21  bobh (Bob Herlien)
 * Added tstring, misc changes
 * 
 * Revision 2.9  98/08/24  13:45:47  13:45:47  bobh (Bob Herlien)
 * Archiving sources after M2/M3 & Eqpac deployments of 1998
 * 
 * Revision 2.8  98/03/17  11:11:34  11:11:34  bobh (Bob Herlien)
 * Archiving sources prior to porting to DOS/Windows
 * 
 * Revision 2.7  97/09/09  09:52:42  09:52:42  bobh (Bob Herlien)
 * Archiving various changes
 * 
 * Revision 2.6  96/05/30  15:07:52  15:07:52  bobh (Bob Herlien)
 * Update for version in use during 1995-6 deployment
 * 
 * Revision 2.5  94/12/15  10:59:37  10:59:37  hebo (Bob Herlien)
 * Accumulated minor changes, mainly due to move to tsunami
 * 
 * Revision 2.4  94/01/21  14:36:10  14:36:10  hebo (Bob Herlien)
 * Added support for date ranges in cfg file
 * 
 * Revision 2.3  94/01/17  11:07:36  11:07:36  hebo (Bob Herlien)
 * Misc changes
 * 
 * Revision 2.2  92/09/24  18:36:45  18:36:45  hebo (Bob Herlien)
 * Changed trfl to trans, fluor on CTD; due to transposing cables on some CTDs
 * 
 * Revision 2.0  92/08/31  15:35:45  15:35:45  hebo (Bob Herlien)
 * Auguest 1992 Deployment.  Changed to allow multiple sensors of same type.
 * 
 * Revision 1.1  92/06/15  09:08:03  09:08:03  hebo (Bob Herlien)
 * *** empty log message ***
 * 
 * Revision 1.0  92/03/04  17:19:03  17:19:03  hebo (Bob Herlien)
 * Initial revision
 * 
*/
/************************************************************************/

#ifdef WIN32
#define strcasecmp(s1, s2)	stricmp(s1, s2)
#define bzero(s, n)		memset(s, 0, n)
#endif

#define deblank(s)	while(isspace(*s)) ((s)++)


#define MAXLOGSIZE	2048		/* Max size of log record	    */
#define MAXLOGNUM	6553		/* Max Nat32 / sizeof(LogRecHdr)    */
#define NUM_CTDS	3		/* Number of CTDs we can decode	    */
#define NUM_NO3S	2		/* Number of NO3s we can decode	    */
#define NUM_SPECS	4		/* Number of Spectros we can decode */
#define CTD_CAL		0		/* Surface CTD uses cal file 0 	    */
#define CTD2_CAL	1		/* Second CTD uses cal file 1 	    */
#define CTD3_CAL	2		/* Third CTD uses cal file 2 	    */
#define NO3_CAL		0		/* First NO3 uses cal file 0	    */
#define NO3_CAL2	1		/* Second NO3 uses cal file 1	    */
#define SPECTRO_CAL	0		/* Surface spectro uses cal file 0  */
#define SPECTRO2_CAL	1		/* 10m spectro uses cal file 1	    */
#define SPECTRO3_CAL	2		/* 20m spectro uses cal file 2	    */
#define SPECPRR_CAL	3		/* PRR-600 spectro uses cal file 3  */

					/************************************/
/*	Data Logging Record Types	*/
#define LOG_EMPTY	0		/* Unused - empty record	    */
#define ERRORF		0		/* Also used for error file ptr	    */
#define OASIS_STAT	1		/* OASIS on/off status		    */
#define ATLAS		2		/* atlas record			    */
#define OASIS_CAN	3		/* OASIS battery, temp, press	    */
#define PAR		4		/* PAR sensors			    */
#define CTD		5		/* CTD Data			    */
#define SPECTRO		6		/* MER Spectroradiometer	    */
#define ADCP		7		/* Acoustic Doppler		    */
#define GPS		8		/* Old style GPS (10^-3 minutes)    */
#define MODEM		9		/* Acoustic Modem		    */
#define PCO2		10		/* pCO2 sensor			    */
#define CTD2		11		/* Second CTD			    */
#define CTD3		12		/* Third CTD			    */
#define SPECTRO2	13		/* Second MER Spectroradiometer	    */
#define SPECTRO3	14		/* Third MER Spectroradiometer	    */
#define NO3		15		/* Nitrate sensor		    */
#define NO32		16		/* Second Nitrate sensor	    */
#define SPEC_PRR	17		/* PRR-600 Spectroradiometer	    */
#define SATLANTIC	18		/* Satlantic Spectroradiometer	    */
#define GPS_TYPE2	19		/* Style 2 GPS (10^-7 degrees)	    */
#define NRL		20		/* NRL satellite controller	    */
#define O2		21		/* Oxygen sensor		    */
#define FLUOR		22		/* Fluorometer			    */
#define TRANSMISS	23		/* Transmissometer		    */
#define NO3_DECODED	24		/* Nitrate sensor, decoded data	    */
#define NO3_DECODED2	25		/* Second Nitrate sensor, decoded   */
#define AC9		26		/* Oxygen sensor		    */
#define CO2_PUMP	27		/* pCO2 Pump			    */
#define H2O_PUMP	28		/* Water Pump			    */
#define SHUTTER0	29		/* Shutter 0 for PRR Spectro	    */
#define SHUTTER1	30		/* Shutter 1 for PRR Spectro	    */
#define SPEC_PRR_VOLTS	31		/* PRR-600 Spectro, in volts	    */
#define METSYS		32		/* Campbell Scientific Met Logger   */
#define TSTRING		33		/* Seabird Inductive Temp String    */
#define GNDFAULT	34		/* Serial Analog Ground Fault	    */
#define MICROCAT	35		/* Seabird serial microCat CTD	    */
#define GPS_TYPE3	36		/* Garmin GPS receiver		    */
#define HOBI_AB		37		/* HOBI Labs alpha-beta		    */
#define HOBI_CB		38		/* HOBI Labs c-beta		    */
#define HOBI_HS2	39		/* HOBI Labs HydroScat 2	    */
#define HOBI_HR1	40		/* HOBI Labs HydroRad file 1	    */
#define HOBI_HR2	41		/* HOBI Labs HydroRad file 2	    */
#define HOBI_HR3	42		/* HOBI Labs HydroRad file 3	    */
#define HOBI_HR4	43		/* HOBI Labs HydroRad file 4	    */
#define HOBI_DAS        44              /* HOBI Labs HydroDAS file          */
#define GASHOUND        45              /* GasHound pCO2 Sensor (TT8B)	    */
#define ISUS            46		/* ISUS Nitrate Analyzer (TT8C)     */
#define EMETER          47
#define SBE47_CT        48              /* Seabird low power CT             */
#define SBE43_O2        49              /* Seabird O2                       */
#define HOBI_HR11	50		/* HOBI Labs 1 ch HydroRad ch 1     */
#define HOBI_HR21	51		/* HOBI Labs 2 ch HydroRad ch 1	    */
#define HOBI_HR22	52		/* HOBI Labs 2 ch HydroRad ch 2	    */
#define HOBI_HR31	53		/* HOBI Labs 3 ch HydroRad ch 1	    */
#define HOBI_HR32	54		/* HOBI Labs 3 ch HydroRad ch 2	    */
#define HOBI_HR33	55		/* HOBI Labs 3 ch HydroRad ch 3	    */
#define HOBI_HR41	56		/* HOBI Labs 4 ch HydroRad ch 1	    */
#define HOBI_HR42	57		/* HOBI Labs 4 ch HydroRad ch 2	    */
#define HOBI_HR43	58		/* HOBI Labs 4 ch HydroRad ch 3	    */
#define HOBI_HR44	59		/* HOBI Labs 4 ch HydroRad ch 4	    */
#define SERIAL_SHUTTER 60   /* MBARI serial shutter             */
#define SPECIAL1	61		/* For temporary or test instruments*/
#define SPECIAL2	62		/* For temporary or test instruments*/

#define SENSORS		(SPECIAL2 + 1)	/* Number of sensors		    */
					/************************************/
#define LOG_ERROR	128		/* Error record			    */
#define LOG_COMMENT	129		/* User comments		    */
#define LOG_BIN		130		/* Miscellaneous Binary data	    */
#define LOG_DISK_ERROR	131		/* Disk Error, 1 word of error reg  */
					/************************************/
	/* OASIS Error types as logged in OASIS data log		    */
#define RAM_ERR		0x8001		/* RAM not initialized		    */
#define LOG_ERR		0x8002		/* Bogus log memory		    */
#define CLOCK_ERR	0x8004		/* Real-time clock failure	    */
#define INT_ERR		0x8008		/* Received spurious interrupt	    */
#define RESTART_ERR	0x8010		/* Restarted with bad keepalive     */
#define COMM_ERR	0x8020		/* Restarted due to no comm in 24hrs*/
#define DISK_ERR	0x8040		/* Error logging to disk	    */
#define ARGOS_ERR	0x8080		/* Error sending to ARGOS	    */
					/************************************/
	/* OASIS Instrument Error types					    */
#define TMOUT_ERR	(-2)		/* Serial timeout		    */
#define MALLOC_ERR	(-3)		/* No buffer space from malloc	    */
#define NO_DATA		(-4)		/* No data from device		    */
#define BAD_DATA	(-5)		/* Bad data from device		    */
#define CKSUM_ERR	(-6)		/* Checksum failure		    */
#define SYNC_ERR	(-7)		/* Failure to sync to device	    */
#define	NOT_AVAIL_ERR	(-8)		/* Device says not avail (eg GPS)   */
#define GPS_OOPS	0x84		/* No 1st fix from GPS, rtn by bug  */

	/* Return types from decode subroutines  ****************************/
#define UUEND		(-3)		/* uugetline() found "end" statement*/
#define	SIZE_ERR	(-9)		/* Data length error		    */
#define CHKSUM_ERR	(-10)		/* Checksum error		    */
#define FMT_ERR		(-11)		/* Data format error		    */
	/* Miscellaneous		 ************************************/
typedef int		Status;		/* Status return type		    */
#define NULLF		(FILE *)0	/* Null file pointer		    */
#define ANALOG_CHANS	8		/* Number of analog channels	    */
#define PAR_CHAN	0		/* Analog channel number for PARs   */
#define OASIS_CHAN	2		/* Analog chan num for OASIS stuff  */
#define FLUOR_CHAN	0		/* Analog channel for fluorometer   */
#define TRANSMISS_CHAN	1		/* Analog channel for transmissometer*/
#define FLOW_CHAN	1		/* Analog channel for pump flowmeter*/
#define SBE43_O2_CHAN	0		/* Analog channel for SBE43 O2 sensor*/
					/************************************/
					/* ATLAS types and defines	    */
#define TEMPS		10		/* Number temperature pods	    */
#define PRESS		2		/* Number pressure pods	 	    */
#define ATLAS_WORDS	18		/* Number words in orig ATLAS record*/
#define ATLAS_LONG	20		/* Number words in new ATLAS record */
					/************************************/
typedef enum				/************************************/
{					/* FileType - Type of input data file*/
    FT_UUENCODE,			/* uuencoded data file		    */
    FT_HEX,				/* ASCII hex data file		    */
    FT_BIN				/* Binary data file (from YModem)   */
} FileType;				/************************************/

typedef struct				/************************************/
{					/* LogRecHdr - Logging Record Header*/
    Byte	log_type;		/* Record type - see below	    */
    Nat16	log_nmbr;		/* Record number		    */
    Int16	log_len;		/* Number of bytes in log_data	    */
    Nat32	log_time;		/* Record time			    */
} LogRecHdr;				/************************************/

#define LOG_SYNC	0x55		/* Binary sync char		    */
#define HDR_SIZE	9		/* sizeof(LogRecHdr) wrong due to pad*/

typedef struct				/************************************/
{					/* Calibr struct for single ATLAS chan*/
    Int32	pod_type;		/* Pod type			    */
    Int32	ident;			/* Pod ID number		    */
    double	a;			/* A calibration coefficient	    */
    double	b;			/* B calibration coefficient	    */
    double	c;			/* C calibration coefficient	    */
    double	d;			/* D calibration coefficient	    */
    double	e;			/* E calibration coefficient	    */
} ACal;					/************************************/

typedef struct				/************************************/
{					/* Struct for ATLAS calibration	    */
    ACal	ac_air;			/* Air temp calibration		    */
    ACal	ac_sst;			/* Sea surface temp calibration	    */
    ACal	ac_temp[TEMPS];		/* Temperature pod calibrations	    */
    ACal	ac_press[PRESS];	/* Pressure pod calibrations	    */
    ACal	ac_rh;			/* Relative humidity calibration    */
    ACal	ac_wind;		/* Wind calibration		    */
} AtlasCal;				/************************************/

typedef struct				/************************************/
{					/* Struct to define ATLAS data	    */
    Nat16	at_time;		/* Date/time			    */
    Nat16	at_air;			/* Air temp			    */
    Nat16	at_sst;			/* Sea surface temp		    */
    Nat16	at_temp[TEMPS];		/* Temperature pods		    */
    Nat16	at_press[PRESS];	/* Pressure Pods		    */
    Nat16	at_rh;			/* Relative humidity		    */
    Nat16	at_wind;		/* East-west, north-south wind vects*/
    Nat16	at_chksum;		/* 16-bit checksum		    */
} Atlas;				/************************************/

typedef struct				/************************************/
{					/* Struct to define new ATLAS data  */
    Nat16	at_time;		/* Date/time			    */
    Nat16	at_air;			/* Air temp			    */
    Nat16	at_sst;			/* Sea surface temp		    */
    Nat16	at_temp[TEMPS];		/* Temperature pods		    */
    Nat16	at_press[PRESS];	/* Pressure Pods		    */
    Nat16	at_rh;			/* Relative humidity		    */
    Nat16	at_winduv;		/* East-west, north-south wind vects*/
    Nat16	at_covane;		/* Raw Compass, Vane values	    */
    Nat16	at_windspeed;		/* Raw wind speed		    */
    Nat16	at_chksum;		/* 16-bit checksum		    */
} AtlasLong;				/************************************/

typedef struct				/************************************/
{					/* Struct for decoded ATLAS data    */
    double	atd_time;		/* Date/time			    */
    double	atd_air;		/* Air temp			    */
    double	atd_sst;		/* Sea surface temp		    */
    double	atd_temp[TEMPS];	/* Temperature pods		    */
    double	atd_press[PRESS];	/* Pressure Pods		    */
    double	atd_rh;			/* Relative humidity		    */
    double	atd_windspd;		/* Wind speed			    */
    double	atd_winddir;		/* Wind direction		    */
    double	atd_windu;		/* Calibrated U (east-west) vector  */
    double	atd_windv;		/* Calibrated V (north-south) vector*/
    double	atd_compass;		/* Raw compass output		    */
    double	atd_vane;		/* Raw wind vane output		    */
    double	atd_rawdir;		/* Raw wind direction (co + va + mag)*/
    double	atd_rawspeed;		/* Raw wind speed output	    */
} AtlasDecode;				/************************************/

typedef struct				/************************************/
{					/* Struct for METSYS data    */
    int         arr_type;               /* array type                       */
    int 	day;		        /* day			            */ 
    int 	hr;		        /* hour			            */
    int 	min;		        /* min   		            */
    double 	batt_volt;	        /* battery voltage	            */
    double	air;		        /* Air temp			    */
    double	rh;			/* Relative humidity		    */
    double	baro_press;		/* Barometric pressure   	    */
    double	windspd;		/* Wind speed			    */
    double	winddir;		/* Wind direction from, deg. mag    */
    double	windu;		        /* Calibrated U (east-west) vector  */
    double	windv;		        /* Calibrated V (north-south) vector*/
    double	altwindspd;		/* Wind speed from 2nd indicator    */
    double	altwinddir;		/* Wind direction from, deg. mag    */
    double	buoy_heading;		/* Buoy heading, deg. mag	    */
    double	rel_winddir;		/* deg. relative to "prop. over box */
    double	sound_velocity;		/* Sound velocity		    */
} Metsys;				/************************************/

#define MAX_MET_FIELDS	20
#define UNITNAMELEN	32		/* Length of buffer for units name  */

typedef struct				/************************************/
{					/* Struct for Analog calibration    */
    double	a;			/* f(x) = (a*x + b) * c + d	    */
    double	b;			/*                  b calibr. value */
    double	c;			/*                  c calibr. value */
    double	d;			/*                  d calibr. value */
    char	units[UNITNAMELEN];	/* Name of units		    */
} Analog;				/************************************/

#define CTD_BYTES_LONG  11
#define CTD_BYTES_SHORT	 9		/* Number bytes in short CTD record */
#define CTD_BYTES_TINY   7

#define CTD_BYTES	CTD_BYTES_TINY	/* Number bytes in CTD record	    */

typedef struct				/************************************/
{					/* Single CTD Calibration struct    */
    int		offset;			/* Byte offset in data stream	    */
    double	a;			/* A or A0 calibration coefficient  */
    double	b;			/* B or A1 calibration coefficient  */
    double	c;			/* C or A2 calibration coefficient  */
    double	d;			/* D calibration coefficient	    */
    double	mf;			/* M or F0 calibration coefficient  */
} CCal;					/************************************/

typedef struct				/************************************/
{					/* Struct for CTD calibration	    */
    int		cc_size;		/* Size of CTD data		    */
    CCal	cc_cond;		/* Conductivity calibration	    */
    CCal	cc_temp;		/* Temperature calibration	    */
    CCal	cc_press;		/* Pressure calibration		    */
    CCal	cc_trans;		/* Transmissometer		    */
    CCal	cc_fluor;		/* Fluorometer			    */
    CCal	cc_sample;		/* Sample number		    */
} CTDCal;				/************************************/

typedef struct				/************************************/
{					/* Struct for decoded CTD data      */
    Nat16	ctd_sample;		/* Sample number		    */
    double	ctd_press;		/* Pressure			    */
    double	ctd_temp;		/* Temperature			    */
    double	ctd_cond;		/* Conductivity			    */
    double	ctd_sal;		/* Salinity			    */
    double	ctd_trans;		/* Transmissometer voltage	    */
    double	ctd_fluor;		/* Fluorometer voltage		    */
} CTDDecode;				/************************************/

typedef enum
{ NO3_DRIFT1, NO3_M1, NO3_M2 } No3Format;

typedef struct				/************************************/
{					/* Struct for No3 calibration	    */
    No3Format	nc_format;		/* Format that data is in	    */
    char	nc_samplechar;		/* Character that denotes sample    */
    Int32	nc_maxCounts;		/* Maximum A/D count		    */
    Int32	nc_absMult;		/* Absorption multiplier	    */
    double	nc_Co;			/* Conductivity reference	    */
    double	nc_Cf;			/* Conductivity frequency reference */
    double	nc_To;			/* Temperature reference	    */
    double	nc_Tf;			/* Temperature frequency reference  */
    double	nc_pressure;		/* Pressure to use		    */
} No3Cal;				/************************************/

#define	NO3_TIMELEN	16

typedef struct				/************************************/
{					/* Struct for decoded No3 data      */
    char	no3_time[NO3_TIMELEN];	/* Time string			    */
    Int16	no3_sample;		/* Sample number		    */
    double	no3_conc;		/* NO3 concentration		    */
    double	no3_temp;		/* Temperature at NO3 analyzer	    */
    double	no3_salinity;		/* Salinity			    */
    double	no3_conductivity;	/* Conductivity			    */
} No3Decode;				/************************************/

					/************************************/
#define	SPEC_BANKS	6		/* Number of A/D banks in spectro   */
					/* or addresses in PRR-600	    */
#define SPEC_CHANS	20		/* Number of A/D channels per bank  */
#define CHAN_NAMELEN	8		/* Length of channel name	    */
#define UNIT_NAMELEN	20		/* Length of units name	 	    */
					/************************************/

typedef struct				/************************************/
{					/* Struct for cal of one spectro chan*/
    Int		type;			/* 0 = unused, 1 = down irrad,	    */
					/* 2 = analog, 3 = up radiance,	    */
					/* 4 = up irrad, 5 = quadratic	    */
					/* 6 = Satlantic (a is add'l mult)  */
    char	name[CHAN_NAMELEN];	/* Name of channel		    */
    double	a;			/* x^2 coef (type 5) or multiplier (6)*/
    double	b;			/* multiplier (PRR) or divisor (MER)*/
    double	c;			/* subtractive offset		    */
    char	units[UNIT_NAMELEN];	/* Units of measure		    */
} SpecChanCal;				/************************************/

typedef struct				/************************************/
{					/* Spectro Calibration struct	    */
    Int		spc_totchans;		/* Total number of channels	    */
    Int		spc_nbanks;		/* Number of active banks	    */
    Int		spc_nchans[SPEC_BANKS];	/* Number of active chans per bank  */
    SpecChanCal spc_cal[SPEC_BANKS][SPEC_CHANS]; /* Channel calibrations    */
} SpecCal;				/************************************/

typedef struct				/************************************/
{					/* Struct 1 chan of decoded spectro */
    double	spc_mean;		/* Mean				    */
    double	spc_std;		/* Standard deviation		    */
    double	spc_min;		/* Minimum value		    */
    double	spc_max;		/* Maximum value		    */
} SpecChanDecode;			/************************************/

typedef struct				/************************************/
{					/* Struct for decoded spectro data  */
    Int		spc_year;		/* Year of collection		    */
    Int		spc_day;		/* Julian day			    */
    Nat32	spc_sec;		/* Seconds since midnight	    */
    SpecChanDecode spc_chan[SPEC_BANKS][SPEC_CHANS]; /* Channel output values*/
} SpecDecode;				/************************************/

typedef struct				/************************************/
{					/* Struct for decoded spectro data  */
    Int		prr_nchans;		/* Number of channels found	    */
    double	prr_chan[SPEC_CHANS];	/* Channel output values	    */
} PrrChanDecode;			/************************************/

typedef struct				/************************************/
{					/* Struct for decoded spectro data  */
    Int		prr_nbanks;		/* Number of banks found	    */
    PrrChanDecode prr_bank[SPEC_BANKS];	/* Bank decode structs		    */
} PrrDecode;				/************************************/

#define MAX_SAT_CHANS	12		/* Max number of satlantic channels */

typedef struct				/************************************/
{					/* Satlantic Spectro calibration type*/
    char	sat_name[CHAN_NAMELEN];	/* Channel name			    */
    double	sat_factor;		/* Calibration factor F		    */
    double	sat_immers;		/* Immersion factor		    */
    double	sat_offset;		/* Offset counts		    */
} SatlantChanCal;			/************************************/

typedef struct				/************************************/
{					/* Struct for of decoded spectro data*/
    Int		sat_unused_chans;	/* Number of channels to skip	    */
    Int		sat_chans;		/* Number of channels to decode	    */
    SatlantChanCal sat_cal[MAX_SAT_CHANS]; /* Channel calibrations	    */
} SatlanticCal;				/************************************/

typedef double	SatlanticDecode[MAX_SAT_CHANS];

#define AC9_CHANS	18		/* Number of channels of AC-9 data  */

typedef struct				/************************************/
{					/* Struct for AC-9 data		    */
    Nat32	ac9_data[AC9_CHANS];	/* Signal for each channel	    */
    Nat32	ac9_ref[AC9_CHANS];	/* Reference for each channel	    */
    Nat32	ac9_temp;		/* AC-9 temperature		    */
    Nat16	ac9_samples;		/* Number of samples taken	    */
} Ac9Data;				/************************************/

typedef struct				/************************************/
{					/* Struct for Decoded AC-9 data	    */
    Nat32	ac9_samples;		/* Number of samples taken	    */
    double	ac9_value[AC9_CHANS];	/* Signal for each channel	    */
    double	ac9_temp;		/* AC-9 temperature		    */
} Ac9Decode;				/************************************/

#define MAX_GF_CALS	6		/* Max number ground fault channels */
#define GFIDENTLEN	48		/* Length of buffer for GF ident    */

typedef struct				/************************************/
{					/* Struct for Ground Fault calibr.  */
					/* f(x) = (x - offset)/divisor	    */
    Int		raw_offset;		/* Offset into raw data		    */
    double	cal_offset;		/*  "offset" in equation above	    */
    double	cal_divisor;		/*  "divisor" in equation above	    */
    MBool	cal_valid;		/* Valid calibration		    */
    char	units[UNITNAMELEN];	/* Name of units		    */
    char	ident[GFIDENTLEN];	/* Field identification		    */
} GndFltCal;				/************************************/

#define MAX_TSTRING	12		/* Max number Seacats in tstring    */

typedef struct				/************************************/
{					/* Config struct for one TSTRING pod*/
    Int         ser_no;                 /* Instrument serial number         */
    Int         depth;                  /* Nominal depth inst. deployed at  */
    Int         press_flag;             /* Pressure sensor,yes=1,no=0       */ 
} TStringPodCfg;			/************************************/

typedef struct				/************************************/
{					/* Struct for TSTRING configuration */
    Int         numValidPods;		/* Number of valid pods in config   */
    TStringPodCfg pod[MAX_TSTRING];	/* Cfg for each pod		    */
} TStringCfg;				/************************************/

typedef struct				/************************************/
{					/* Struct for TSTRING data	    */
    Int         valid;			/* Has valid data		    */
    double	temp;		        /* Temp			            */
    double	cond;			/* Conductivity		            */
    double	press;		        /* Pressure   	                    */
    double	jdate;			/* yymmm.ttttt			    */
    Int         recnum;                 /* record number (4/18/01 klh)      */
} Tstring;				/************************************/


/************************************************/
/* Declarations for log records found/missing	*/
/************************************************/

#define NLOGS	20 

#define NRANGES	20 

typedef struct				/************************************/
{					/* Range of found records	    */
    Nat32	min;			/* Beginning of range		    */
    Nat32	max;			/* End of range			    */
} RcdRange;				/************************************/

typedef struct				/************************************/
{					/* Structure defining one OASIS log */
    Nat32	nRanges;		/* Number of ranges found	    */
    RcdRange	range[NRANGES];		/* Log record number ranges	    */
} OneLog;				/************************************/

typedef struct				/************************************/
{					/* Structure defining logs found    */
    Int32	firstLog;		/* First log found		    */
    Int32	nLogs;			/* Number of logs found		    */
    OneLog	logs[NLOGS];		/* Logs found			    */
} LogStruct;				/************************************/
