/****************************************************************************/
/* Copyright 1998 MBARI                                                     */
/****************************************************************************/
/* $Header: tstring.c,v 4.4 2001/06/19 12:16:06 oasisa Exp $		    */
/* Summary  : Driver Routines for Seabird Inductive Temp String		    */
/* Filename : tstring.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.4 $							    */
/* Created  : 06/01/98							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 1jun98 rah - created							    */
/* $Log:	tstring.c,v $
 * Revision 4.4  2001/06/19  12:16:06  12:16:06  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:45:07  11:45:07  oasisa (Oasis users)
 * Initial revision
 * 
 * Revision 4.3  99/06/16  10:21:42  10:21:42  bobh (Bob Herlien)
 * Mar/May '99 Deployments of M3/M2
 * 
 * Revision 4.2  98/09/10  10:11:43  10:11:43  bobh (Bob Herlien)
 * Sept/Oct '98 deployments of M1, Eqpac 1 & 2
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <80C196.h>			/* 80196 Register mapping           */
#include <task.h>			/* OASIS Multitasking definitions   */

#define		MAX_PODNUM	16
#define		POD_BUFSIZE	80	/* Changed for added rec #, orig: 64*/
#define		PWRON_SECS	6
#define		PWRON_TICKS	(PWRON_SECS * TICKS_PER_SECOND)
#define		PROMPT_SECS	1
#define		NEWPOD_TICKS	20
#define		ECHO_TICKS	20
#define		TURNOFF_TIME	TICKS_PER_SECOND


/********************************/
/*	External Functions	*/
/********************************/

Extern char	*drvSerPortAndMalloc( Driver *dp, Nat16 size );
Extern Void	drvSerReleaseAndFree( Driver *dp, char *buffer );
Extern Void	drv_pwroff( Driver *dp );
Extern Void	drvLog( Driver *dp, Byte *samplep, Int16 len );
Extern Void	drvLogError( Driver *dp, Errno err );
Extern Void	xputc( Int16 c );
Extern Void	xputs( const char *s );
Extern Void	xprintf( const char *format, ... );
Extern Int16	xgets_tmout_flg( char *s, Int16 len, Nat16 tmout, Word flags );
Extern Void	xflush_ser( Nat16 timeout );
Extern Void	bcopy( const Byte *src, Byte *dst, Nat16 len );
Extern Int16	getCtdPrompt( Nat16 timeout );
Extern Void	tmpFree( char *ptr );
Extern Void	delay_secs( Nat16 secs );
Extern Nat16	bitsOn( Nat16 num );


/************************************************************************/
/* Function    : get_tpod						*/
/* Purpose     : Make one attempt to get Temp pod data			*/
/* Inputs      : Driver pointer, Pod number, Buffer, buffer length	*/
/* Outputs     : Number of bytes returned				*/
/************************************************************************/
	Int16
get_tpod( Driver *dp, Nat16 podNum, char *podBuf )
{
    Int16	tries;
    Reg Int16	len, c;

    for ( tries = dp->drv_parms[PARM0]; tries; tries-- )
    {	
	xputc('\r');			/* Get CTD's attention		*/
	if ( getCtdPrompt(PROMPT_SECS) != ERROR )
	{				/* If got prompt, try to send cmd*/
	    xprintf("#%02dSL\r", podNum);
	    xflush_ser( ECHO_TICKS );		/* Throw away echo chars*/
	
	    len = xgets_tmout_flg(podBuf, POD_BUFSIZE,
				  dp->drv_parms[TIMEOUT], INCLUDE_TERMCHAR);

	    if ( (len > 0) && (((c = podBuf[len-1]) == '\r') || (c == '\n')) )
		return( len );
	
	    xputs("\033\r");
	    xflush_ser( NEWPOD_TICKS );
	    xputs("PWRON\r");
	    xflush_ser( PWRON_TICKS );
	}
    }

    return( 0 );				/* Timeout, return 0	*/

} /* get_tpod() */


/************************************************************************/
/* Function    : tstring_drv						*/
/* Purpose     : Temperature String driver				*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
tstring_drv( Driver *dp )
{
    Reg Int16	podLen;
    Nat16	podNum, tbufLen;
    char	*tbuf, *podbuf;
    
    podbuf = drvSerPortAndMalloc(dp, (bitsOn(dp->drv_parms[PARM1])+1)*POD_BUFSIZE);
    if ( podbuf == NULL )			/* Get ser port & buffer*/
	return;

    xflush_ser( PWRON_TICKS );
    tbuf = podbuf + POD_BUFSIZE;

    for ( podNum = tbufLen = 0; podNum < MAX_PODNUM; podNum++ )
	if ( dp->drv_parms[PARM1] & (1 << podNum) )
	    if ( (podLen = get_tpod(dp, podNum, podbuf)) > 0 )
	    {
		bcopy( (Byte *)podbuf, (Byte *)tbuf + tbufLen, podLen );
		tbufLen += podLen;
	    }

    xputs("\r");
    delay_secs( 1 );
    xputs("PWROFF\r");				/* Turn them all off	*/
    xflush_ser( TURNOFF_TIME );
    drvLog( dp, (Byte *)tbuf, tbufLen );	/* Log Temp data	*/
    drvSerReleaseAndFree( dp, podbuf );		/* Release serial port	*/
    drv_pwroff( dp );				/* Turn off power	*/

} /* tstring_drv() */
