/****************************************************************************/
/* Copyright 1997 MBARI                                                     */
/****************************************************************************/
/* $Header: tinycom.c,v 1.1 2001/10/17 10:32:57 oasisa Exp $                                                              */
/* Summary  : Radio Routines for type 2 Shutters for PRR Spectro            */
/* Filename : tinycom.c                                                     */
/* Author   : Kent Headley (klh)                                            */
/* Project  : OASIS Mooring                                                 */
/* $Revision: 1.1 $                                                            */
/* Created  :  10/08/01                                                     */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:                                                    */
/*                                                                          */
/****************************************************************************/


/**************************************************************************** 

This module implements a reduced set of functions for shutter testing and 
calibration, suitable for use over the packet radio link, see shutcal.c for
further background.

Each input command line executes one operation, and provides a one line response.

The command line:

shutcom shutter0 ccw 200

Causes shutter number 0 to move in the counterclockwise direction for
two hundred "ticks", a tick being 0.01 second. The response line would look 
like:

shutter 0 arrived at 345 Inter   -or-   shutter 0 overload at 473 Open

The numeric value being the position, and the string Closed | Inter | Open 
representing the state of the paddles based on the current value of the position
relative to the closed or home position.


The available command options are:

ccw nnn Move in CCW direction for nnn ticks.

cw nnn  Move in CW direction for nnn ticks.

set     Set the current position as closed and update the drv_parms.

open    Call the shutter open routine with the current drv_parms.

close   Call the shutter close routine with the current drv_parms.

where   Read the current position of the paddles.


***************************************************************************/


#include <types.h>
#include <const.h>
#include <oasis.h>
#include <task.h>
#include <io.h>
#include <custom.h>
#include <tinyshut.h>
#include <string.h>

/********************************/
/*      External Functions      */
/********************************/

Extern  Void    xprintf( const char *format, ... );
Extern  Void    drv_pwron( Driver *dp );
Extern  Void    drv_pwroff( Driver *dp );
Extern  Driver * drvr_find( char *name );
/*
Extern  Void    runShutter(Driver *dp, MBool open, SemID sem);
*/
Extern  Void    runShutter(Driver *dp );
Extern  Nat16   io_atod( Nat16 channel );
Extern  char    *cmp_ulc( const char *s, const char *cs );

		
/********************************/
/*      External Data           */
/********************************/

Extern Reg Byte         ioctrl;                 /* Copy of oasis_ctrl               */
Extern Int16            ad_use_cnt;             /* Number of people using A/D       */
Extern const shutter_const shutter_io[NUMBER_OF_SHUTTERS];

/********************************/
/*      Module Local Data       */
/********************************/

const char *  const stateString[]=      /* see shut22.h for enumeration of State type */
  {"Open","Closed","Inter"};

const char *  const cmdNames[]=         /* legal command list */
{
    "cw",
    "ccw",
    "set",
    "open",
    "close",
    "where"
};

/************************************************************************/
/* Function    : shutterOn                                              */
/* Purpose     : handles power switching and init                       */
/* Inputs      : Pointers to drv_parms and shutter struct               */
/* Outputs     : none                                                   */
/************************************************************************/
	Void
shutterOn(Driver *dp, Shutter_data *sd) /* powerup and initialize shutter */
{
    initMotor(sd);                      /* set up the I/O port */

    ad_use_cnt++;                       /* Show we're using analog voltage  */
    ioctrl &= ~ANALOG_DSBL;             /* Turn on analog power             */
    oasis_ctrl = ioctrl;

    drv_pwron(dp);                      /* Turn on motor drive power */
    task_delay(TICKS_PER_SECOND/2);     /* and allow things to settle */

}   /* shutterOn() */

/************************************************************************/
/* Function    : shutterOff                                             */
/* Purpose     : handles power switching                                */
/* Inputs      : Pointers to drv_parms and shutter struct               */
/* Outputs     : none                                                   */
/************************************************************************/
	Void
shutterOff(Driver *dp)  /* power down the shutter */
{
    drv_pwroff(dp);                     /* Turn off motor drive power */

    if ( --ad_use_cnt <= 0)             /* No longer using analog voltage   */
    {                                   /* If no one else is,               */
	ioctrl |= ANALOG_DSBL;          /* Turn off analog power            */
	oasis_ctrl = ioctrl;
    }
}

/************************************************************************/
/* Function    : shutterJog                                             */
/* Purpose     : Moves shutter for specified time                       */
/* Inputs      : Pointers to drv_parms and shutter struct               */
/* Outputs     : String to stdout                                       */
/************************************************************************/
	Void
shutterJog(Driver *dp, Shutter_data *sd, MBool dir, Nat16 time)
{
    shutterOn(dp,sd);
    setDirMotor(dir,sd);
    startMotor(sd);
    for(; time > 0; time--)
    {
	if(!readMotor(sd))              /* determine if we have hit anything */
	{
	    stopMotor(sd);
	    xprintf("Shutter #%d overload at %d  %s\n",
		    sd->ShutterNumber,readShutter(sd),stateString[sector(sd)]);
	    shutterOff(dp);
	    return;
	}
	task_delay(1);
    }
    stopMotor(sd);
    xprintf("Shutter #%d arrived at %d  %s\n",
	    sd->ShutterNumber,readShutter(sd),stateString[sector(sd)]);
    shutterOff(dp);
}
/************************************************************************/
/* Function    : shutRun                                                */
/* Purpose     : handles power switching when calling runShutter        */
/* Inputs      : same parms as runShutter                               */
/* Outputs     : none                                                   */
/************************************************************************/
	Void
shutRun( Driver *dp, MBool open )
{
    xprintf("\nCalling runShutter...");
    dp->drv_usrparm = open;
    runShutter(dp);
    drv_pwron(dp);                      /* Turn on motor drive power */
    task_delay(20);
    xprintf("\nreturned from runShutter\n");
}

/************************************************************************/
/* Function    : shutcom                                                */
/* Purpose     : Shutter immediate commands                             */
/* Inputs      : Parms from user interface                              */
/* Outputs     : Results to stdout, can change drv_parms[PARM1], error  */
/************************************************************************/
	Int16
shutcom( Nat16 pmask, char *name, char *cmdString, Nat16 time )
{

    Shutter_data  sd;           /* structure used by all shutter functions for I/O addr resolution */
    Motor motorstate;           /* enumerated type for state definition */
    Driver *dp;                 /* driver structure defines pwr bits and home position */
    Nat16       cmdnum;
    Semaphore   sem;



    if ( ((pmask & 1) == 0) ||
	 ((dp = drvr_find(name)) == DRV_NULL) )
    {
	xprintf("Bad shutter driver name\n");
	return( ERROR );
    }
    if ((pmask & 2) == 0)
    {
	xprintf("No shutter command\n");
	return( ERROR );
    }

    sd.ShutterNumber = dp->drv_parms[PARM0];    /* init struct */
    sd.ClosePosition = dp->drv_parms[PARM1];
    sem_init( &sem, 0 );                        /* create a semaphore, needed by runShutter */

    for (cmdnum=0; cmdnum < (sizeof(cmdNames) / sizeof(char *)); cmdnum++)
    {
	if (cmp_ulc(cmdString, cmdNames[cmdnum]) != NULL)
	{
	    switch(cmdnum)
	    {
		case 0 :        /* "cw", move for n ticks */
		    if (((pmask & 4) == 0) || (time < 0) || (time > 3000))
			time = 100;
		    shutterJog(dp, &sd, CW, time);
		    break;

		case 1 :        /* "ccw", move for n ticks */
		    if (((pmask & 4) == 0) || (time < 0) || (time > 3000))
			time = 100;
		    shutterJog(dp, &sd, CCW, time);
		    break;

		case 2 :        /* "set", set closed position */
		    shutterOn(dp,&sd);
		    xprintf("Shutter #%d home set at %d\n",
				sd.ShutterNumber,(dp->drv_parms[PARM1] = readShutter(&sd)));
		    shutterOff(dp);
		    break;

		case 3 :        /* "open", open the shutter */
		    xprintf("Opening shutter #%d\n",sd.ShutterNumber);
		    /*runShutter(dp, TRUE, &sem);*/
		    shutRun(dp,TRUE);
		    break;

		case 4 :        /* "close", close the shutter */
		    xprintf("Closing shutter #%d\n",sd.ShutterNumber);
		    /*runShutter(dp, FALSE, &sem);*/
		    shutRun(dp,FALSE);
		    break;

		case 5 :        /* "where", read the position of the shutter */
		    shutterOn(dp,&sd);
		    xprintf("Shutter #%d position is %d  %s\n"
			,sd.ShutterNumber,readShutter(&sd),stateString[sector(&sd)]);
		    shutterOff(dp);

	    } /* end switch */

	return(OK); /* or whatever its called */

	} /* end if */
    } /* end for */

    xprintf("Bad shutter command name\n");
    return( ERROR );

}   /*  shutcom()  */

