/****************************************************************************/
/* Copyright 1997 MBARI                                                     */
/****************************************************************************/
/* $Header: shutcal.c,v 4.4 2001/06/19 12:15:32 oasisa Exp $                    */
/* Summary  : Interactive Routines for type 2 Shutters for PRR Spectro      */
/* Filename : shutcal10.c                                                   */
/* Author   : Dave Wright (dkw)                                             */
/* Project  : OASIS Mooring                                                 */
/* $Revision: 4.4 $                                                         */
/* Created  :  6/30/97                                                      */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:                                                    */
/* 3:43 PM 6/30/97 dkw - created                                            */
/* 7:01 PM 7/14/97 dkw - much further along                                 */
/* $Log:	shutcal.c,v $
 * Revision 4.4  2001/06/19  12:15:32  12:15:32  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:44:48  11:44:48  oasisa (Oasis users)
 * Initial revision
 * 
 * Revision 4.2  98/09/11  14:03:45  14:03:45  bobh (Bob Herlien)
 * Found bug testing at 38.4Kbaud, added task_delay
 * 
 * Revision 3.9  97/10/29  16:03:42  16:03:42  bobh (Bob Herlien)
 * EqPac Deployment of Nov 1997
 *                                              */
/*                                                                          */
/****************************************************************************/


/**************************************************************************** 

This module implements a simple, interactive means of testing and calibrating
the shutters (see shut22.c). It provides a continuous display of the shaft
position, the motor driver status, and the open, closed or in-between state
of the paddles. The display line looks something like this:

0375  Stopped    Closed

With the possible motor states being: Stopped | Running | Overload.
And the possible paddle states being: Open | Closed | Intermediate.

The command options are available by single keystroke as follows:

4    Start moving in CCW direction.

5    Stop moving.

6    Start moving in CW direction.

h    Set the current position as closed and update the drv_parms.

o    Call the shutter open routine with the current drv_parms.

c    Call the shutter close routine with the current drv_parms.

<cr> Exit.

It is intended that this module be uploaded to OASIS as needed for pre-deployment
testing and calibration of the shutters. As it presently exists, 6:05 PM 7/15/97
IT IS NOT SUITABLE FOR USE OVER THE RADIO LINK..! I may fix that in the future
when everything else is under control... dkw
****************************************************************************/


#include <types.h>
#include <const.h>
#include <oasis.h>
#include <task.h>
#include <io.h>
#include <custom.h>
/*
#include <tinyshut.h>
#undef stopMotor
#undef startMotor
#undef initMotor
#undef sector
*/
#include <shut22.h>



/********************************/
/*      External Functions      */
/********************************/

Extern  Int16   xgetc_tmout( Nat16 tmout  );
Extern  Void    xprintf( const char *format, ... );
Extern  Void    xflush_ser( Nat16 tmout );
Extern  Void    drv_pwron( Driver *dp );
Extern  Void    drv_pwroff( Driver *dp );
Extern  Driver *drvr_find( char *name );
Extern  Void    runShutter(Driver *dp );
Extern  Void    initMotor(Shutter_data *sd);

/********************************/
/*      External Data           */
/********************************/

Extern Reg Byte         ioctrl;                 /* Copy of oasis_ctrl               */
Extern Int16            ad_use_cnt;             /* Number of people using A/D       */

/********************************/
/*      Module Local Data       */
/********************************/

const char *  const stateString[]=      /* see shut22.h for enumeration of State type */
  {"Open","Closed","Inter"};

const char *  const motorString[]=      /* see shut22.h for enumeration of Motor type */
  {"Running","Stopped","Overload"};


/************************************************************************/
/* Function    : shutRun                                                */
/* Purpose     : handles power switching when calling runShutter        */
/* Inputs      : same parms as runShutter                               */
/* Outputs     : none                                                   */
/************************************************************************/
	Void
shutRun( Driver *dp, MBool open )
{
    xprintf("\nCalling runShutter...");
    dp->drv_usrparm = open;
    runShutter(dp);
    drv_pwron(dp);                      /* Turn on motor drive power */
    task_delay(20);
    xprintf("\nreturned from runShutter\n");
}

/************************************************************************/
/* Function    : shutCal                                                */
/* Purpose     : Interactive control of shutter                         */
/* Inputs      : Console data                                           */
/* Outputs     : Continuous to stdout, can change drv_parms[PARM1]      */
/************************************************************************/
	Int16
shutCal( Nat16 pmask, char *name )
{

    MBool  runstate=FALSE, quitflag=FALSE;
    Motor motorstate;           /* enumerated type for state definition */
    Driver *dp;                 /* driver structure defines pwr bits and home position */
    Shutter_data  sd;           /* structure used by all shutter functions for I/O addr resolution */

    if ( ((pmask & 1) == 0) ||
	 ((dp = drvr_find(name)) == DRV_NULL) )
    {
	xprintf("Bad shutter driver name\n");
	return( ERROR );
    }

    sd.ShutterNumber = dp->drv_parms[PARM0];
    sd.ClosePosition = dp->drv_parms[PARM1];

    initMotor(&sd);                     /* set up the I/O port */

    ad_use_cnt++;                       /* Show we're using analog voltage  */
    ioctrl &= ~ANALOG_DSBL;             /* Turn on analog power             */
    oasis_ctrl = ioctrl;
    drv_pwron(dp);                      /* Turn on motor drive power */

    xprintf("\n\nShutter #%d jog mode\n",sd.ShutterNumber);

    xflush_ser( TICKS_PER_SECOND/2 );   /* Wait 500 ms for pwr to stabilize */
    
    /* interactive loop starts here... */

    while(!quitflag)
    {
	switch(xgetc_tmout(0)) /* handle keypress, if any */
	{
	    case '5':   /* stop the shutter */
		stopMotor(&sd);
		runstate=FALSE;
		break;

	    case '4':   /* run shutter counterclockwise */
		if(readMotor(&sd) && readDirMotor(&sd))
		    stopMotor(&sd);     /* if already moving in opposite direction */
		setDirMotor(CCW,&sd);
		startMotor(&sd);
		runstate=TRUE;
		break;

	    case '6':   /* run shutter clockwise */
		if(readMotor(&sd) && !readDirMotor(&sd))
		    stopMotor(&sd);     /* if already moving in opposite direction */
		setDirMotor(CW,&sd);
		startMotor(&sd);
		runstate=TRUE;
		break;

	    case 'h':   /* mark current position as home */
		dp->drv_parms[PARM1] = sd.ClosePosition = readShutter(&sd);  
		break;          /* mark current position as home */

	    case 'c':   /* call closing routine */
		shutRun(dp, FALSE);
		break;

	    case 'o':   /* call opening routine */
		shutRun(dp, TRUE);
		break;

	    case '\r':  /* leave the interactive mode */
	    case '\n':
		quitflag=TRUE;
		break;

	    default:    /* this handles the no character case, we allow other tasks a chance */
		dispatch;

	} /* end switch */

	if(readMotor(&sd))              /* determine if we have hit anything */
	    motorstate = RUN;
	else
	    motorstate = (runstate)? OVERLOAD : STOP ;

	xprintf("%04d  %10s %10s\r",readShutter(&sd),motorString[motorstate],stateString[sector(&sd)]); /* output it  */
	task_delay( TICKS_PER_SECOND/10 );
	
    } /* end while() */
    xprintf("\n Leaving interactive mode, closed position is %d \n",dp->drv_parms[PARM1]);

    drv_pwroff(dp);                     /* Turn off motor drive power */

    if ( --ad_use_cnt <= 0)             /* No longer using analog voltage   */
    {                                   /* If no one else is,               */
	ioctrl |= ANALOG_DSBL;          /* Turn off analog power            */
	oasis_ctrl = ioctrl;
    }

    return( OK );

}  /*  shutCal()  */
