/****************************************************************************/
/* Copyright 1991 MBARI                                                     */
/****************************************************************************/
/* $Header: sensors.c,v 4.4 2001/06/19 12:15:22 oasisa Exp $		    */
/* Summary  : Driver Routines for Default Sensors for OASIS Mooring	    */
/* Filename : sensors.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.4 $							    */
/* Created  : 02/15/91							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 15feb91 rah - created						    */
/* $Log:	sensors.c,v $
 * Revision 4.4  2001/06/19  12:15:22  12:15:22  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:44:42  11:44:42  oasisa (Oasis users)
 * Initial revision
 * 
 * Revision 4.0  98/03/09  11:44:39  11:44:39  bobh (Bob Herlien)
 * M3 Deployment of March '98, new Sat-Pac driver
 * 
 * Revision 3.5  96/07/17  13:01:26  13:01:26  bobh (Bob Herlien)
 * July '96 deployment of M2 with ARGOS code
 * 
 * Revision 3.4  96/06/18  15:24:20  15:24:20  bobh (Bob Herlien)
 * June '96 deployment of M1
 * 
 * Revision 3.3  95/04/13  13:46:54  13:46:54  hebo (Bob Herlien)
 * Drifter Deployment for Coop (flip) cruise
 * 
 * Revision 3.2  95/04/11  14:03:26  14:03:26  hebo (Bob Herlien)
 * Drifter Deployment on IronEx
 * 
 * Revision 3.0  95/02/21  18:42:42  18:42:42  hebo (Bob Herlien)
 * February '95 Deployment
 * 
 * Revision 2.6  94/09/16  14:35:03  14:35:03  hebo (Bob Herlien)
 * Change CTD to use "TS" for opto-isolated CTD interface
 * 
 * Revision 2.5  94/01/11  17:41:10  17:41:10  hebo (Bob Herlien)
 * Fixed bug in "getlog", changed PRR format
 * 
 * Revision 2.4  93/10/29  18:48:16  18:48:16  hebo (Bob Herlien)
 * November 1993 Deployment
 * 
 * Revision 2.3  93/10/12  08:50:20  08:50:20  hebo (Bob Herlien)
 * Oct '93 Deployment of M2
 * 
 * Revision 2.0  92/08/26  10:32:18  10:32:18  hebo (Bob Herlien)
 * August 1992 Deployment
 * 
 * Revision 1.3  92/03/05  14:23:08  14:23:08  hebo (Bob Herlien 408-647-3748)
 * New defaults, restart check, perm power stuff, analog command, gps changes
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <custom.h>			/* ARGOS definition		    */
#include <80C196.h>			/* 80196 Register mapping           */
#include <task.h>			/* OASIS Multitasking definitions   */


/********************************/
/*	External Functions	*/
/********************************/

Extern Nat16	io_atod(Nat16 channel);
Extern Void	drvLogWords( Driver *dp, Nat16 *samplep, Int16 wrds );
Extern Void	drv_pwron( Driver *dp );
Extern Void	drv_pwroff( Driver *dp );
Extern Void	newline( Void );
Extern Void	bzero( void *s, int n );

#ifdef ARGOS
Extern Void	argosAnalogSample( Driver *dp, Nat16 *ad_sample );
#endif


/********************************/
/*	External Data		*/
/********************************/

Extern Reg Byte		ioctrl;		/* Copy of oasis_ctrl	 	    */


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal Int16	ad_use_cnt;		/* Number of people using A/D	    */


/************************************************************************/
/* Function    : sensor_init						*/
/* Purpose     : Do normal (every cycle) initialization of this module	*/
/* Inputs      : None							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
sensor_init( Void )
{
    ad_use_cnt = 0;			/* Nobody's using A/D yet	*/

} /* sensor_init() */


/************************************************************************/
/* Function    : get_analog						*/
/* Purpose     : Read analog value for one or more A/D ports		*/
/* Inputs      : A/D port, number of ports, number of samples to average*/
/*		 ticks to delay between samples, buffer to put results	*/
/* Output      : None							*/
/* Comments    : Used by analog_drv() and analog() user function	*/
/*		 Returns A/D values in array result[]			*/
/************************************************************************/
	Void
get_analog( Nat16 port, Nat16 nports, Nat16 samples, Nat16 *result )
{
    Reg Nat16	i;			/* Temps OK to destroy by task sw*/
    Nat16	cnt;			/* Temps we need between task switch*/
    Nat32	ad_accum[NAD_PORTS];	/* Accumulator for averaging	    */

    ad_use_cnt++;			/* Show we're using analog voltage  */
    ioctrl &= ~ANALOG_DSBL;		/* Turn on analog power		    */
    oasis_ctrl = ioctrl;
    task_delay( TICKS_PER_SECOND/2 );	/* Wait 500 ms for pwr to stabilize */

    bzero( ad_accum, sizeof(ad_accum) ); /* Zero accumulator buffer	    */
    if ( nports > NAD_PORTS )
	nports = NAD_PORTS;

    for ( cnt = 0; cnt < samples; cnt++ )
    {					/* Loop to read A/D channels	    */
	task_delay( 1 );		/* Delay between iterations	    */

	for ( i = 0; i < nports; i++ )
	    ad_accum[i] += io_atod(port +i); /* Read nports chans & accumulat*/
    }

    if ( --ad_use_cnt <= 0)		/* No longer using analog voltage   */
    {					/* If no one else is,		    */
	ioctrl |= ANALOG_DSBL;		/* Turn off analog power	    */
	oasis_ctrl = ioctrl;
    }

    for ( i = 0; i < nports; i++ )		/* Calculate average values */
	result[i] = (Nat16)(ad_accum[i] / samples);

} /* get_analog() */


/************************************************************************/
/* Function    : analog_drv						*/
/* Purpose     : Driver for General Purpose Analog Sensor		*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/* Comments    : Used for Pressure sensor, batteries, PARs		*/
/*		 Turns on power, waits for device to stabilize for	*/
/*		 TIMEOUT seconds.  Then reads PARM1 nmbr channels,	*/
/*		 starting at PARM0, and averages PARM2 readings, taken	*/
/*		 each tick.  Averages accumulated as 32 bits, then 	*/
/*		 divided by PARM2 and stored as 16 bits			*/
/************************************************************************/
	Void
analog_drv( Driver *dp )
{
    Nat16	nports;
    Nat16	ad_sample[NAD_PORTS];	/* A/D samples		    */

    drv_pwron( dp );			/* Turn on device		*/

    delay_secs( dp->drv_parms[TIMEOUT] );

    nports = dp->drv_parms[PARM1];

    get_analog(dp->drv_parms[PARM0], nports, dp->drv_parms[PARM2], ad_sample);

    drvLogWords( dp, ad_sample, nports );

    drv_pwroff( dp );			/* Turn off device		*/

#ifdef ARGOS
    argosAnalogSample( dp, ad_sample );
#endif

} /* analog_drv() */


/************************************************************************/
/* Function    : onoff_drv						*/
/* Purpose     : Simple driver to turn a device on and off		*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
onoff_drv( Driver *dp )
{
    drv_pwron( dp );			/* Turn on device		*/

    delay_secs ( dp->drv_parms[TIMEOUT] );

    drv_pwroff( dp );			/* Turn off device		*/

} /* onoff_drv() */


#ifdef ACOUSTIC_MODEM

/************************************************************************/
/* Function    : modem_wake						*/
/* Purpose     : Acoustic Modem serial wakeup function			*/ 
/* Inputs      : Driver ptr, Boolean (TRUE for on, FALSE for off)	*/
/* Outputs     : TRUE							*/
/************************************************************************/
	MBool
modem_wake( Driver *dp, MBool on )
{
    newline();				/* Send it a CR to keep it alive*/
    return( TRUE );

} /* modem_wake() */

#endif
