/****************************************************************************/
/* Copyright 1998 MBARI                                                     */
/****************************************************************************/
/* $Header: sat.c,v 4.5 2001/11/20 08:59:09 oasisa Exp $		    */
/* Summary  : Driver Routines for Satlantic S/N25 on OASIS mooring	    */
/* Filename : sat.c				        		    */
/* Author   : Danelle E. Cline						    */
/* Project  : OASIS M3 Mooring						    */
/* $Revision: 4.5 $							    */
/* Created  : 02/15/95							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 26feb98  Danelle C.- created						    */
/* $Log:	sat.c,v $
 * Revision 4.5  2001/11/20  08:59:09  08:59:09  oasisa (Oasis users)
 * Created SATPAC_OPTIONS in custom.h (originally for SOFEX to manipulate -l an d -n log options)
 * 
 * Revision 4.4  2001/06/19  12:15:13  12:15:13  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:44:36  11:44:36  oasisa (Oasis users)
 * Initial revision
 * 
 * Revision 4.0  98/03/09  11:44:41  11:44:41  bobh (Bob Herlien)
 * M3 Deployment of March '98, new Sat-Pac driver
 * 
* 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions   	    */
#include <const.h>			/* MBARI constants	            */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions            */
#include <log.h>			/* Log record definitions           */
#include <custom.h>			/* CTD_TS definition	            */
#include <80C196.h>			/* 80196 Register mapping           */
#include <task.h>			/* OASIS Multitasking definitions   */
#include <stdio.h>			/* Standard I/O functions	    */
#include <sat.h>
#include <ctype.h>

/*************************************************************************/
/*					External Functions		 */
/*************************************************************************/
Extern Reg DateTime	dttm;		/* Current date & time in DateTime  */
Extern Void	drvLogWords(Driver *dp, Word *samplep, Int16 len );
Extern Void	drvLogError( Driver *dp, Errno err );
Extern char	*drvSerPortAndMalloc( Driver *dp, Nat16 size );
Extern Void	drvSerReleaseAndFree( Driver *dp, char *buffer );
Extern Void	xputs( const char *s );
Extern Int16	xgets_tmout( char *s, Int16 len, Nat16 tmout );
Extern Int16	xgetc_tmout(Nat16 tmout);
Extern char 	*find_str( char *src, char *tgt );
Extern Void	tmpFree( char *ptr );
Extern MBool	delimit( Reg char c );	
Extern Void	bzero( void *s, int n );	
Extern	Void	xprintf( const char *format, ... );	
Extern Void	xflush_ser( Nat16 timeout );
Extern MBool	getnum( char **s, Int16 *result, Nat16 radix );
Extern Void	drv_pwron( Driver *dp );
Extern Void	drv_pwroff( Driver *dp );
Extern char	*skipField( Reg char *p, Nat16 nfields );

/*************************************************************************/
/*						Driver local data								 */
/*************************************************************************/
MLocal Nat16 sample[NUM_CHAN_LOG];          /*holds raw a/d cnt sample   */
	

/************************************************************************/
/* Function    : getSatPrompt						*/
/* Purpose     : Set date/time and get prompt				*/
/* Inputs      : Timeout in seconds					*/
/* Outputs     : OK or ERROR						*/
/************************************************************************/
Int16 getSatPrompt( Nat16 tmout )
{
    Reg Int16 c;
    
    xputs("date\r");                                  /*send date command */

    while ( (c = xgetc_tmout(tmout)) != '?' ) /*Look for SAT-DAT response */
	if ( c == ERROR )
            return( ERROR );
		/*set date, marker for end of data w/global dttm struct*/
    delay_secs(1);
    xprintf("%02d/%02d/%02d %02d:%02d:%02d\r",dttm.dt_mo,dttm.dt_day,
             dttm.dt_yr, dttm.dt_hr,dttm.dt_min,dttm.dt_sec);             
    while ( (c = xgetc_tmout(tmout)) != '>' ) /*Look for SAT-DAT response */
        if ( c == ERROR )                                        
            return( ERROR );
    delay_secs(1);
    return( OK );
}


/************************************************************************/
/* Function    : SatlanticDrv					 	*/
/* Purpose     : Implements data log and acqusition algorithm		*/
/* Inputs      : Driver ptr						*/
/* Outputs     : none               					*/
/************************************************************************/
Void SatlanticDrv(Driver *dp)
{
char *satbuf;
char *timedate;		
Nat16 samples;
Nat16 ntrys;
satbuf = NULL;


    if( (satbuf = drvSerPortAndMalloc(dp, BUFF_SIZE)) == NULL)
	return;

/*Initialize */
    bzero(satbuf,BUFF_SIZE);
    bzero(sample, NUM_CHAN_LOG);                        
    samples = 0;
    ntrys = 0;

    delay_secs(4);                                    /*delay for power up*/
					   /*should be at PicoDos> prompt */		
    if ( getSatPrompt(dp->drv_parms[TIMEOUT]) == OK )
    {	

	bzero(satbuf, BUFF_SIZE);                              /*Clear buffer */ 
	xprintf("log -a%d ", dp->drv_parms[PARM0]);
	task_delay(TICKS_PER_SECOND/2);
	xprintf("-o0123456789");
	#ifdef SATPAC_OPTIONS
	task_delay(TICKS_PER_SECOND/2);
	xprintf(" %s\r",SATPAC_OPTIONS); /* originally " -l\r" */
	#endif

     /*timeout at least length of averaging period, add 5 secs for safety */ 
	while((samples < dp->drv_parms[PARM2]) && (ntrys < dp->drv_parms[PARM1]))
	{
	    samples = GetSamples(satbuf, timedate, dp->drv_parms[PARM0] + 5); 
	    ntrys++;
	}
    }

    if(samples > 0)
	drvLogWords(dp, sample, samples );
    else
	drvLogError(dp, SYNC_ERR);
	
    drvSerReleaseAndFree( dp, satbuf );		/* Release serial port & buffer	*/
    drv_pwroff(dp);	
}

/************************************************************************/
/* Function    : GetSamples                                             */
/* Purpose     : Reads serial port, and calls ParseSamples              */
/* Inputs      : Ptr to data buf, timedate ptr for update, and timeout	*/ 
/* Outputs     : number of channels sampled				*/
/************************************************************************/
Nat16 GetSamples(char *buf, char *timedate, Nat16 timeout)
{
Nat16 samples;

	samples = 0;
	xflush_ser(TICKS_PER_SECOND/2);
	if(xgets_tmout(buf, BUFF_SIZE, timeout) > 0)	
		samples = ParseSamples(buf, timedate);											
	return(samples);
}


/************************************************************************/
/* Function    : ParseSamples                                           */
/* Purpose     : Parses NUM_CHAN_LOG, 0 - 2^15 cnt ASCII data           */
/*                               into samplebuf.                        */
/* Inputs      : Ptr to data buf, timedate ptr for update               */ 
/* Outputs     : number of channels sampled				*/
/************************************************************************/
Nat16 ParseSamples(char *buff, char *timedate)
{
    Reg char *p;
    char *s;
    Nat16 numsamples;
    Nat16 j;

/*Initialize */
    p = buff;
    timedate = NULL;

    if(find_str(p, "Collect") != NULL)                    /*in schedular mode */
	return(0);

    if(find_str(p, "Hibernate") != NULL ||
       find_str(p, "Current") != NULL)                    /*in schedular mode */
    {
	s = skipField(s , 5);
        if(*s != '9' && *s++ != '8')                 /*valid date 98-XX-XX*/
		timedate = NULL;
	else
		timedate = s--;
	return(0);
    }

    if(find_str(p,"No")  != NULL)     /*In the middle of breaking log command*/
	return(0);

    if((s = find_str(p, "log")) != NULL)      /*echoed log command, skip over */
    {
	for(j=0;j<4;j++)
	    s = skipField(s,1);
	p = s;
    }

    for ( numsamples = 0; (p != NULL) && (numsamples < NUM_CHAN_LOG); )
    {			
	j = 0;
        if(*p == '9')	/*Found date stamp, end of sample */
            break;

	if(getnum(&p,(Int16 *) &j,10) != ERROR)
        {
            *(sample + numsamples) = j;
	    numsamples++;
	    p = skipField(p,1);              
        }       
    }

    return(numsamples);
}
