/************************************************************************/
/* Copyright 1994 MBARI							*/
/************************************************************************/
/* $Header: log.h,v 4.5 2001/10/17 10:33:55 oasisa Exp $			*/
/* Summary  : Log record types for the OASIS mooring controller		*/
/* Filename : log.h							*/
/* Author   : Robert Herlien (rah)					*/
/* Project  : OASIS Mooring						*/
/* $Revision: 4.5 $							*/
/* Created  : 12/15/94 from oasis.h					*/
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/************************************************************************/
/* Modification History:						*/
/* 15dec94 rah - created from similar def'ns in oasis.h			*/
/* $Log:	log.h,v $
 * Revision 4.5  2001/10/17  10:33:55  10:33:55  oasisa (Oasis users)
 * updated with new HRij channel data types
 * 
 * Revision 4.4  2001/06/19  12:14:29  12:14:29  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:44:09  11:44:09  oasisa (Oasis users)
 * Initial revision
 * 
 * Revision 4.3  99/06/16  10:21:24  10:21:24  bobh (Bob Herlien)
 * Mar/May '99 Deployments of M3/M2
 * 
 * Revision 4.2  98/09/09  10:47:55  10:47:55  bobh (Bob Herlien)
 * Sept/Oct '98 deployments of M1, Eqpac 1 & 2
 * 
 * Revision 3.8  97/09/12  10:50:39  10:50:39  bobh (Bob Herlien)
 * Redeploy M1
 * 
 * Revision 3.7  97/07/23  11:17:53  11:17:53  bobh (Bob Herlien)
 * July '97 M1 deployment, new shutter code
 * 
 * Revision 3.5  96/07/17  13:01:13  13:01:13  bobh (Bob Herlien)
 * July '96 deployment of M2 with ARGOS code
 * 
 * Revision 3.4  96/06/18  15:24:03  15:24:03  bobh (Bob Herlien)
 * June '96 deployment of M1
 * 
 * Revision 3.2  95/04/11  14:03:17  14:03:17  hebo (Bob Herlien)
 * Drifter Deployment on IronEx
 * 
 * Revision 3.1  95/03/09  19:30:48  19:30:48  hebo (Bob Herlien)
 * March '95 Deployment of M1A
 * 
 * Revision 3.0  95/02/21  18:42:32  18:42:32  hebo (Bob Herlien)
 * February '95 Deployment
 * 
*/
/************************************************************************/

#ifndef INClogh
#define INClogh		1

/****************************************/
/* Log Memory layout			*/
/****************************************/

#define LOGRAM		0xe000		/* Start of Logging RAM		    */
#define LOG_RAMSIZE	0x2000		/* Size of each RAM bank	    */
#define BANKS		128		/* Number of RAM banks		    */


/****************************************/
/*	Data Logging			*/
/****************************************/

/* Log memory is divided into Log Blocks.  Each log block is a power of two */
/* number of memory banks, and must have a total size <= 64Kbytes.  If a    */
/* disk is present, disk blocks are the same size as log memory blocks.	    */
/* The max disk capacity is 64K blocks * 64K bytes = 4 GB.		    */
/* Memory addresses within a block are represented as being contiguous, even*/
/* though the memory is in fact banked.					    */
/* Each log block, in memory or on disk, starts with a LogRecNum indicating */
/* the number of records in the block, and is followed by the records.	    */

typedef Nat16	LogBlk;			/* Block number, on disk or memory  */
typedef Nat16	LogAddr;		/* Address within block		    */
typedef Nat16	LogRecNum;		/* Record number		    */

typedef struct				/************************************/
{					/* Struct to remember where we are  */
    LogBlk	lp_blk;			/* Actual Block number		    */
    LogBlk	lp_memblk;		/* Memory block it's in		    */
    LogAddr	lp_addr;		/* Address within block		    */
    LogRecNum	lp_rcd;			/* Record number		    */
} LogPtr;				/************************************/

typedef enum				/************************************/
{					/* Return code for isLogged	    */
    LOG_OK = 0,				/* Record is in memory		    */
    LOG_BAD_BLK,			/* Bad Block number		    */
    LOG_BAD_RCD				/* Record number is too high	    */
} LogRtn;				/************************************/


/************************************************************************/
/* Defines and Macros for moving around in log memory and on disk	*/
/************************************************************************/

#define BLKSIZE		(16 * 1024)	/* Change these two defns to change */
#define BLK_SHFT	14		/*  the log block size		    */
#define ADDR_SHFT	13		/* Change with LOG_RAMSIZE	    */
#define MAX_BLKNUM	0xffff		/* LogBlk is a word		    */

#ifdef DISK
#define DISK_BLK	1		/* Memory block used as disk buffer */
#define MEMBLK_OFFSET	2		/* 1st used by permMem, 2nd for disk*/
#else
#define MEMBLK_OFFSET	1		/* 1st block used for perm memory   */
#endif

#define LOG_START_ADDR	sizeof(LogRecNum)	/* Log starts with #rcds    */
#define ADDR_MASK	(LOG_RAMSIZE - 1)	/* LogAddr -> bank addr mask*/
#define BANK_SHFT	(BLK_SHFT - ADDR_SHFT)	/* LogAddr -> bank number   */
#define MEMBLKS		(BANKS >> BANK_SHFT)	/* Total blocks in bankd RAM*/
#define MEMBLKS_USED	(MEMBLKS - MEMBLK_OFFSET) /* Nmbr blks used for logs*/

#define MemBlk(blk)	((blk % MEMBLKS_USED) + MEMBLK_OFFSET)
#define	Addr(addr)	((Byte *)((addr & ADDR_MASK) | LOGRAM))
#define Bank(memblk,addr) ((memblk << BANK_SHFT) + (addr >> ADDR_SHFT))


/****************************************/
/*	Log Record Definitions		*/
/****************************************/

typedef struct				/************************************/
{					/* LogRecHdr - Logging Record Header*/
    Byte	log_syncc;		/* Sync chr - used internally by log*/
    Byte	log_type;		/* Record type - see below	    */
    LogRecNum	log_rcd;		/* Record number		    */
    Int16	log_len;		/* Number of bytes in log_data	    */
    TimeOfDay	log_time;		/* Record time			    */
} LogRecHdr;				/************************************/

typedef struct				/************************************/
{					/* LogRec - Logging Record	    */
    LogRecHdr	log_hdr;		/* Log Header			    */
    Byte	log_data[MAXLOGSIZE];	/* Log Data			    */
} LogRec;				/************************************/

					/************************************/
#define LOG_SYNC	0x55		/* Sync char to ensure start of rec */
					/************************************/

/*	Data Logging Record Types	*/
#define LOG_EMPTY	0		/* Unused - empty record	    */
#define OASIS_STAT	1		/* OASIS on/off status -  1 word    */
#define ATLAS		2		/* atlas record     - 32 words	    */
#define OASIS_CAN	3		/* OASIS battery, temp, press: 4 wrds*/
#define PAR		4		/* PAR sensors       - 2 words	    */
#define CTD		5		/* CTD Data          - variable	    */
#define SPECTRO		6		/* Spectroradiometer - variable	    */
#define ADCP		7		/* Acoustic Doppler  - variable	    */
#define GPS		8		/* GPS				    */
#define MODEM		9		/* Acoustic Modem		    */
#define PCO2		10		/* pCO2 sensor			    */
#define CTD2		11		/* Second CTD			    */
#define CTD3		12		/* Third CTD			    */
#define SPECTRO2	13		/* Second Spectroradiometer	    */
#define SPECTRO3	14		/* Third Spectroradiometer	    */
#define NO3		15		/* Nitrate sensor		    */
#define NO32		16		/* Second Nitrate sensor	    */
#define SPEC_PRR	17		/* PRR type Spectroradiometer	    */
#define SATLANTIC	18		/* Satlantic Spectroradiometer	    */
#define GPS_TYPE2	19		/* New GPS, with velocity	    */
#define NRL		20		/* NRL next download time	    */
#define O2		21		/* Oxygen sensor		    */
#define FLUOR		22		/* Fluorometer			    */
#define TRANSMISS	23		/* Transmissometer		    */
#define AC9		26		/* WetLabs AC-9			    */
#define CO2_PUMP	27		/* pCO2 Pump			    */
#define H2O_PUMP	28		/* Water Pump			    */
#define SHUTTER0	29		/* Shutter 0 for PRR Spectro	    */
#define SHUTTER1	30		/* Shutter 1 for PRR Spectro	    */
#define SPEC_PRR_VOLTS	31		/* Unused here, for decode program  */
#define METSYS		32		/* Campbell Scientific Met Logger   */
#define TSTRING		33		/* Seabird Inductive Temp String    */
#define GNDFAULT	34		/* Serial Analog Ground Fault	    */
#define MICROCAT	35		/* Seabird serial microCat	    */
#define GPS_TYPE3	36		/* New GPS, with velocity	    */
#define HOBI_AB		37		/* HOBI Labs alpha-beta		    */
#define HOBI_CB		38		/* HOBI Labs c-beta		    */
#define HOBI_HS2	39		/* HOBI Labs HydroScat 2	    */
#define HOBI_HR1	40		/* HOBI Labs HydroRad file 1	    */
#define HOBI_HR2	41		/* HOBI Labs HydroRad file 2	    */
#define HOBI_HR3	42		/* HOBI Labs HydroRad file 3	    */
#define HOBI_HR4	43		/* HOBI Labs HydroRad file 4	    */
#define HOBI_DAS        44              /* HOBI Labs HydroDAS file          */
#define GASHOUND        45              /* GasHound pCO2 Sensor (TT8B)	    */
#define ISUS            46		/* ISUS Nitrate Analyzer (TT8C)     */
#define EMETER          47		/* Energy Meter			    */
#define SBE47_CT        48              /* Seabird low power CT             */
#define SBE43_O2        49              /* Seabird Oxygen                   */
#define HOBI_HR11	50		/* HOBI Labs 1 ch HydroRad ch 1     */
#define HOBI_HR21	51		/* HOBI Labs 2 ch HydroRad ch 1	    */
#define HOBI_HR22	52		/* HOBI Labs 2 ch HydroRad ch 2	    */
#define HOBI_HR31	53		/* HOBI Labs 3 ch HydroRad ch 1	    */
#define HOBI_HR32	54		/* HOBI Labs 3 ch HydroRad ch 2	    */
#define HOBI_HR33	55		/* HOBI Labs 3 ch HydroRad ch 3	    */
#define HOBI_HR41	56		/* HOBI Labs 4 ch HydroRad ch 1	    */
#define HOBI_HR42	57		/* HOBI Labs 4 ch HydroRad ch 2	    */
#define HOBI_HR43	58		/* HOBI Labs 4 ch HydroRad ch 3	    */
#define HOBI_HR44	59		/* HOBI Labs 4 ch HydroRad ch 4	    */
#define SPECIAL1	60		/* For temporary or test instruments*/
#define SPECIAL2	61		/* For temporary or test instruments*/
					/************************************/
#define LOG_ERROR	128		/* Error record			    */
#define LOG_COMMENT	129		/* User comments		    */
#define LOG_BIN		130		/* Misc Binary data		    */
#define LOG_DISK_ERROR	131		/* Disk Error, 1 word of error reg  */
					/************************************/
/* An Error record contains 1 word.  If the MSB of the word is clear, it    */
/* identifes a sensor type, as above.  If the MSB is set, it identifies	    */
/* one or more OASIS errors, as defined in oasis.h (e.g., RAM_ERR, etc)	    */

#endif	/* INClogh */
