/****************************************************************************/
/* Copyright 2000 MBARI                                                     */
/****************************************************************************/
/* $Header: isus.c,v 4.4 2001/06/19 12:14:21 oasisa Exp $								    */
/* Summary  : Driver Routines for ISUS Nitrate Anaylyzer on OASIS mooring   */
/*            This is for an instrument based the TattleTale 8 (TT8) serial */
/*            data acquisition unit.                                        */
/* Filename : isus.c							    */
/* Author   : Kent Headley (klh)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.4 $								    */
/* Created  : 06/19/00 from no3.c					    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 19jun00 klh - created from no3.c					    */
/* 
 * $Log:	isus.c,v $
 * Revision 4.4  2001/06/19  12:14:21  12:14:21  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:44:04  11:44:04  oasisa (Oasis users)
 * Initial revision
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <custom.h>			/* Deployment-specific definitions  */
#include <task.h>			/* OASIS Multitasking definitions   */
#include <argos.h>			/* ARGOS definitions		    */
#include <stdio.h>			/* Standard I/O functions	    */
#include <ctype.h>			/* Standard ctype.h defs	    */

#define NO3_BUFSIZE	1536		/* Size of buffer for nitrate	    */
#define ISUS_BUFSIZE    1536
					/* Different ways of waking the     */
					/* instrument			    */
#undef USE_WAKESTR
#define USE_WAKEBREAK


/********************************/
/*	External Functions	*/
/********************************/

Extern Void	drvLog( Driver *dp, Byte *samplep, Int16 len );
Extern Void	drvLogError( Driver *dp, Errno err );
Extern char	*drvSerPortAndMalloc( Driver *dp, Nat16 size );
Extern Void	drvSerReleaseAndFree( Driver *dp, char *buffer );
Extern Void	xputs( const char *s );
Extern Void     xputc( Int16 c );
Extern Void	xflush_ser( Nat16 tmout );
Extern Int16	xgets_tmout( char *s, Int16 len, Nat16 tmout );
Extern char 	*find_str( char *src, char *tgt );
Extern Void	tmpFree( char *ptr );
Extern Void     ser_break(Nat16 port, Int16 count );
Extern MBool	getnum( char **s, Int16 *result, Nat16 radix );
Extern Int16	xgetn_tmout( char *s, Int16 len, Nat16 tmout );
Extern Void     bcopy(const Byte *src, Byte *dst, Nat16 len);
#ifdef ARGOS_ISUS
Extern Void	argosIsusSample( Driver *dp, Nat16 l1n, Byte *data );
#endif


/********************************/
/*	External Data		*/
/********************************/


/********************************/
/*	Module Local Data	*/
/********************************/



/************************************************************************/
/* Function    : wakeup_isus						*/
/* Purpose     : Wake up the NO3 analyzer, pass wakeup string		*/
/* Inputs      : Driver ptr, wakeup string, buffer to use		*/
/* Outputs     : TRUE if woke up nitrate, else FALSE			*/
/************************************************************************/
	MBool
wakeup_isus( Driver *dp, char *wakestr, char *buffer )
{
    Nat16	i;
#ifdef USE_WAKESTR
    for ( i = dp->drv_parms[PARM0]; i; i-- )
    {
	xputs(wakestr);			/* Wake up nitrate sensor	*/

	while ( xgets_tmout(buffer, 30, dp->drv_parms[TIMEOUT]) >= 0 )
	    if ( find_str(buffer, "ACK") != NULL )
	    {					/* Look for signon message*/
		xflush_ser( TICKS_PER_SECOND );	/* Flush serial buffer	  */
		return( TRUE );			/* Got it, return TRUE	  */
	    }
    }

    return( FALSE );
#endif /*USE_WAKESTR*/

#ifdef USE_WAKEBREAK
    ser_break(dp->drv_parms[SER_PORT], TICKS_PER_SECOND/2);

    for ( i = dp->drv_parms[PARM0]; i; i-- )
    {
      if(dp->drv_parms[PARM2]>0)
	task_delay((dp->drv_parms[PARM2]*(TICKS_PER_SECOND/10)));
      xputc('W');
      while ( xgets_tmout(buffer, 30, dp->drv_parms[TIMEOUT]) >= 0 )
	if ( find_str(buffer, "ACK") != NULL ){	/* Look for ACK message	    */
	  xflush_ser( TICKS_PER_SECOND );	/* Flush serial buffer	    */
	  return( TRUE );			/* Got it, return TRUE	    */
	}
   }
   return( FALSE );
#endif /*USE_WAKEBREAK*/   

} /* wakeup_isus() */


/************************************************************************/
/* Function    : isus_wake						*/
/* Purpose     : Nitrate serial wakeup function				*/ 
/* Inputs      : Driver ptr, Boolean (TRUE for on, FALSE for off)	*/
/* Outputs     : TRUE							*/
/************************************************************************/
	MBool
isus_wake( Driver *dp, MBool on, char *buffer )
{
  if ( on )
    wakeup_isus( dp, "UUU\r", buffer );

  return (TRUE);

} /* isus_wake() */



/************************************************************************/
/* Function    : isus_drv						*/
/* Purpose     : Nitrate Sensor driver					*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/* PARMS       : 0: Wakeup retries (sends break, then PARM0 wakeup chars*/
/*		 1: <0:0>=Sync At Midnight <1:15>=Max records to store 	*/
/*		    (Discards others)					*/
/*		 2: Time to delay between break and wakeup chars (0.1 s)*/
/************************************************************************/
	Void
isus_drv( Driver *dp )
{
    Nat16	bufLen;
    char	*isus_buf,**pp;		/* Buffer for nitrate data	*/
    Reg Nat16	curLen;
    Int16       nRecs;
    Nat16       recLen,i,maxRecs;
    Nat16       ISUS_REC_WIDTH=sizeof(recLen);
    MBool       gotisus;

#ifdef SYNC_AT_MIDNIGHT
    if ( dp->drv_flags & DO_ARGOS )
    {
	if ( (dp->drv_parms[PARM1] & 0x1) == 0 )
	    return;
    }
#endif
    
    if ( (isus_buf = drvSerPortAndMalloc(dp, ISUS_BUFSIZE)) == NULL )
	return;

    if ( wakeup_isus(dp, "UUU\r", isus_buf) )
    {					/* Wake up NO3, ask for data	*/
       xputs("R");			/* Request Number of Samples    */

       if((bufLen=xgets_tmout(isus_buf, 16, dp->drv_parms[TIMEOUT])) > 0 ){
       
	 /*isus_buf[bufLen]='\0';*/

       nRecs = 0;
       maxRecs=(dp->drv_parms[PARM1]>>1);
       pp = &isus_buf;
       if (getnum(pp,&nRecs,10)==FALSE)
	 nRecs=0;
       gotisus=FALSE;

       for(i=0;i<nRecs;i++){
         recLen=curLen=bufLen=0;
         xflush_ser(TICKS_PER_SECOND/4);
         xputs( "D" );				/* Send data request string	*/
						/* Get ASCII  data */
         bufLen=xgets_tmout(isus_buf,ISUS_BUFSIZE,dp->drv_parms[TIMEOUT]);
	 
						/* Log it			*/
	 if ( (bufLen > 0)  ){
	   if(i < maxRecs){
#ifdef ARGOS_ISUS
	     argosIsusSample( dp, bufLen, (Byte *)&isus_buf[0] );
#endif
	     drvLog( dp, (Byte *)isus_buf, bufLen );
	     gotisus=TRUE;
	   }
        }else
	if ( bufLen == 0 ){
	  drvLogError( dp, BAD_DATA );
#ifdef ARGOS_ISUS
	  argosIsusSample( dp, 7, (Byte *)"BADDATA" );
#endif
	}else{
	  drvLogError( dp, NO_DATA );		/* Else no wakeup		*/
#ifdef ARGOS_ISUS
	  argosIsusSample( dp, 7, (Byte *)"NO_DATA" );
#endif
	}           
	 task_delay(TICKS_PER_SECOND/4);
       }/*end while nRec */       
       }
    }/* end if wakeup */
    else
      drvLogError( dp, SYNC_ERR );		/* Else no wakeup	*/
    
    drvSerReleaseAndFree( dp,isus_buf );	/* Release serial port & buf*/

} /* isus_drv() */
