/****************************************************************************/
/* Copyright 2001 MBARI                                                     */
/****************************************************************************/
/* $Header: hydrorad.c,v 1.1 2001/10/17 10:29:21 oasisa Exp $		    */
/* Summary  : Driver Routines for HobiLabs HydroRADs                        */
/* Filename : hydrorad.c   				         	    */
/* Author   : Kent Headley (klh)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 1.1 $  							    */
/* Created  : 08/16/2001						    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 16aug01 klh - created						    */
/* $Log:	hydrorad.c,v $
 * Revision 1.1  2001/10/17  10:29:21  10:29:21  oasisa (Oasis users)
 * Initial revision
 * 
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <custom.h>			/* CTD_TS definition		    */
#include <80C196.h>			/* 80196 Register mapping           */
#include <task.h>			/* OASIS Multitasking definitions   */
#include <stdio.h>			/* Standard I/O functions	    */
#include <string.h>			/* String library functions	    */
#include <ctype.h>			/* Standard character types	    */
#include <hobi.h>			/* Hobi defines			    */

#define CTRL_C		3		/* Control-C character		    */
#define HOBI_BUFSIZE	1500
#define HOBI_WAKETIME	1
#define HOBI_CDELAY	1
#define HOBI_FLUSHTIME	(TICKS_PER_SECOND/2)

/********************************/
/*	External Data		*/
/********************************/
Extern Reg DateTime	dttm;		/* Current date & time in DateTime  */
Extern Reg TimeOfDay	tod;		/* Current time in TimeOfDay format */
Extern volatile Reg Nat16 tick;		/* 10 ms ticker			    */

/********************************/
/*	External Functions	*/
/********************************/
Extern Void	drvLog( Driver *dp, Byte *samplep, Int16 len );
Extern Void	drvLogError( Driver *dp, Errno err );
Extern Void	drv_pwroff( Driver *dp );
Extern char	*drvSerPortAndMalloc( Driver *dp, Nat16 size );
Extern Void	drvSerReleaseAndFree( Driver *dp, char *buffer );
Extern Void	xputc( Int16 c );
Extern Void	xputs( const char *s );
Extern Void	xprintf( const char *format, ... );
Extern Int16	xgets_tmout( char *s, Int16 len, Nat16 tmout );
Extern Int16	xgetn_tmout( char *s, Int16 len, Nat16 tmout );
Extern Void	xflush_ser( Nat16 tmout );
Extern Void	logPutRec( LogRecHdr *hdrp, Byte *dp );
Extern char     *find_str( char *src, char *tgt );
Extern Void     delay_secs( Nat16 secs );

/************************************************************************/
/* Function    : wakeup_hydrorad					*/
/* Purpose     : Wakeup HOBI Instrument					*/
/* Inputs      : Driver ptr, buffer					*/
/* Outputs     : TRUE if woke, else FALSE				*/
/************************************************************************/
	MBool
wakeup_hydrorad( Driver *dp, char *buffer )
{
    Int16	i;
    Nat16	st_tick;
    
    for ( i = dp->drv_parms[PARM0]; i; i-- )
    {
	xputc( CTRL_C );
	st_tick = tick;
	if ( xgets_tmout(buffer, HOBI_BUFSIZE, HOBI_WAKETIME) > 0 )
	{
	    deblank(buffer);
	    /*	    if ( strcmp(buffer, "Ready") == 0 )*/
	    if ( find_str(buffer, "Ready")!= NULL )
		return( TRUE );
	}

	xflush_ser( TICKS_PER_SECOND - (tick - st_tick) );
    }
    
    return( FALSE );

} /* wakeup_hobi() */


/************************************************************************/
/* Function    : hydrorad_drv						*/
/* Purpose     : Driver for HOBI Labs HydroRads 			*/
/*
 This driver is for stand-alone hydrorads; Hydrorads are normally 
power-permed, but have failed to wake up at times. This driver assumes
that the hydrorad will not be power-permed; consequently, the driver
must reset the date on the hydrorad at wakeup and wait for data collec-
tion to finish before exiting.   
*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
hydrorad_drv( Driver *dp )
{
    Reg Int16	hobiLen;
    char	*hobiBuf;
    Int16	cnt,i;
    Int16 hobiTypes[5][4]={
      {0,0,0,0},
      {HOBI_HR11,0,0,0},
      {HOBI_HR21,HOBI_HR22,0,0},
      {HOBI_HR31,HOBI_HR32,HOBI_HR33,0},
      {HOBI_HR41,HOBI_HR42,HOBI_HR43,HOBI_HR44}
    };
    Nat16	tmout;
    LogRecHdr	rh;

    if ( (hobiBuf = drvSerPortAndMalloc(dp, HOBI_BUFSIZE)) == NULL )
    {
	drvLogError( dp, MALLOC_ERR );		/* No malloc space	*/
	return;
    }
    
    if ( wakeup_hydrorad(dp, hobiBuf) )
    {						/* Got mem and wakeup	*/
						/* Set Date and Time    */
      xprintf("date %02d/%02d/%02d %02d:%02d:%02d\r",
	      dttm.dt_mo, dttm.dt_day,dttm.dt_yr,
	      dttm.dt_hr, dttm.dt_min, dttm.dt_sec );

	xflush_ser( HOBI_FLUSHTIME );
	tmout = dp->drv_parms[TIMEOUT];

	for ( cnt = 0; cnt < dp->drv_parms[PARM1]; cnt++ )
	{					/* HydroRad code	*/
	    xprintf("SEND%d\r", cnt+1);
	    rh.log_len = xgetn_tmout(hobiBuf, HOBI_BUFSIZE, tmout);
	    rh.log_type = hobiTypes[dp->drv_parms[PARM1]][cnt];
	    logPutRec( &rh, (Byte *)hobiBuf );

	    for(i=0;i<3;i++){
	      xputc( CTRL_C );
	      task_delay(TICKS_PER_SECOND/2);
	      xputc( CTRL_C );
	      xgetn_tmout(hobiBuf, HOBI_BUFSIZE, HOBI_WAKETIME);
	      hobiBuf[HOBI_BUFSIZE-1]='\0';
	      if ( find_str(hobiBuf, "HydroRad>")!= NULL )
		break;
	    }
	    xflush_ser( HOBI_FLUSHTIME );
	}
	xputs("DEL DATA*.BIN\r");
	xflush_ser( HOBI_FLUSHTIME );
	xputs("Y\r");
	xflush_ser( HOBI_FLUSHTIME );
	xputs("COLLECT\r");
	dp->drv_cnt=0;
	while(dp->drv_cnt < dp->drv_parms[PARM2]){
						/* wait... */
	  delay_secs(HOBI_CDELAY);
						/* get string       */
	  while( xgets_tmout(hobiBuf, HOBI_BUFSIZE, HOBI_WAKETIME) > 0 )
	  {
	    deblank(hobiBuf);
						/* quit if sleeping */
	    if ( find_str(hobiBuf, "Sleeping")!= NULL )
	      dp->drv_cnt=dp->drv_parms[PARM2];
	  }
	  dp->drv_cnt += HOBI_CDELAY;
	}/*end while */
    }
    else{
	drvLogError( dp, SYNC_ERR );		/* No wakeup		*/
    }
    drvSerReleaseAndFree( dp, hobiBuf );	/* Release serial port	*/
    drv_pwroff( dp );				/* Turn off power	*/

} /* hobi_drv() */




