/****************************************************************************/
/* Copyright 2001 MBARI                                                     */
/****************************************************************************/
/* $Header: hs2.c,v 1.1 2001/10/17 10:30:15 oasisa Exp $		    */
/* Summary  : Driver Routines for Hobilabs HS2 on OASIS mooring		    */
/* Filename : hydroscat.c	       					    */
/* Author   : Kent Headley (klh)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 1.1 $							    */
/* Created  : 08/22/2001						    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 08aug01 klh - created						    */
/* $Log:	hs2.c,v $
 * Revision 1.1  2001/10/17  10:30:15  10:30:15  oasisa (Oasis users)
 * Initial revision
 * 
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <custom.h>			/* CTD_TS definition		    */
#include <80C196.h>			/* 80196 Register mapping           */
#include <task.h>			/* OASIS Multitasking definitions   */
#include <stdio.h>			/* Standard I/O functions	    */
#include <string.h>			/* String library functions	    */
#include <ctype.h>			/* Standard character types	    */
#include <hobi.h>			/* Hobi defines			    */

#define CTRL_C		3		/* Control-C character		    */
#define HOBI_BUFSIZE	500		/* Specified max is 1340	    */
#define HOBI_WAKETIME	1
#define HOBI_CDELAY	1
#define HOBI_FLUSHTIME	(TICKS_PER_SECOND/2)
#define GETH_CMD	"H\r"

/********************************/
/*	External Functions	*/
/********************************/

Extern Void	drvLog( Driver *dp, Byte *samplep, Int16 len );
Extern Void	drvLogError( Driver *dp, Errno err );
Extern Void	drv_pwroff( Driver *dp );
Extern char	*drvSerPortAndMalloc( Driver *dp, Nat16 size );
Extern Void	drvSerReleaseAndFree( Driver *dp, char *buffer );
Extern Void	xputc( Int16 c );
Extern Void	xputs( const char *s );
Extern Void	xprintf( const char *format, ... );
Extern Int16	xgets_tmout( char *s, Int16 len, Nat16 tmout );
Extern Int16	xgetn_tmout( char *s, Int16 len, Nat16 tmout );
Extern Void	xflush_ser( Nat16 tmout );
Extern Void	logPutRec( LogRecHdr *hdrp, Byte *dp );
Extern char     *find_str( char *src, char *tgt );
Extern Void     delay_secs( Nat16 secs );
#ifdef ARGOS_HS2
Extern Void argosHS2Sample( Driver *dp, Nat16 len, Byte *data );
#endif

/********************************/
/*	External Data   	*/
/********************************/

Extern Reg DateTime	dttm;		/* Current date & time in DateTime  */
Extern Reg TimeOfDay	tod;		/* Current time in TimeOfDay format */
Extern volatile Reg Nat16 tick;		/* 10 ms ticker			    */


/************************************************************************/
/* Function    : wakeup_hydroscat					*/
/* Purpose     : Wakeup HOBI Instrument					*/
/* Inputs      : Driver ptr, buffer					*/
/* Outputs     : TRUE if woke, else FALSE				*/
/************************************************************************/
	MBool
wakeup_hydroscat( Driver *dp, char *buffer, Nat16 len )
{
    Int16	i;
    Nat16	st_tick;
    
    for ( i = dp->drv_parms[PARM0]; i; i-- )
    {
	xputc( CTRL_C );
	st_tick = tick;
	if ( xgets_tmout(buffer, len, HOBI_WAKETIME) > 0 )
	{
	    deblank(buffer);
	    if ( find_str(buffer, "Ready")!= NULL )
		return( TRUE );
	}

	xflush_ser( TICKS_PER_SECOND - (tick - st_tick) );
    }
    
    return( FALSE );

} /* wakeup_hydroscat() */


/************************************************************************/
/* Function    : hydroscat_drv						*/
/* 
Purpose     : Driver for HOBI Labs HS2 (stand alone, powered by
OASIS). This driver made due to reliability problem with hobi
instruments: Normally, they are self powered, but sometimes they don't
wake up. It turns out to be safer to power them up for each sample,
and cut power after the sample is taken.
*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
hydroscat_drv( Driver *dp )
{
    Reg Int16	hobiLen;
    char	*hobiBuf,*cp;
    Int16	cnt;
    Nat16	tmout;
    LogRecHdr	rh;
    Nat16       newchars,newlen;

    if ( (hobiBuf = drvSerPortAndMalloc(dp, HOBI_BUFSIZE)) == NULL )
    {
	drvLogError( dp, MALLOC_ERR );		/* No malloc space	*/
	return;
    }
    
    if ( wakeup_hydroscat(dp, hobiBuf,HOBI_BUFSIZE)==TRUE )
    {						/* Got mem and wakeup	*/
						/* Set Date and Time    */
      xflush_ser( HOBI_FLUSHTIME );

      xprintf("date,%02d/%02d/%02d %02d:%02d:%02d\r",
	      dttm.dt_mo, dttm.dt_day,dttm.dt_yr,
	      dttm.dt_hr, dttm.dt_min, dttm.dt_sec );

      xflush_ser( HOBI_FLUSHTIME );
						/* Send Sampling command */
      xputs("START\r");

      tmout = dp->drv_parms[TIMEOUT];
      dp->drv_cnt=0;
      cp = hobiBuf;
      hobiLen=0;
      memset(hobiBuf,'\0',HOBI_BUFSIZE);

      /* Read until SLEEP received or TIMEOUT */
      while( dp->drv_cnt < dp->drv_parms[PARM2] ){
	/* Get Data */
	if(hobiLen<HOBI_BUFSIZE){
	  newchars=xgetn_tmout(cp, HOBI_BUFSIZE-hobiLen, tmout);
	  newlen=hobiLen+newchars;
	  if( (newchars > 0) && (newlen<HOBI_BUFSIZE) ){
	    cp=hobiBuf+newlen;
	    *cp='\0';
	    hobiLen=strlen(hobiBuf);
	  }
	}

	/* Quit if SLEEP */
	if((find_str(hobiBuf,"Sleeping") != NULL))
	  break;

	/* Be a good citizen */
	delay_secs(HOBI_CDELAY);

	/* Increment TIMEOUT */
	dp->drv_cnt+=HOBI_CDELAY;

      }/* end if while !SLEEP or TIMEOUT...*/

      /* terminate data and log it */
      hobiBuf[HOBI_BUFSIZE-1]='\0';
      cp=find_str(hobiBuf,"Burst completed");/* case insensitive; cp->end of target */
      if(cp!=NULL){
	while(!(*cp=='\r' || *cp=='\n' || *cp=='\0'))cp++;
	*cp='\0';
	hobiLen=strlen(hobiBuf);

	/* Now get H Packet */
	if( HOBI_BUFSIZE>hobiLen )
	  if( wakeup_hydroscat(dp,cp,(HOBI_BUFSIZE-hobiLen))==TRUE ){
	    xflush_ser( HOBI_FLUSHTIME );
	    *cp='\r';
	    *(++cp)='\0';
	    xputs(GETH_CMD);
	    xgets_tmout(cp,HOBI_BUFSIZE-hobiLen,tmout);
	    hobiLen=strlen(hobiBuf);
	  }
	/* get rid of extra stuff at beginning of buffer */
	/* NOT TESTED
	for(cp=hobiBuf; (*cp != '*')  && (cp<hobiBuf+hobiLen);cp++ )(
	  *cp=' ';
	  hobiLen--;
	}
	deblank(hobiBuf);
	*/
	/* Log data */
	drvLog( dp, (Byte *)hobiBuf, hobiLen );
#ifdef ARGOS_HS2
	argosHS2Sample( dp, hobiLen, (Byte *)&hobiBuf[0] );
#endif
      }/* end LOG DATA */
      else{
	drvLogError(dp,NO_DATA);
#ifdef ARGOS_HS2
	argosHS2Sample( dp, 7, (Byte *)"NO_DATA" );
#endif
      }/* end NO_DATA */

    }/* end if WAKEUP */
    else{
	drvLogError( dp, SYNC_ERR );		/* No wakeup		*/
#ifdef ARGOS_HS2
	argosHS2Sample( dp, 8, (Byte *)"SYNC_ERR" );
#endif
    }/* end NO WAKEUP*/

    drvSerReleaseAndFree( dp, hobiBuf );	/* Release serial port	*/
    drv_pwroff( dp );				/* Turn off power	*/

} /* hydroscat_drv() */

