/****************************************************************************/
/* Copyright 1999 MBARI                                                     */
/****************************************************************************/
/* $Header: hobi.c,v 4.4 2001/06/19 12:14:04 oasisa Exp $			    */
/* Summary  : Driver Routines for Seabird serial microCat on OASIS mooring  */
/* Filename : hobi.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.4 $							    */
/* Created  : 06/21/99							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 21jun99 rah - created						    */
/* $Log:	hobi.c,v $
 * Revision 4.4  2001/06/19  12:14:04  12:14:04  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:43:52  11:43:52  oasisa (Oasis users)
 * Initial revision
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <custom.h>			/* CTD_TS definition		    */
#include <80C196.h>			/* 80196 Register mapping           */
#include <task.h>			/* OASIS Multitasking definitions   */
#include <stdio.h>			/* Standard I/O functions	    */
#include <string.h>			/* String library functions	    */
#include <ctype.h>			/* Standard character types	    */
#include <hobi.h>			/* Hobi defines			    */

#define CTRL_C		3		/* Control-C character		    */
#define HOBI_BUFSIZE	1500		/* Specified max is 1340	    */
#define HOBI_WAKETIME	1
#define HOBI_FLUSHTIME	(TICKS_PER_SECOND/2)


/********************************/
/*	External Functions	*/
/********************************/

Extern Void	drvLog( Driver *dp, Byte *samplep, Int16 len );
Extern Void	drvLogError( Driver *dp, Errno err );
Extern Void	drv_pwroff( Driver *dp );
Extern char	*drvSerPortAndMalloc( Driver *dp, Nat16 size );
Extern Void	drvSerReleaseAndFree( Driver *dp, char *buffer );
Extern Void	xputc( Int16 c );
Extern Void	xputs( const char *s );
Extern Void	xprintf( const char *format, ... );
Extern Int16	xgets_tmout( char *s, Int16 len, Nat16 tmout );
Extern Int16	xgetn_tmout( char *s, Int16 len, Nat16 tmout );
Extern Void	xflush_ser( Nat16 tmout );
Extern Void	logPutRec( LogRecHdr *hdrp, Byte *dp );
Extern char     *find_str( char *src, char *tgt );
#ifdef ARGOS_HS2
Extern Void argosHS2Sample( Driver *dp, Nat16 len, Byte *data );
#endif

/********************************/
/*	External Functions	*/
/********************************/

Extern Reg TimeOfDay	tod;		/* Current time in TimeOfDay format */
Extern volatile Reg Nat16 tick;		/* 10 ms ticker			    */


/************************************************************************/
/* Function    : wakeup_hobi						*/
/* Purpose     : Wakeup HOBI Instrument					*/
/* Inputs      : Driver ptr, buffer					*/
/* Outputs     : TRUE if woke, else FALSE				*/
/************************************************************************/
	MBool
wakeup_hobi( Driver *dp, char *buffer )
{
    Int16	i;
    Nat16	st_tick;
    
    for ( i = dp->drv_parms[PARM0]; i; i-- )
    {
	xputc( CTRL_C );
	st_tick = tick;
	if ( xgets_tmout(buffer, HOBI_BUFSIZE, HOBI_WAKETIME) > 0 )
	{
	    deblank(buffer);
	    /*	    if ( strcmp(buffer, "Ready") == 0 )*/
	    if ( find_str(buffer, "Ready")!= NULL )
		return( TRUE );
	}

	xflush_ser( TICKS_PER_SECOND - (tick - st_tick) );
    }
    
    return( FALSE );

} /* wakeup_hobi() */


/************************************************************************/
/* Function    : hobi_drv						*/
/* Purpose     : Driver for HOBI Labs AB, CB, and HS2			*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
hobi_drv( Driver *dp )
{
    Reg Int16	hobiLen;
    char	*hobiBuf;
    Int16	cnt;
    Nat16	tmout;
    LogRecHdr	rh;

    if ( (hobiBuf = drvSerPortAndMalloc(dp, HOBI_BUFSIZE)) == NULL )
    {
	drvLogError( dp, MALLOC_ERR );		/* No malloc space	*/
	return;
    }
    
    if ( wakeup_hobi(dp, hobiBuf) )
    {						/* Got mem and wakeup	*/
	xflush_ser( HOBI_FLUSHTIME );
	tmout = dp->drv_parms[TIMEOUT];

	for ( cnt = 0; cnt < dp->drv_parms[PARM1]; cnt++ )
	{					/* HydroRad code	*/
	    xprintf("SEND%d\r", cnt+1);
	    rh.log_len = xgetn_tmout(hobiBuf, HOBI_BUFSIZE, tmout);
	    rh.log_type = dp->drv_parms[SAMPLE_TYPE] + cnt;
	    logPutRec( &rh, (Byte *)hobiBuf );
	}

	xputs("COLLECT\r");

	if ( cnt == 0 )				/* ABeta, CBeta, HS2 code*/
	{
	    hobiLen = xgetn_tmout(hobiBuf, HOBI_BUFSIZE, tmout);
#ifdef ARGOS_HS2
	     argosHS2Sample( dp, hobiLen, (Byte *)&hobiBuf[0] );
#endif
	    drvLog( dp, (Byte *)hobiBuf, hobiLen );
	}
    }
    else{
	drvLogError( dp, SYNC_ERR );		/* No wakeup		*/
#ifdef ARGOS_HS2
	argosHS2Sample( dp, 8, (Byte *)"SYNC_ERR" );
#endif
    }
    drvSerReleaseAndFree( dp, hobiBuf );	/* Release serial port	*/
    drv_pwroff( dp );				/* Turn off power	*/

} /* hobi_drv() */

