/****************************************************************************/
/* Copyright 2001 MBARI                                                     */
/****************************************************************************/
/* $Header: emeter.c,v 4.5 2001/10/17 10:27:41 oasisa Exp $	       	    */
/* Summary  : Driver Routines for xantrex emeter battery monitor	    */
/* Filename : emeter.c							    */
/* Author   : Kent Headley (klh)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.5 $							    */
/* Created  : 06/1/01							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 1jun01 klh - created						    */
/* $Log:	emeter.c,v $
 * Revision 4.5  2001/10/17  10:27:41  10:27:41  oasisa (Oasis users)
 * Cruising Equipment Power Meter Driver (klh)
 * 
 * Revision 4.4  2001/06/19  12:13:31  12:13:31  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:43:25  11:43:25  oasisa (Oasis users)
 * Initial revision
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <custom.h>			/* CTD_TS definition		    */
#include <80C196.h>			/* 80196 Register mapping           */
#include <task.h>			/* OASIS Multitasking definitions   */
#include <stdio.h>			/* Standard I/O functions	    */
#include <string.h>			/* String library functions	    */
#include <ctype.h>			/* Standard character types	    */

#define EMETER_BUFSIZE	256		/* Specified max is 1340	    */
#define EMETER_WAKETIME	3
#define EMETER_FLUSHTIME	(TICKS_PER_SECOND/2)

/********************************/
/*	External Functions	*/
/********************************/

Extern Void	drvLog( Driver *dp, Byte *samplep, Int16 len );
Extern Void	drvLogError( Driver *dp, Errno err );
Extern Void	drv_pwroff( Driver *dp );
Extern char	*drvSerPortAndMalloc( Driver *dp, Nat16 size );
Extern Void	drvSerReleaseAndFree( Driver *dp, char *buffer );
Extern Void	xputc( Int16 c );
Extern Void	xputs( const char *s );
Extern Void	xprintf( const char *format, ... );
Extern Int16	xgets_tmout( char *s, Int16 len, Nat16 tmout );
Extern Int16	xgetn_tmout( char *s, Int16 len, Nat16 tmout );
Extern Void	xflush_ser( Nat16 tmout );
Extern Void	logPutRec( LogRecHdr *hdrp, Byte *dp );
Extern char     *find_str( char *src, char *tgt );


/********************************/
/*	External Functions	*/
/********************************/

Extern Reg TimeOfDay	tod;		/* Current time in TimeOfDay format */
Extern volatile Reg Nat16 tick;		/* 10 ms ticker			    */


/************************************************************************/
/* Function    : wakeup_emeter						*/
/* Purpose     : Wakeup emeter					*/
/* Inputs      : Driver ptr, buffer					*/
/* Outputs     : TRUE if woke, else FALSE				*/
/************************************************************************/
	MBool
wakeup_emeter( Driver *dp, char *buffer )
{
    Int16	i;
    Nat16	st_tick;
    
    for ( i = dp->drv_parms[PARM0]; i; i-- )
    {
      if ( xgets_tmout(buffer, EMETER_BUFSIZE, EMETER_WAKETIME) > 0 )
	{
	  return( TRUE );
	}

	xflush_ser( TICKS_PER_SECOND - (tick - st_tick) );
    }
    
    return( FALSE );

} /* wakeup_emeter() */


/************************************************************************/
/* Function    : emeter_drv						*/
/* Purpose     : Driver for xantrex emeter battery monitor		*/
/* Inputs      : Driver Pointer						*/
/* PARMs       : 
           TIMEOUT: seconds to wait on serial read
                 0: wakeup retries
                 1: read retries
                 2: unused
 */
/* Outputs     : None							*/
/************************************************************************/
	Void
emeter_drv( Driver *dp )
{
    Reg Int16	emeterLen=0;
    char	*emeterBuf;
    Int16	cnt;
    Nat16	tmout,i;

    if ( (emeterBuf = drvSerPortAndMalloc(dp, EMETER_BUFSIZE)) == NULL )
    {
	drvLogError( dp, MALLOC_ERR );		/* No malloc space	*/
	return;
    }
    
       if ( wakeup_emeter(dp, emeterBuf)==TRUE )
      {
    						/* Got mem and wakeup	*/
	tmout = dp->drv_parms[TIMEOUT];
	for ( i = dp->drv_parms[PARM1]; (i>0 && emeterLen<=0) ; i-- ){
	  xgets_tmout(emeterBuf, EMETER_BUFSIZE, tmout);
	  if( xgets_tmout(emeterBuf, EMETER_BUFSIZE, tmout)>=0)
	    {
	      emeterBuf[EMETER_BUFSIZE-1]='\0';
	      emeterLen=strlen(emeterBuf);
	      drvLog( dp,(Byte *)emeterBuf,emeterLen );
	      break;
	    }else{
	      drvLogError(dp,NO_DATA);
	    }
	}
      }else{
	drvLogError( dp, SYNC_ERR );		/* No wakeup		*/
      }

    drvSerReleaseAndFree( dp, emeterBuf );	/* Release serial port	*/
    drv_pwroff( dp );				/* Turn off power	*/

} /* emeter_drv() */




