/****************************************************************************/
/* Copyright 1995 MBARI                                                     */
/****************************************************************************/
/* $Header: ctd.c,v 4.5 2001/10/17 10:26:17 oasisa Exp $			    */
/* Summary  : Driver Routines for Seabird CTD on OASIS mooring		    */
/* Filename : ctd.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.5 $							    */
/* Created  : 02/15/95 from sensors.c					    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 15feb95 rah - created from sensors.c					    */
/* $Log:	ctd.c,v $
 * Revision 4.5  2001/10/17  10:26:17  10:26:17  oasisa (Oasis users)
 * update ctd buffer size (SBE firmware change
 * 
 * Revision 4.4  2001/06/19  12:13:07  12:13:07  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:42:50  11:42:50  oasisa (Oasis users)
 * Initial revision
 * 
 * Revision 4.2  98/09/09  10:48:02  10:48:02  bobh (Bob Herlien)
 * Sept/Oct '98 deployments of M1, Eqpac 1 & 2
 * 
 * Revision 3.7  97/07/23  11:18:09  11:18:09  bobh (Bob Herlien)
 * July '97 M1 deployment, new shutter code
 * 
 * Revision 3.4  96/06/18  15:24:23  15:24:23  bobh (Bob Herlien)
 * June '96 deployment of M1
 * 
 * Revision 3.1  95/03/09  19:30:58  19:30:58  hebo (Bob Herlien)
 * March '95 Deployment of M1A
 * 
 * Revision 3.0  95/02/21  18:42:44  18:42:44  hebo (Bob Herlien)
 * February '95 Deployment
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <custom.h>			/* CTD_TS definition		    */
#include <80C196.h>			/* 80196 Register mapping           */
#include <task.h>			/* OASIS Multitasking definitions   */
#include <stdio.h>			/* Standard I/O functions	    */

#define CTD_BUFSIZE	32		/* Leave a little extra room for CTD*/


/********************************/
/*	External Functions	*/
/********************************/

Extern Void	drvLog( Driver *dp, Byte *samplep, Int16 len );
Extern Void	drv_ser_port( Driver *dp );
Extern Void	drv_ser_release( Driver *dp );
Extern Void	drv_pwroff( Driver *dp );
Extern Void	xputc( Int16 c );
Extern Void	xputs( const char *s );
Extern Int16	xgetc_tmout( Nat16 tmout );
Extern Int16	xgets_tmout( char *s, Int16 len, Nat16 tmout );
Extern Int16	getByte( char *p, Nat16 radix );
Extern Int16	getCtdPrompt( Nat16 timeout );
#ifdef PUMP_SYNC_CTD
Extern Void	drvSync( char *name );
#endif

#ifdef QUUX
/************************************************************************/
/* Function    : getCtdPrompt						*/
/* Purpose     : Look for CTD prompt					*/
/* Inputs      : Timeout in seconds					*/
/* Outputs     : OK or ERROR						*/
/************************************************************************/
	Int16
getCtdPrompt( Nat16 timeout )
{
    Reg Int16	c;
    Int16	i;
    
    for (i = 10; i && ((c = xgetc_tmout(timeout)) != ERROR); i--)
	if ( c == '>' )			/* Look for prompt		*/
	    return( OK );		/* If got it, return OK		*/

    return( ERROR );			/* If timed out, return ERROR	*/

} /* getCtdPrompt() */
#endif

/************************************************************************/
/* Function    : get_ctd						*/
/* Purpose     : Make one attempt to get CTD data			*/
/* Inputs      : Buffer, Driver pointer					*/
/* Outputs     : Number of bytes received from CTD			*/
/************************************************************************/
	Nat16
get_ctd( char *ctd_buf, Driver *dp )
{
    Reg Int16	i, c;			/* Loop cntr & char buffer	*/
    Reg char	*p;			/* Buffer ptr			*/
    Nat16	tmout;			/* Timeout			*/

    xputc('\r');			/* Get CTD's attention		*/
    tmout = dp->drv_parms[TIMEOUT];	/* Get timeout			*/

    if ( getCtdPrompt(tmout) != OK )	/* Look for prompt		*/
	return( 0 );			/* If timed out, return no data	*/

#if CTD_TS
    xputs("TS\r");			/* Take Sample command		*/
    tmout = dp->drv_parms[PARM2];
#else
    xputs("SS\r");			/* Send Sample command		*/
#endif

    while ( xgets_tmout(ctd_buf, CTD_BUFSIZE, tmout) >= 0 )
    {
	for ( p = ctd_buf, i = 0; *p; p++ )
	    if ( (c = getByte(p, 16)) != ERROR )
	    {				/* Convert ASCII to hex in place*/
		ctd_buf[i++] = c;	/* If ASCII, increment ptr by 2 */
		p++;			/*  and continue converting	*/
	    }	 			/* If not hex, we might still be*/
					/*  getting CTD prompt		*/
	if ( i > 0 )			/* If got some hex bytes,	*/
	    return( i );		/* Return bytes we got		*/
    }

    return( 0 );			/* If timed out, return 0	*/

} /* get_ctd() */


/************************************************************************/
/* Function    : ctd_drv						*/
/* Purpose     : CTD driver						*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
ctd_drv( Driver *dp )
{
    Int16	i, size, len;
    char	ctd_buf[CTD_BUFSIZE];	/* Buffer for CTD data		*/
    
    drv_ser_port( dp );			/* Get serial port		*/

#ifdef PUMP_SYNC_CTD
    drvSync( PUMP_SYNC_CTD );		/* Wait for water pump		*/
#endif

    size = dp->drv_parms[PARM0];	/* Find number bytes to look for*/
    if ( size > CTD_BUFSIZE )		/* If more than buffer, truncate*/
	size = CTD_BUFSIZE;

    for ( i = dp->drv_parms[PARM1]; 
	  i && (len = get_ctd(ctd_buf, dp)) < size; i-- )
	;

    drv_ser_release( dp );		/* Release serial port		*/
    drvLog( dp, (Byte *)ctd_buf, len );	/* Log CTD data			*/
    drv_pwroff( dp );			/* Turn off power		*/

} /* ctd_drv() */
