/****************************************************************************/
/* Copyright 1996 MBARI                                                     */
/****************************************************************************/
/* $Header: argosbcn.c,v 4.4 2001/06/19 12:12:27 oasisa Exp $		    */
/* Summary  : Driver Routines for Argos satellite interface on OASIS mooring*/
/* Filename : argosBeacon.c						    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.4 $							    */
/* Created  : 04/22/96							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 15jun98 rah - created from argos.c					    */
/* $Log:	argosbcn.c,v $
 * Revision 4.4  2001/06/19  12:12:27  12:12:27  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:42:30  11:42:30  oasisa (Oasis users)
 * Initial revision
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <custom.h>			/* ARGOS_BIT definition		    */
#define NUM_ARGOS_BUFS	1
#include <argos.h>			/* ARGOS buffer definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <task.h>			/* OASIS Multitasking definitions   */
#include <gps10.h>			/* Magellan GPS definitions         */

#define ACK		0x06
#define NACK		0x15
#define NULL_CMD	0xb0
#define STORE1_CMD	0x10		/* PTT index num OR'd into low nibble*/
#define STORE2_CMD	0x20
#define ST_XMIT1_CMD	0x30
#define ST_XMIT2_CMD	0x40
#define XMIT1_CMD	0x50
#define XMIT2_CMD	0x60

#define OASIS_ANALOG_DONE 0x8000
#define ARGOS_GPS_DONE	  0x4000
#define RANDOM_DIVISOR	3121		/* Yields random nmbr between 0 & 20*/
					/* For 9-21, make this 180	    */

/********************************/
/*	External Data		*/
/********************************/

Extern Reg Byte		io_wakebits;	/* Copy of wakeup bits		    */
Extern Reg Word		error;		/* Error vector			    */
Extern Reg DateTime	dttm;		/* Current date & time in DateTime  */
Extern Reg Byte		localHour;	/* Hour of day (0 - 23) in local time*/
Extern Nat16		deploymentMode;	/* In deployment mode mode	    */


/********************************/
/*	External Functions	*/
/********************************/

Extern Void	drv_ser_port( Driver *dp );
Extern Void	drv_ser_release( Driver *dp );
Extern Void	drv_pwroff( Driver *dp );
Extern Void	xputc( Int16 c );
Extern Void	xputn( const char *s, Int16 len );
Extern Int16	xgetc_tmout( Nat16 tmout );
Extern Void	xflush_ser( Nat16 tmout  );
Extern Void	bcopy( const Byte *src, Byte *dst, Nat16 len );
Extern Void	bzero( void *s, int n );	/* Zero memory		    */
Extern Nat16	swap( Nat16 w );
Extern Nat16	random( Void );
Extern Void	calcArgosChksums( ArgosBeaconMsg *dp, Nat16 numBufs );


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal ArgosBeaconMsg	newData;	/* Data we're still collecting  */
MLocal ArgosBeaconMsg	sendData;	/* Data to send			*/

MLocal Nat16		argosDataState;


/************************************************************************/
/* Function    : copyArgosBufs						*/
/* Purpose     : Copy newData into sendData buffer, calc chksums	*/
/* Inputs      : None							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
copyArgosBufs( Void )
{
    bcopy( (Byte *)&newData, (Byte *)&sendData, sizeof(sendData) );
    calcArgosChksums( &sendData, NUM_ARGOS_BUFS );

} /* copyArgosBufs() */


/************************************************************************/
/* Function    : swapArgosBufs						*/
/* Purpose     : Copy newData into sendData buffer, clear old data	*/
/* Inputs      : None							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
swapArgosBufs( Void )
{
    copyArgosBufs();
    bzero( (void *)&newData, sizeof(newData) );
    argosDataState = 0;

} /* swapArgosBufs() */


/************************************************************************/
/* Function    : wakeup_argos						*/
/* Purpose     : Wake up ARGOS PTT					*/ 
/* Inputs      : None							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
wakeup_argos( Void )
{
    io_wakebits |= ARGOS_BIT;			/* Wake up the PTT	*/
    wakeport = io_wakebits;			/* Send to hardware	*/
    task_delay( 4 );				/* Leave it on 30-40 ms	*/
    io_wakebits &= ~ARGOS_BIT;			/* Turn off wakeup bit	*/
    wakeport = io_wakebits;			/* Send to hardware	*/
    xflush_ser( 1 );				/* Throw away any input	 */

} /* wakeup_argos() */


/************************************************************************/
/* Function    : argos_wake						*/
/* Purpose     : ARGOS serial wakeup function				*/ 
/* Inputs      : Driver ptr, Boolean (TRUE for on, FALSE for off)	*/
/* Outputs     : TRUE							*/
/************************************************************************/
	MBool
argos_wake( Driver *dp, MBool on, char *buffer )
{
    if ( on )
	wakeup_argos();

    return( TRUE );

} /* argos_wake() */


/************************************************************************/
/* Function    : send_ptt						*/
/* Purpose     : Wake up PTT, send one buffer				*/
/* Inputs      : Driver ptr						*/
/* Outputs     : TRUE if successful, else FALSE				*/
/************************************************************************/
	MBool
send_ptt( Driver *dp )
{
    Reg Int16	ch;				/* Scratch char buffer	*/
    Reg Nat16	bufNum, firstArgosId, numArgosIds;

    wakeup_argos();				/* Wake up argos PTT	*/

    firstArgosId = dp->drv_parms[PARM1];
    numArgosIds = dp->drv_parms[PARM2];

    xputc( ST_XMIT1_CMD | 0 );

    xputn( (char *)&sendData, ARGOS_LEN );
    ch = xgetc_tmout( dp->drv_parms[TIMEOUT] );	/* Get response		 */
    return( ch == ACK );			/* Check for ACK	 */

} /* send_ptt() */


/************************************************************************/
/* Function    : argos_drv						*/
/* Purpose     : Driver to send data to Argos PTT			*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
argosBeacon_drv( Driver *dp )
{
    Nat16	i;

    if ( deploymentMode )
	copyArgosBufs();

    drv_ser_port( dp );			/* Setup serial port		*/

    for ( i = dp->drv_parms[PARM0]; (i > 0) && (!send_ptt(dp)); i-- )
	delay_secs( dp->drv_parms[TIMEOUT] );

    drv_ser_release( dp );		/* Release serial port		*/

    if ( i <= 0 )			/* If retried out, set error	*/
	error |= ARGOS_ERR;

    dp->drv_wakeup += ((Int16)(random()/RANDOM_DIVISOR) - 10);

} /* argosBeacon_drv() */


/************************************************************************/
/* Function    : argosAnalogSample					*/
/* Purpose     : Put Analog sample into ARGOS buffer			*/
/* Inputs      : Driver pointer, A/D sample pointer			*/
/* Outputs     : None							*/
/************************************************************************/
	Void
argosAnalogSample( Driver *dp, Nat16 *ad_sample )
{
    Reg Nat16	sampleNum;

    switch( dp->drv_parms[SAMPLE_TYPE] )
    {
      case OASIS_CAN:
	if ( (localHour >= 12) && ((argosDataState & OASIS_ANALOG_DONE) == 0) )
	{
	    newData.ab_oasisCan[0] = ad_sample[0];
	    newData.ab_oasisCan[1] = ad_sample[1];
	    newData.ab_oasisCan[2] = ad_sample[4];
	    argosDataState |= OASIS_ANALOG_DONE;
	}
    }

} /* argosAnalogSample() */


/************************************************************************/
/* Function    : argosOasisError					*/
/* Purpose     : OR oasis error into argos message			*/
/* Inputs      : None							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
argosOasisError( Void )
{
    newData.ab_oasisErrs |= error;

} /* argosOasisError() */



/************************************************************************/
/* Function    : argosAnalogSample					*/
/* Purpose     : Put Analog sample into ARGOS buffer			*/
/* Inputs      : Driver pointer, A/D sample pointer			*/
/* Outputs     : None							*/
/************************************************************************/
	Void
argosGPSSample( Driver *dp, GpsRec *gpsPtr )
{
    if ( (localHour >= 12) && ((argosDataState & ARGOS_GPS_DONE) == 0) )
    {
	newData.ab_gpsLat = gpsPtr->gps_lat;
	newData.ab_gpsLon = gpsPtr->gps_long;
	argosDataState |= ARGOS_GPS_DONE;
    }

} /* argosGPSSample() */
