/************************************************************************/
/* Copyright 1997 MBARI							*/
/************************************************************************/
/* $Header: argos.c,v 3.9 2002/06/10 14:39:54 oasisa Exp $			*/
/* Summary  : Program to analyze ARGOS receiver data			*/
/* Filename : argos.c							*/
/* Author   : Bob Herlien (rah)						*/
/* Project  : OASIS							*/
/* $Revision: 3.9 $							    */
/* Created  : 07/15/96							*/
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/************************************************************************/
/* Modification History:						*/
/* 15jul96 rah - created						*/
/* $Log:	argos.c,v $
 * Revision 3.9  2002/06/10  14:39:54  14:39:54  oasisa (Oasis users)
 * fix argos mail input for win32
 * fix hr3 byte order for unix
 * 
 * Revision 3.8  2002/04/23  12:01:42  12:01:42  oasisa (Oasis users)
 * Support for HS2, HR3
 * 
 * Revision 3.7  2002/03/01  15:38:43  15:38:43  oasisa (Oasis users)
 * Support for Local receiver input, WIN32
 * 
 * Revision 3.6  2001/08/30  15:23:05  15:23:05  oasisa (Oasis users)
 * Base Dir mods for Win32
 * clean unused variables
 * cast ulong to short to clear warning
 * 
 * Revision 3.4  2001/05/04  12:57:35  12:57:35  oasisa (Oasis users)
 * change bcopy() to memmove() for WIN32 compatibility
 * 
 * Revision 3.3  2001/05/02  10:52:17  10:52:17  oasisa (Oasis users)
 * New ISUS format.
 * 
 * Revision 3.2  2001/04/25  09:09:30  09:09:30  oasisa (Oasis users)
 * Y2K fix, PTT filtering for local rcvr messages, HS2 stream processing
 * 
 * Revision 3.0  99/05/12  10:11:29  10:11:29  bobh (Bob Herlien)
 * Added tstring, misc changes
 * 
 * Revision 2.9  98/08/24  13:45:58  13:45:58  bobh (Bob Herlien)
 * Archiving sources after M2/M3 & Eqpac deployments of 1998
 * 
 * Revision 2.8  98/03/17  11:11:44  11:11:44  bobh (Bob Herlien)
 * Archiving sources prior to porting to DOS/Windows
 * 
 * Revision 1.1  97/10/27  09:53:18  09:53:18  bobh (Bob Herlien)
 * Initial revision
 * 
*/
/************************************************************************/

#include <stdio.h>			/* Unix standard I/O		*/
#include <stdlib.h>			/* Unix standard library	*/
#include <mbariTypes.h>			/* MBARI standard types		*/
#include <mbariConst.h>			/* MBARI standard constants	*/
#include <time.h>			/* Unix time library		*/
#include <decode.h>			/* OASIS data definitions	*/
#include <string.h>			/* Unix string library		*/
#include <argos.h>			/* ARGOS buffer definitions	*/

#define SPEC_VOLTS	FALSE
#define NUM_MSGS	100		/* Max messages to store & compare */
#define BUFSIZE		1024		/* Line buffer size		   */
#define FLUOR_CHAN	0		/* Analog channel for Wetstar fluor*/
#define TEMP_CHAN	2		/* Analog channel for temperature  */
#define OBATT_CHAN	3		/* Analog channel for OASIS battery*/
#define ABATT_CHAN	6		/* Analog channel for ARGOS battery*/
#define YEAR0		70
#define MAX_PTTS        8

typedef enum				/********************************/
{					/* Return value from getXXXMsg	*/
    MSG_OK,				/* Successful conversion	*/
    BAD_MSG,				/* Bad message data		*/
    END_OF_FILE				/* End of input file		*/
} MsgRtn;				/********************************/

typedef enum				/********************************/
{					/* Type of input		*/
    SVC_ARGOS,				/* From Serivce Argos		*/
    RCVR_NORM,				/* From local receiver, normal mode*/
    RCVR_DIAG,				/* Local receiver, diagnostic mode*/
    ALPHA_OMEGA				/* Alpha-Omega receiver		*/
} InputMode;				/********************************/


/********************************/
/*	External Functions	*/
/********************************/

Extern char	*get_can_name( char *dataname );
Extern Nat16	getIntelword( Byte *p );
Extern Nat32	getIntellong( Byte *p );
Extern Status	read_cfg( char **cfgname );
Extern double	decode_prr_chan( Int16 ival, SpecChanCal *sccp );
Extern Flt64	decode_ac9_temp( Int32 count );
#ifdef WIN32
Extern Int	getopt( Int argc, char **argv, char *fmt );
#endif


/********************************/
/*	External Data		*/
/********************************/

Extern char	*optarg;		/* Option argument from getopt()    */
Extern Int	optind;			/* Option index from getopt()	    */
Extern Analog	analog[ANALOG_CHANS];	/* Analog calibration		    */
Extern SpecCal  spec_cal;		/* Struct to hold Spect calibrations*/
Extern Int32	revision;		/* Software & msg type revision nmbr*/
Extern MBool	smallConfig;		/* TRUE for single PTT config	    */
Extern Nat32	temp_offset;		/* Offset for NO3 temperatures	    */


/********************************/
/*	Module Local Data	*/
/********************************/

#ifdef WIN32
MLocal InputMode	inputMode = RCVR_DIAG;
#else
MLocal InputMode	inputMode = SVC_ARGOS;
#endif
MLocal MBool		printChksumErrors = FALSE;
MLocal char		*cfgp = NULL;	/* Ptr to name of OASIS config file */
MLocal MBool		decode_all = TRUE;
MLocal MBool		decode_msg[NUM_MSG_TYPES];
MLocal MBool            decode_all_ptt=TRUE;
MLocal Nat32            ptt_count=0;
MLocal Nat32            decode_ptt[MAX_PTTS];
MLocal Int32		gmt_offset = 0;

MLocal Nat32		numMsgs = 0;
MLocal Nat32		msgCnt = 0;
MLocal Nat32		goodMsgs = 0;
MLocal Nat32		badMsgs = 0;
MLocal char		buffer[BUFSIZE];
MLocal ArgosInMsg	msgs[NUM_MSGS];
MLocal Int32            mType;

MLocal char		*msgNamesRev2[] =
{ "Chemical", "Noon 0m Spectro", "1030 0m Spectro", "Misc",
  "Noon 20m Spectro", "AC-9 Part 1", "AC-9 Part 2" };

MLocal char		*msgNamesRev3_4[] =
{ "Chemical", "Noon 0m Spectro", "1230 0m Spectro", "Misc",
  "Noon 20m Spectro", "1230 20m Spectro", "Shutter/SatDark",
  "Midnight 0m Spectro", "Midnight 20m Spectro", "Satlantic 1" };

MLocal char		*msgNamesRev5[] =
{ "Chemical", "Noon 0m Spectro", "1000 0m Spectro", "Misc",
  "Noon 20m Spectro", "1000 20m Spectro", "Shutter/SatDark",
  "Midnight 0m Spectro", "Midnight 20m Spectro", "Satlantic 1" };

MLocal char		*msgNamesRev6[] =
{ "B0", "B1", "B2", "B3", "B4",
  "B5", "B6", "B7", "B8", "B9" };

MLocal Int32		msgTypesRev1[] =
{ CHEM, SPEC_0M_NOON, SPEC_0M_AM, SPEC_20M, AC9_1, AC9_2, MISC };

MLocal char		*ac9ChanNames[] =
{ "a650", "a676", "a715", "c510", "c555", "c630", "a412", "a440", "a488",
  "c650", "c676", "c715", "a510", "a555", "a630", "c412", "c440", "c488" };


/********************************/
/*	Forward Declarations	*/
/********************************/

MBool	processCommandLine ( Int32 argc, char **argv );
Void	decodeFile( char *filename );
Void	printOneMsg( ArgosInMsg *msgp );


/************************************************************************/
/* Function    : main							*/
/* Purpose     : Main routine						*/
/* Inputs      : argc, argv						*/
/* Outputs     : Integer, 0 for success, 1 for failure			*/
/************************************************************************/
	Int32
main( Int32 argc, char **argv )
{
    char	*filename, *cp;
    Reg Int32	i;

    if ( !processCommandLine(argc, argv) )
	exit( 1 );

    putenv( "TZ=GMT0" );

    for ( i = optind; i < argc; i++ )
    {
	filename = argv[i];
	get_can_name( filename );
	cp = cfgp;
	if ( read_cfg(&cp) != OK )
	    printf("Can't read configuration file \"%s\"\n", cp);
	else
	    decodeFile( filename );
    }

    return( 0 );

} /* main() */


/************************************************************************/
/* Function : use_msg							*/
/* Purpose  : Print Usage Message					*/
/* Inputs   : Name of program						*/
/* Outputs  : None							*/
/************************************************************************/
	Void
use_msg( char *s )
{
    fprintf( stderr, "Usage: %s [-b] [-c cfg_file] [-i msgnum] [-d] [-n] [-a] [-g gmt_offset]\n",s );
    fprintf( stderr, "-b prints messages with checksum errors\n" );
    fprintf( stderr, "-c specifies a configuration file\n" );
    fprintf( stderr, "-i decodes a specified message number\n" );
    fprintf( stderr, "-d is for a Telonics receiver in diagnostic mode\n" );
    fprintf( stderr, "-n is for a Telonics receiver in normal mode\n" );
    fprintf( stderr, "-a is for an Alpha-Omega receiver\n" );
    fprintf( stderr, "-e is for a email received from Service Argos\n" );
    fprintf( stderr, "-g specifies the offset from GMT in hours for receiver\n");
    fprintf( stderr, "-p decodes a specified PTT number\n");
#ifdef WIN32
    fprintf( stderr, "default is for Telonics receiver in diagnostic mode\n" );
#else
    fprintf( stderr, "default is for email from Service Argos\n");
#endif
    
} /* use_msg() */


/************************************************************************/
/* Function : processCommandLine					*/
/* Purpose  : Read the arguments from the command line			*/
/* Inputs   : argc, argv from main() routine				*/
/* Outputs  : TRUE if arguments OK, else FALSE				*/
/************************************************************************/
	MBool
processCommandLine ( Int32 argc, char **argv )
{
    Int32	c, i;

    putenv( "TZ=GMT0" );

    for ( i = 0; i < NUM_MSG_TYPES; i++ )
	decode_msg[i] = FALSE;

    while ( (c = getopt(argc, argv, "abc:dei:g:np:")) != EOF )
	switch( c )
	{
	  case 'a':
	    inputMode = ALPHA_OMEGA;
	    break;

	  case 'b':
	    printChksumErrors = TRUE;
	    break;

	  case 'c':
	    cfgp = optarg;
	    break;

	  case 'd':
	    inputMode = RCVR_DIAG;
	    break;

	  case 'e':
	    inputMode = SVC_ARGOS;
	    break;

	  case 'g':
	    gmt_offset = atoi(optarg);
	    break;

	  case 'i':
	    decode_all = FALSE;
	    decode_msg[atoi(optarg)] = TRUE;
	    break;

	  case 'n':
	    inputMode = RCVR_NORM;
	    break;

	  case 'p':
	    decode_all_ptt=FALSE;
	    if(ptt_count<=MAX_PTTS)
	      decode_ptt[ptt_count++] = atoi(optarg);
	    break;

	  default:
	    use_msg( argv[0] );
	    return( FALSE );
	}

    return( TRUE );

} /* processCommandLine() */


/************************************************************************/
/* Function    : getLine						*/
/* Purpose     : Get one line of ARGOS message				*/
/* Inputs      : FILE pointer, buffer ptr				*/
/* Outputs     : MSG_OK or END_OF_FILE					*/
/************************************************************************/
	MsgRtn
getLine( FILE *fp, char * buffer )
{
    Reg Int32	len;

    buffer[0] = '\0';

    while( TRUE )
    {
	if ( fgets(buffer, BUFSIZE, fp) == NULL )
	    return( END_OF_FILE );

	if ( (len = strlen(buffer)) > 0 )
	{
	    if ( buffer[len] == '\n' )
		buffer[len] = '\0';
	    return( MSG_OK );
	}
    }

} /* getLine() */


/************************************************************************/
/* Function    : getSvcArgosMsg						*/
/* Purpose     : Get an ARGOS message in Service Argos format		*/
/* Inputs      : FILE ptr, Buffer ptr, message ptr, time ptr, number of msgs*/
/* Outputs     : Message return code					*/
/************************************************************************/
	MsgRtn
getSvcArgosMsg( FILE *fp, char *buffer, WordMsg *msgp,
	        struct tm *tmPtr, Int32 *nmsgs )
{
    Nat32		i, j, dat[4];
    Nat16 w;

    if ( getLine(fp, buffer) == END_OF_FILE )
	return( END_OF_FILE );

    if ( sscanf(buffer, " %d-%d-%d %d:%d:%d %d %x %x %x %x",
		&tmPtr->tm_year, &tmPtr->tm_mon, &tmPtr->tm_mday,
		&tmPtr->tm_hour, &tmPtr->tm_min, &tmPtr->tm_sec,
		nmsgs, &dat[0], &dat[1], &dat[2], &dat[3]) < 11 )
	return( BAD_MSG );

    tmPtr->tm_year -= 1900;
    tmPtr->tm_mon -= 1;

    for ( i = 0; i < 4; i++ )
	msgp->wd_data[i] = (Nat16)dat[i];

    for ( i = 4; i < ARGOS_WORDS; i += 4 )
    {
	if ( getLine(fp, buffer) == END_OF_FILE )
	    return( END_OF_FILE );

	if ( sscanf(buffer, " %x %x %x %x", 
		    &dat[0], &dat[1], &dat[2], &dat[3]) < 4 )
	    return( BAD_MSG );

	for ( j = 0; j < 4; j++ )
	    msgp->wd_data[i + j] = (Nat16)dat[j];
    }

#ifdef WIN32
        /* swap byte order for WIN32 */
	for(i=0;i<ARGOS_WORDS;i++){
		w=msgp->wd_data[i];
		msgp->wd_data[i] = ((w&0x00FF)<<8);
		msgp->wd_data[i]+= ((w&0xFF00)>>8);
		//printf("w %04x wd_data %04x\n",w,msgp->wd_data[i]);
	}
	//printf("end of msg\n\n");
#endif

    return( MSG_OK );

} /* getSvcArgosMsg() */


/************************************************************************/
/* Function    : getNormalMsg						*/
/* Purpose     : Get an ARGOS message in normal format for local rcvr	*/
/* Inputs      : FILE ptr, Buffer ptr, message ptr for return message	*/
/* Outputs     : Message return code					*/
/************************************************************************/
	MsgRtn
getNormalMsg( FILE *fp, char *buffer, ByteMsg *msgp, struct tm *tmPtr )
{
    Nat32		i, j, dat[8];
    Nat32 ptt;
    MBool isValidPTT=FALSE;

    if ( getLine(fp, buffer) == END_OF_FILE )
	return( END_OF_FILE );

    if ( sscanf(buffer, " %d-%d-%d %d:%d:%d ID %d", &tmPtr->tm_mon,
		    &tmPtr->tm_mday, &tmPtr->tm_year, &tmPtr->tm_hour,
		    &tmPtr->tm_min, &tmPtr->tm_sec,&ptt) < 7 )
	return( BAD_MSG );

    if(!decode_all_ptt){
      for(i=0;i<ptt_count;i++)
	if(decode_ptt[i]==ptt)
	  isValidPTT=TRUE;
      if(!isValidPTT)
	return(BAD_MSG);
    }

    tmPtr->tm_mon -= 1;
    if ( tmPtr->tm_year < YEAR0 )
	tmPtr->tm_year += 100;			/* Y2k fix 4/16/2001 klh*/

    for ( i = 0; i < ARGOS_LEN; i += 8 )
    {
	if ( getLine(fp, buffer) == END_OF_FILE )
	    return( END_OF_FILE );

	if ( sscanf(buffer, " %d %d %d %d %d %d %d %d",
		    &dat[0], &dat[1], &dat[2], &dat[3],
		    &dat[4], &dat[5], &dat[6], &dat[7]) < 8 )
	    return( BAD_MSG );

	for ( j = 0; j < 8; j++ )
	    msgp->by_data[i + j] = (Byte)dat[j];
    }

    return( MSG_OK );

} /* getNormalMsg() */


/************************************************************************/
/* Function    : getDiagnosticMsg					*/
/* Purpose     : Get an ARGOS message in diagnostic format for local rcvr*/
/* Inputs      : FILE ptr, Buffer ptr, message ptr for return message	*/
/* Outputs     : Message return code					*/
/************************************************************************/
	MsgRtn
getDiagnosticMsg( FILE *fp, char *buffer, ByteMsg *msgp, struct tm *tmPtr )
{
    Nat32		i, j, dat[8];
    Nat32 ptt;
    MBool isValidPTT=FALSE;

    if ( getLine(fp, buffer) == END_OF_FILE )
	return( END_OF_FILE );

    if ( sscanf(buffer, " %d-%d-%d %d:%d:%d ID %d", &tmPtr->tm_mon,
		    &tmPtr->tm_mday, &tmPtr->tm_year, &tmPtr->tm_hour,
		    &tmPtr->tm_min, &tmPtr->tm_sec, &ptt) < 7 )
	return( BAD_MSG );

    if(!decode_all_ptt){
      for(i=0;i<ptt_count;i++)
	if(decode_ptt[i]==ptt)
	  isValidPTT=TRUE;
      if(!isValidPTT)
	return(BAD_MSG);
    }

    tmPtr->tm_mon -= 1;
    if ( tmPtr->tm_year < YEAR0 )
	tmPtr->tm_year += 100;			/* Y2k fix 5/22/2000 rah*/
	
    if ( getLine(fp, buffer) == END_OF_FILE )
	return( END_OF_FILE );

    if ( sscanf(buffer, " %*x %*x %*x %*x %*x %*x %x %x",
		     &dat[0], &dat[1]) < 2)
	    return( BAD_MSG );

    msgp->by_data[0] = (Byte)dat[0];
    msgp->by_data[1] = (Byte)dat[1];

    for ( i = 2; i < ARGOS_LEN - 6; i += 8 )
    {
	if ( getLine(fp, buffer) == END_OF_FILE )
	    return( END_OF_FILE );

	if ( sscanf(buffer, " %x %x %x %x %x %x %x %x",
		    &dat[0], &dat[1], &dat[2], &dat[3],
		    &dat[4], &dat[5], &dat[6], &dat[7]) < 8 )
	    return( BAD_MSG );

	for ( j = 0; j < 8; j++ )
	    msgp->by_data[i + j] = (Byte)dat[j];
    }

    if ( getLine(fp, buffer) == END_OF_FILE )
	return( END_OF_FILE );

    if ( sscanf(buffer, " %x %x %x %x %x %x",
		    &dat[0], &dat[1], &dat[2], &dat[3], &dat[4], &dat[5]) < 6 )
	    return( BAD_MSG );

    for ( j = 0; j < 6; j++ )
	msgp->by_data[i + j] = (Byte)dat[j];

    getLine( fp, buffer );

    return( MSG_OK );

} /* getDiagnosticMsg() */


/************************************************************************/
/* Function    : getAlphaOmegaMsg					*/
/* Purpose     : Get an ARGOS message in Alpha-Omega Receiver format	*/
/* Inputs      : FILE ptr, Buffer ptr, message ptr for return message	*/
/* Outputs     : Message return code					*/
/* Comment     : Assumes A-O rcvr in byte format, which is not default	*/
/************************************************************************/
	MsgRtn
getAlphaOmegaMsg( FILE *fp, char *buffer, ByteMsg *msgp, struct tm *tmPtr )
{
    Nat32	pttNum, i, dat[32];
    MBool isValidPTT=FALSE;

    if ( getLine(fp, buffer) == END_OF_FILE )
	return( END_OF_FILE );

    if ( sscanf(buffer, " %d/%d/%d %d:%d:%d %d %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x",
		&tmPtr->tm_mon, &tmPtr->tm_mday, &tmPtr->tm_year,
		&tmPtr->tm_hour, &tmPtr->tm_min, &tmPtr->tm_sec, &pttNum,
		&dat[0],  &dat[1],  &dat[2],  &dat[3],  &dat[4],  &dat[5],
		&dat[6],  &dat[7],  &dat[8],  &dat[9],  &dat[10], &dat[11],
		&dat[12], &dat[13], &dat[14], &dat[15], &dat[16], &dat[17],
		&dat[18], &dat[19], &dat[20], &dat[21], &dat[22], &dat[23],
		&dat[24], &dat[25], &dat[26], &dat[27], &dat[28], &dat[29],
		&dat[30], &dat[31]) < 39 )
	return( BAD_MSG );

    if(!decode_all_ptt){
      for(i=0;i<ptt_count;i++)
	if(decode_ptt[i]==pttNum)
	  isValidPTT=TRUE;
      if(!isValidPTT)
	return(BAD_MSG);
    }

    tmPtr->tm_year -= 1900;
    tmPtr->tm_mon -= 1;

    for ( i = 0; i < 32; i++ )
	msgp->by_data[i] = (Byte)dat[i];

    return( MSG_OK );

} /* getAlhpaOmegaMsg() */


/************************************************************************/
/* Function    : msgMatch						*/
/* Purpose     : See if two ARGOS messages are the same			*/
/* Inputs      : Two Message ptrs					*/
/* Outputs     : TRUE if match, else FALSE				*/
/************************************************************************/
	MBool
msgMatch( ByteMsg *msgp1, ByteMsg *msgp2 )
{
    Int32	i;

    for ( i = 0; i < ARGOS_LEN; i++ )
	if ( msgp1->by_data[i] != msgp2->by_data[i] )
	    return( FALSE );

    return( TRUE );

} /* msgMatch() */


/************************************************************************/
/* Function    : findMatchingMessage					*/
/* Purpose     : Find a matching ARGOS message from those already received*/
/* Inputs      : Message ptr						*/
/* Outputs     : Message number of matching message, or ERROR		*/
/************************************************************************/
	Int32
findMatchingMessage( ByteMsg *msgp )
{
    Nat32	i;

    for ( i = 0; i < msgCnt; i++ )
	if ( msgMatch(msgp, &msgs[i].msg.rawb) )
	    return( i );

    return( ERROR );

} /* findMatchingMessage() */


/************************************************************************/
/* Function    : getMessages						*/
/* Purpose     : Get all incoming ARGOS messages			*/
/* Inputs      : FILE pointer						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
getMessages( FILE *fp, char *buffer )
{
    struct tm		msgTime, *tmPtr;
    time_t		msg_time_t;
    MsgRtn		state;
    ArgosUnion		msg;
    Int32		msgNum, nmsgs;

    numMsgs = msgCnt = 0;
    goodMsgs = badMsgs = 0;
    bzero( (void *)msgs, sizeof(msgs) );
    time( &msg_time_t );
    tmPtr = gmtime( &msg_time_t );
    memcpy( (void *)&msgTime, (void *)tmPtr, sizeof(struct tm) );
    msgTime.tm_isdst = 0;
    state = MSG_OK;

    while ( state != END_OF_FILE )
    {
	nmsgs = 1;

	switch( inputMode )
	{
	  case SVC_ARGOS:
	    state = getSvcArgosMsg( fp, buffer, &msg.raww, &msgTime, &nmsgs );
	    break;

	  case RCVR_NORM:
	    state = getNormalMsg( fp, buffer, &msg.rawb, &msgTime );
	    break;

	  case RCVR_DIAG:
	    state = getDiagnosticMsg( fp, buffer, &msg.rawb, &msgTime );
	    break;

	  case ALPHA_OMEGA:
	    state = getAlphaOmegaMsg( fp, buffer, &msg.rawb, &msgTime );
	    break;

	  default:
	    return;
	}

	if ( state == MSG_OK )
	{
	    msg_time_t = mktime( &msgTime );
	    msg_time_t -= 3600 * gmt_offset;

	    if ( (msgNum = findMatchingMessage(&msg.rawb)) != ERROR )
	    {
		if ( msg_time_t < msgs[msgNum].first_msg )
		    msgs[msgNum].first_msg = msg_time_t;
		if ( msg_time_t > msgs[msgNum].last_msg )
		    msgs[msgNum].last_msg = msg_time_t;
		msgs[msgNum].msg_cnt += nmsgs;
	    }
	    else
	    {
		if ( msgCnt >= NUM_MSGS )
		{
		    printOneMsg( msgs );
		    memmove( (void *)msgs, (void *)&msgs[1],
			     (NUM_MSGS - 1) * sizeof(ArgosInMsg) );
		    msgCnt--;
		}
		msgs[msgCnt].first_msg = msg_time_t;
		msgs[msgCnt].last_msg = msg_time_t;
		msgs[msgCnt].msg_cnt = nmsgs;
		memcpy( (void *)&msgs[msgCnt].msg.rawb, (void *)&msg,
		        sizeof(ArgosUnion) );
		numMsgs += nmsgs;
		msgCnt++;
	    }
	}
    }

} /* getMessages() */


/************************************************************************/
/* Function    : printTime						*/
/* Purpose     : Print time to screen					*/
/* Inputs      : Time							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printTime( time_t timeToPrint )
{
    struct tm	*tmPtr;

    tmPtr = gmtime( &timeToPrint );
    printf( "%04d/%02d/%02d %02d:%02d:%02d", tmPtr->tm_year + 1900,
	    tmPtr->tm_mon + 1, tmPtr->tm_mday, tmPtr->tm_hour,
	    tmPtr->tm_min, tmPtr->tm_sec );

} /* printTime() */


/************************************************************************/
/* Function    : printNO3						*/
/* Purpose     : Print an NO3 buffer					*/
/* Inputs      : NO3 message ptr					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printNO3( No3Msg *no3p )
{
    Nat32	i, val1, val2;
    Reg Byte	*p;

    printf("NO3   ");	
    p = no3p->no3;

    for ( i = 0; i < CHEM_SAMPLES/2; i++ )
    {
	val1 = *p++ << 4;
	val1 |= ((*p >> 4) & 0x0f);
	val2 = ((*p++ & 0x0f) << 8);
	val2 |= *p++;
	printf( "%03d   %03d   ", val1, val2 );
    }

    printf("\nTemp ");
    for ( i = 0; i < CHEM_SAMPLES; i++ )
	printf("%5.2f ", (Flt32)((Nat32)(no3p->temp[i]) + temp_offset)/20.0);

    printf( "\n" );

} /* printNO3() */


/************************************************************************/
/* Function    : printWetstar						*/
/* Purpose     : Print a Wetstar buffer					*/
/* Inputs      : Fluor message ptr					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printWetstar( FluorMsg *fp )
{
    Nat32	i, val1, val2;
    Reg Byte	*p;


    printf("0m Wetstar   ");	
    for ( i = 0; i < CHEM_SAMPLES; i++ )
	printf("%3d   ", fp->fl_0m[i]);

    printf("\n20m Wetstar   ");	
    p = fp->fl_20m;

    for ( i = 0; i < CHEM_SAMPLES/2; i++ )
    {
	val1 = *p++ << 4;
	val1 |= ((*p >> 4) & 0x0f);
	val2 = ((*p++ & 0x0f) << 8);
	val2 |= *p++;
	printf( "%3d   %3d   ", val1, val2 );
    }

    printf("\n");

} /* printWetstar() */


/************************************************************************/
/* Function    : printWetstarRev4					*/
/* Purpose     : Print a Wetstar buffer					*/
/* Inputs      : Fluor message ptr					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printWetstarRev4( Misc4Msg *fp )
{
    Nat32	i, val1, val2;
    Reg Byte	*p;
    Flt64	flt1, flt2;

    printf(" 0m Wetstar   ");	
    p = fp->fl_0m;

    for ( i = 0; i < CHEM_SAMPLES/2; i++ )
    {
	val1 = *p++ << 4;
	val1 |= ((*p >> 4) & 0x0f);
	val2 = ((*p++ & 0x0f) << 8);
	val2 |= *p++;
	flt1 = (((analog[FLUOR_CHAN].a * (Flt32)(val1 >> 2)) +
		 analog[FLUOR_CHAN].b) * analog[FLUOR_CHAN].c) +
		 analog[FLUOR_CHAN].d;
	flt2 = (((analog[FLUOR_CHAN].a * (Flt32)(val2 >> 2)) +
		 analog[FLUOR_CHAN].b) * analog[FLUOR_CHAN].c) +
		 analog[FLUOR_CHAN].d;
	printf( "%6.3f %6.3f ", flt1, flt2 );
    }

    printf("\n20m Wetstar   ");	

    for ( i = 0; i < FLUOR20_WORDS; i++ )
    {
	val1 = getIntelword( (Byte *)&fp->fl_20m[i] );
	flt1 = decode_prr_chan((Int16)val1, &spec_cal.spc_cal[2][11]);
	if ( flt1 < 20.0 )
	    printf( "%6.3f ", flt1 );
	else
	    printf(" Error ");
    }

    printf("\n");

} /* printWetstarRev4() */


/************************************************************************/
/* Function    : printIsus						*/
/* Purpose     : Print ISUS data (stored in sp_lu[] of SpecMsg)		*/
/* Inputs      : ?							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printIsus(void *msgp)
{
  Nat16 i;
  char w[2];

  printf("\nISUS   ");	
  if (mType == SPEC_0M_1000){
    for(i=0;i<(ASPEC_CHANS);i++){
      memmove((Byte *)&w[0],(Byte *)&((SpecMsg *)msgp)->sp_lu[i],2);
      if(w[0] < 0x20 || w[0] >= 0x7f)
	printf("\\x%02x",w[0]&0x0f);
      else
	printf("%c",w[0]);
      
      if(w[1] < 0x20 || w[1] >= 0x7f)
	printf("\\x%02x",w[1]&0x0f);
      else
	printf("%c",w[1]);
    }
  }else{
  if (mType == CHEM){
    for(i=0;i<(CH_UNUSED_SIZE);i++){
      memmove((Byte *)&w[0],(Byte *)&((Chem5Msg *)msgp)->ch_unused[i],2);
      if(w[0] < 0x20 || w[0] >= 0x7f)
	printf("\\x%02x",w[0]&0x0f);
      else
	printf("%c",w[0]);

      if(w[1] < 0x20 || w[1] >= 0x7f)
	printf("\\x%02x",w[1]&0x0f);
      else
	printf("%c",w[1]);
    }
  }
  }
  printf("\n");
  return;
}/*printIsus()*/

/************************************************************************/
/* Function    : printHS2						*/
/* Purpose     : Print Hobi HS2 data (stored in sp_lu[] of SAT1Msg)    	*/
/* Inputs      : ?							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printHS2(void *msgp)
{
  char w[2];

  printf("\nHS2\n");	
    memmove((Byte *)&w[0],(Byte *)&((SpecMsg *)msgp)->sp_lu[0],2);
    printf("\tSnorm1: %02X%02X ",w[0]&0xFF,w[1]&0xFF);
    memmove((Byte *)&w[0],(Byte *)&((SpecMsg *)msgp)->sp_lu[1],2);
    printf("Snorm2: %02X%02X ",w[0]&0xFF,w[1]&0xFF);
    memmove((Byte *)&w[0],(Byte *)&((SpecMsg *)msgp)->sp_lu[2],2);
    printf("Snorm3: %02X%02X\n",w[0]&0xFF,w[1]&0xFF);
    memmove((Byte *)&w[0],(Byte *)&((SpecMsg *)msgp)->sp_lu[3],2);
    printf("\tGain1: %01X Gain2: %01X Gain3: %01X\n",(w[0]>>4)&0x0F,w[0]&0x0F,(w[1]>>4)&0x0F);
    memmove((Byte *)&w[0],(Byte *)&((SpecMsg *)msgp)->sp_lu[4],1);
    printf("\tTempRaw: %01X ",w[0]&0xFF);

  printf("\n");
  return;
}/*printHS2()*/

/************************************************************************/
/* Function    : printHR3						*/
/* Purpose     : Print Hobi HR3 data (stored in sp_lu[] of SAT1Msg)    	*/
/* Inputs      : ?							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printHR3(void *msgp)
{
  char w[2];

  printf("\nHR3\n");	
    memmove((Byte *)&w[0],(Byte *)&((SpecMsg *)msgp)->sp_lu[0],2);
    printf("\t %02X%02X",w[0]&0xFF,w[1]&0xFF);
    memmove((Byte *)&w[0],(Byte *)&((SpecMsg *)msgp)->sp_lu[1],2);
    printf(" %02X%02X",w[0]&0xFF,w[1]&0xFF);
    memmove((Byte *)&w[0],(Byte *)&((SpecMsg *)msgp)->sp_lu[2],2);
    printf(" %02X%02X",w[0]&0xFF,w[1]&0xFF);
    memmove((Byte *)&w[0],(Byte *)&((SpecMsg *)msgp)->sp_lu[3],2);
    printf(" %02X%02X",w[0]&0xFF,w[1]&0xFF);
    memmove((Byte *)&w[0],(Byte *)&((SpecMsg *)msgp)->sp_lu[4],2);
    printf(" %02X%02X",w[0]&0xFF,w[1]&0xFF);
    memmove((Byte *)&w[0],(Byte *)&((SpecMsg *)msgp)->sp_lu[5],2);
    printf(" %02X%02X",w[0]&0xFF,w[1]&0xFF);
    memmove((Byte *)&w[0],(Byte *)&((SpecMsg *)msgp)->sp_lu[6],2);
    printf(" %02X%02X",w[0]&0xFF,w[1]&0xFF);

  printf("\n");
  return;
}/*printHS2()*/

/************************************************************************/
/* Function    : printChem4Msg						*/
/* Purpose     : Print the miscellaneous portion of the Chem msg for rev 4*/
/* Inputs      : Chem4 message ptr					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printChem4Msg( Chem4Msg *cp )
{
    Reg Byte	*p;
    Nat16	ipitch, iroll, itemp, idepth;
    Nat32	i;
    Int32	val1, val2;

    printf("CO2   ");	
    p = cp->ch_pco2;

    for ( i = 0; i < CHEM_SAMPLES/2; i++ )
    {
	val1 = *p++ << 4;
	val1 |= ((*p >> 4) & 0x0f);
	if ( val1 & 0x800 )
	    val1 |= 0xfffff000;				/* Sign extend	*/
	val2 = ((*p++ & 0x0f) << 8);
	val2 |= *p++;
	if ( val2 & 0x800 )
	    val2 |= 0xfffff000;				/* Sign extend	*/
	printf( "%03d   %03d   ", val1, val2 );
    }

    printf("\nNoon   PRR pitch, roll, temp, depth: ");
    ipitch = getIntelword( (Byte *)&cp->ch_prr_pitch );
    iroll = getIntelword( (Byte *)&cp->ch_prr_roll );
    itemp = getIntelword( (Byte *)&cp->ch_prr_temp );
    idepth = getIntelword( (Byte *)&cp->ch_prr_depth );

    printf("%6.1f %6.1f %7.2f %7.2f\n",
	   decode_prr_chan(ipitch, &spec_cal.spc_cal[0][10]),
	   decode_prr_chan(iroll, &spec_cal.spc_cal[0][11]),
	   decode_prr_chan(itemp, &spec_cal.spc_cal[2][8]),
	   decode_prr_chan(idepth, &spec_cal.spc_cal[2][9]) );
    
    printf("1230PM PRR pitch, roll, temp, depth: ");
    ipitch = getIntelword( (Byte *)&cp->ch_prr_pitch1230 );
    iroll = getIntelword( (Byte *)&cp->ch_prr_roll1230 );
    itemp = getIntelword( (Byte *)&cp->ch_prr_temp1230 );
    idepth = getIntelword( (Byte *)&cp->ch_prr_depth1230 );

    printf("%6.1f %6.1f %7.2f %7.2f\n",
	   decode_prr_chan(ipitch, &spec_cal.spc_cal[0][10]),
	   decode_prr_chan(iroll, &spec_cal.spc_cal[0][11]),
	   decode_prr_chan(itemp, &spec_cal.spc_cal[2][8]),
	   decode_prr_chan(idepth, &spec_cal.spc_cal[2][9]) );
    
} /* printChem4Msg() */

/************************************************************************/
/* Function    : printChem5Msg						*/
/* Purpose     : Print the miscellaneous portion of the Chem msg for r5 */
/* Inputs      : Chem5 message ptr					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printChem5Msg( Chem5Msg *cp )
{
    Reg Byte	*p;
    Nat16	itemp, idepth;
    Nat32	i;
    Int32	val1, val2;

    printf("CO2   ");	
    p = cp->ch_pco2;

    for ( i = 0; i < CHEM_SAMPLES/2; i++ )
    {
	val1 = *p++ << 4;
	val1 |= ((*p >> 4) & 0x0f);
	if ( val1 & 0x800 )
	    val1 |= 0xfffff000;				/* Sign extend	*/
	val2 = ((*p++ & 0x0f) << 8);
	val2 |= *p++;
	if ( val2 & 0x800 )
	    val2 |= 0xfffff000;				/* Sign extend	*/
	printf( "%03d   %03d   ", val1, val2 );
    }

    printf("\nNoon   temp, depth: ");
    itemp = getIntelword( (Byte *)&cp->ch_prr_temp );
    idepth = getIntelword( (Byte *)&cp->ch_prr_depth );

    printf("%7.2f %7.2f",
	   decode_prr_chan(itemp, &spec_cal.spc_cal[2][8]),
	   decode_prr_chan(idepth, &spec_cal.spc_cal[2][9]) );

    printIsus( (void *)cp );

} /* printChem5Msg() */

/************************************************************************/
/* Function    : printSpec						*/
/* Purpose     : Print a PRR Spectro message				*/
/* Inputs      : Spectro message ptr, Address tags for Ed, Lu chans	*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printSpec( SpecMsg *msgp, Int32 edAdr, Int32 luAdr, Int32 edChan, Int32 luChan )
{
    Reg Int32	i;
    Reg Nat16	ival;

    for ( i = 0; i < ASPEC_CHANS; i++ )
    {
	ival = getIntelword( (Byte *)&msgp->sp_ed[i] );
#if SPEC_VOLTS
	printf("%-5.5s %9.5f  ", spec_cal.spc_cal[edAdr][i+edChan].name,
	      decode_prr_chan(ival, NULL));
#else
	printf("%-5.5s %9.5f  ", spec_cal.spc_cal[edAdr][i+edChan].name,
	      decode_prr_chan(ival, &spec_cal.spc_cal[edAdr][i+edChan]));
#endif

	if ( (i % 4) == 3 )
	    printf("\n");
    }

    for ( i = 0; i < ASPEC_CHANS; i++ )
    {
	if ( (i % 4) == 0 )
	    printf("\n");

	ival = getIntelword( (Byte *)&msgp->sp_lu[i] );
#if SPEC_VOLTS
	printf("%-5.5s %9.5f  ", spec_cal.spc_cal[luAdr][i+luChan].name,
	       decode_prr_chan(ival, NULL));
#else
	printf("%-5.5s %9.5f  ",
	       spec_cal.spc_cal[luAdr][i+luChan].name,
	       decode_prr_chan(ival, &spec_cal.spc_cal[luAdr][i+luChan]));
#endif
    }

} /* printSpec() */

/************************************************************************/
/* Function    : printSpecR5						*/
/* Purpose     : Print a PRR Spectro message				*/
/* Inputs      : Spectro message ptr, Address tags for Ed, Lu chans	*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printSpecR5( SpecMsg *msgp, Int32 edAdr, Int32 luAdr, Int32 edChan, Int32 luChan )
{
    Reg Int32	i;
    Reg Nat16	ival;

    for ( i = 0; i < ASPEC_CHANS; i++ )
    {
	ival = getIntelword( (Byte *)&msgp->sp_ed[i] );
#if SPEC_VOLTS
	printf("%-5.5s %9.5f  ", spec_cal.spc_cal[edAdr][i+edChan].name,
	      decode_prr_chan(ival, NULL));
#else
	printf("%-5.5s %9.5f  ", spec_cal.spc_cal[edAdr][i+edChan].name,
	      decode_prr_chan(ival, &spec_cal.spc_cal[edAdr][i+edChan]));
#endif

	if ( (i % 4) == 3 )
	    printf("\n");
    }

    if(  mType == SPEC_0M_1000 )
    {
      printIsus( (void *)msgp );
    }else
    if( mType == SAT1 )
    {
      printHS2((void *)msgp);/*Lu holds HS2*/
    }else
    {
      for ( i = 0; i < ASPEC_CHANS; i++ )
      {
	  if ( (i % 4) == 0 )
	    printf("\n");

	  ival = getIntelword( (Byte *)&msgp->sp_lu[i] );
#if SPEC_VOLTS
	  printf("%-5.5s %9.5f  ", spec_cal.spc_cal[luAdr][i+luChan].name,
		 decode_prr_chan(ival, NULL));
#else
	  printf("%-5.5s %9.5f  ",
		 spec_cal.spc_cal[luAdr][i+luChan].name,
		 decode_prr_chan(ival, &spec_cal.spc_cal[luAdr][i+luChan]));
#endif
       }
    }

} /* printSpecR5() */


/************************************************************************/
/* Function    : printSpecR6						*/
/* Purpose     : Print a PRR Spectro  message				*/
/* Inputs      : Spectro message ptr */
/* Outputs     : None							*/
/************************************************************************/
	Void
printSpecR6( SpecMsg *msgp )
{
    Reg Int32	i;
    Nat32 *pNEd,*pNLu;
    Nat32	j,k,pixels[2][3];
	Byte *pIn[2], *pOut[2];
	Flt32 *pFloat[2];

	pNEd = (Nat32 *) &msgp->sp_ed[0];
	pNLu = (Nat32 *) &msgp->sp_lu[0];
	for ( i = 0; i < 3; i++ )
	{
		pixels[0][i]=0;
		pixels[1][i]=0;
		pIn[0] = (Byte *)&pNEd[i];
		pOut[0] = (Byte *)&pixels[0][i];
		pIn[1] = (Byte *)&pNLu[i];
		pOut[1] = (Byte *)&pixels[1][i];
#ifdef WIN32
		/* swap byte order for WIN32 */
		for(j=0,k=3;j<4;j++,k--){
			*(pOut[0]+j) = *(pIn[0]+k);
			*(pOut[1]+j) = *(pIn[1]+k);
		}
#else
		for(j=0,k=0;j<4;j++,k++){
			*(pOut[0]+j) = *(pIn[0]+k);
			*(pOut[1]+j) = *(pIn[1]+k);
		}
#endif	
		
		pFloat[0] = (Flt32 *)&pixels[0][i];
		pFloat[1] = (Flt32 *)&pixels[1][i];

		switch(mType){
		case SPEC_0M_DARK:
			printf( "\t%d 443 %f\n", i , *pFloat[0] );
			printf( "\t%d 490 %f\n", i , *pFloat[1] );
			break;
		case SPEC_20M_DARK:
			printf( "\t%d 510 %f\n", i , *pFloat[0] );
			printf( "\t%d 555 %f\n", i , *pFloat[1] );
			break;
		default:
			break;

		}/*end switch*/
	}/* end for i*/

	return;

} /* printSpecR6() */

/************************************************************************/
/* Function    : crcOneWord						*/
/* Purpose     : Check CRC-12 on one 16 bit word			*/
/* Inputs      : Initial CRC, word to check				*/
/* Outputs     : Resulting CRC						*/
/************************************************************************/
	Nat16
crcOneWord( Nat16 initCRC, Nat16 wordToCheck )
{
    Reg Nat32	bitCnt;
    Reg Nat16	crc;

    crc = initCRC;

    for ( bitCnt = 0; bitCnt < 16; bitCnt++ )
    {
	crc <<= 1;
	if ( wordToCheck & (0x8000 >> bitCnt) )
	    crc |= 1;
	if ( crc & CRC_MSB )
	    crc ^= CRC12;
    }

    return( crc );

} /* crcOneWord() */


/************************************************************************/
/* Function    : checkCRC						*/
/* Purpose     : Check CRC-12 on incoming message			*/
/* Inputs      : Word Message Pointer					*/
/* Outputs     : TRUE if CRC is OK, else FALSE				*/
/* Comment     : OASIS uses the standard CRC-12 polynomial 0x180f	*/
/*		 However, it puts the checksum and type information in	*/
/*		 the first word of the message, which it checks LAST	*/
/************************************************************************/
	MBool
checkCRC( WordMsg *msgp )
{
    Reg Nat32	cnt;
    Reg Nat16	crc;

    for ( cnt = 1, crc = 0; cnt < ARGOS_WORDS; cnt++ )
	crc = crcOneWord( crc, getIntelword((Byte *)&msgp->wd_data[cnt]) );

    return( crcOneWord(crc, getIntelword((Byte *)&msgp->wd_data[0])) == 0 );

} /* checkCRC() */


/************************************************************************/
/* Function    : checkSum						*/
/* Purpose     : Calculate Checksum on incoming message (Revision 1)	*/
/* Inputs      : Byte Message Pointer					*/
/* Outputs     : TRUE if checksum is OK, else FALSE			*/
/* Comment     : Revision 1 of OASIS software used a simple checksum	*/
/************************************************************************/
	MBool
checkSum( ByteMsg *msgp )
{
    Reg Nat32	cnt;
    Byte	cksum;

    for ( cnt = 2, cksum = 0; cnt < ARGOS_LEN; cnt++ )
	cksum ^= msgp->by_data[cnt];

    return( cksum == msgp->by_data[1] );

} /* checkSum() */


/************************************************************************/
/* Function    : printOneMsgRev2					*/
/* Purpose     : Print one received ARGOS message in Rev 1/2 format	*/
/* Inputs      : Message Pointer					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printOneMsgRev2( ArgosInMsg *msgp )
{
    Nat32	i, msgType, lval;
    Int32	ac9cnt;
    Flt64	flt1, flt2, flt3;
    Int16	co2;
    Nat16	val, msbs, mcp10, mcp30, fluor;
    MBool	isGood;

    if ( revision == 1 )
    {
	msgType = msgp->msg.rawb.by_data[0];
	if ( (msgType & 0x0f) != ((msgType >> 4) & 0x0f) )
	    isGood = FALSE;
	else
	    isGood = checkSum( &msgp->msg.rawb );
	msgType = msgTypesRev1[msgType & 0x0f];
    }
    else
    {
	msgType = (msgp->msg.rawb.by_data[1] >> 4) & 0x0f;
	isGood = checkCRC( &msgp->msg.raww );
    }

    if ( isGood )
	goodMsgs += msgp->msg_cnt;
    else
    {
	badMsgs += msgp->msg_cnt;
	if ( !printChksumErrors )
	    return;
    }

    if ( msgType >= NUM_MSG_TYPES_REV2 )
    {
	printf("Bad Message Type %d received %d times\n",
	       msgType, msgp->msg_cnt);
	return;
    }

    if ( !decode_all && !decode_msg[msgType] )
	return;

    printf("%s Msg rcvd %d times from ", msgNamesRev2[msgType], msgp->msg_cnt);

    printTime( msgp->first_msg );
    printf( " to " );
    printTime( msgp->last_msg );
    printf( " GMT\n" );

    if ( !isGood )
	printf("Checksum Error!\n");

    for ( i = 0; i < ARGOS_LEN; i++ )
    {
	printf("%02x ", msgp->msg.rawb.by_data[i]);
	if ( (i % 16) == 15 )
	    printf( "\n" );
    }

    switch( msgType )
    {
      case CHEM:
	printf("CO2   ");
	msbs = getIntelword( (Byte *)&msgp->msg.chem.ch_pco2_msbs );

	for ( i = 0; i < CHEM_SAMPLES; i++ )
	{
	    co2 = msgp->msg.chem.ch_pco2[i] | (((msbs >> (i + i)) & 3) << 8);
	    if ( co2 & 0x200 )
		co2 |= 0xfc00;				/* Sign extend	*/
	    printf( "%03d   ", co2 );
	}
	printf( "\n" );
	printNO3( &msgp->msg.chem.ch_no3 );
	break;

      case SPEC_0M_NOON:
	printSpec( &msgp->msg.spec0m, SPEC_0M_ED, SPEC_0M_LU, 0,
		   SPEC_NOON_LU_CHAN );
	fluor = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	flt1 = (((analog[FLUOR_CHAN].a * (Flt32)(fluor)) +
		 analog[FLUOR_CHAN].b) * analog[FLUOR_CHAN].c) +
		 analog[FLUOR_CHAN].d;
	printf( "Fluor %9.5f\n", flt1 );
	break;

      case SPEC_0M_AM:
	printSpec( &msgp->msg.spec0m_1030, SPEC_0M_ED, SPEC_0M_LU, 0,
		   SPEC_AM_LU_CHAN );
	fluor = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	flt1 = (((analog[FLUOR_CHAN].a * (Flt32)(fluor)) +
		 analog[FLUOR_CHAN].b) * analog[FLUOR_CHAN].c) +
		 analog[FLUOR_CHAN].d;
	printf( "Fluor %9.5f\n", flt1 );
	break;

      case SPEC_20M:
	printSpec( &msgp->msg.spec20m, SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	fluor = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
#if SPEC_VOLTS
	printf("Fluor %9.5f\n", decode_prr_chan(fluor, NULL));
#else
	printf("Fluor %9.5f\n",
	       decode_prr_chan(fluor, &spec_cal.spc_cal[2][12]));
#endif
	break;

      case AC9_1:
	printf("AC-9  ");
	for ( i = 0; i < AC9_1_CHANS; i++ )
	{
	    val = getIntelword( (Byte *)&msgp->msg.ac9.ac_data[i] );
	    printf( "%s %5.3f  ", ac9ChanNames[i], (Flt32)val / 1000.0 );
	    if ( (i % 6) == 5 )
		printf( "\n      " );
	}
	printf( "\n" );
	break;

      case AC9_2:
	printf("AC-9  ");
	for ( i = 0; i < AC9_2_CHANS; i++ )
	{
	    val = getIntelword( (Byte *)&msgp->msg.ac92.ac_data[i] );
	    printf( "%s %5.3f  ", ac9ChanNames[i + AC9_1_CHANS],
		    (Flt32)val / 1000.0 );
	}

	ac9cnt = getIntelword( (Byte *)&msgp->msg.ac92.ac_temp );
	flt1 = decode_ac9_temp( ac9cnt );
	printf("Temp  %5.3f", flt1);

	printf("\nAC-9 Chl   ");
	for ( i = 0; i < CHEM_SAMPLES; i++ )
	{
	    val = getIntelword( (Byte *)&msgp->msg.ac92.ac_chl[i] );
	    printf( "%5.3f ", (Flt32)val / 1000.0 );
	}

	printf( "\n" );
	break;

      case MISC:
	val = getIntelword( (Byte *)&msgp->msg.misc.ms_pco2 );
	co2 = val & 0x3ff;
	if ( co2 & 0x200 )			/* Sign extend	*/
	    co2 |= 0xfc00;
	printf( "CO2 Calibr %3d at %02d00 local\n", co2,
	        ((val >> 10) & 7) * 3 );
	    
	lval = getIntellong( (Byte *)&msgp->msg.misc.ms_oasis );
	flt1 = (((analog[TEMP_CHAN].a * (Flt32)(lval & 0x3ff)) +
		  analog[TEMP_CHAN].b) * analog[TEMP_CHAN].c) +
		  analog[TEMP_CHAN].d;
	flt2 = ((analog[OBATT_CHAN].a * (Flt32)((lval >> 10) & 0x3ff)) +
		  analog[OBATT_CHAN].b) * analog[OBATT_CHAN].c;
	flt3 = ((analog[ABATT_CHAN].a * (Flt32)((lval >> 20) & 0x3ff)) +
		  analog[ABATT_CHAN].b) * analog[ABATT_CHAN].c;

	printf("OASIS temp %5.2f C  Battery %5.2f V  Argos Batt %5.2f V\n",
	       flt1, flt2, flt3);

	if ( !smallConfig )
	{
	    mcp10 = getIntelword( (Byte *)&msgp->msg.misc.ms_mcp10 );
	    mcp30 = getIntelword( (Byte *)&msgp->msg.misc.ms_mcp30 );
#if SPEC_VOLTS
	    printf("10m MCP %9.5f  30m MCP %9.5f\n",
		   decode_prr_chan(mcp10, NULL), decode_prr_chan(mcp30, NULL));
#else
	    printf("10m MCP %9.5f  30m MCP %9.5f\n",
		   decode_prr_chan(mcp10, &spec_cal.spc_cal[2][10]),
		   decode_prr_chan(mcp30, &spec_cal.spc_cal[2][11]));
#endif
	    printNO3( &msgp->msg.misc.ms_no3 );
	}
	break;

      default:
	printf("Bad Message Type\n");
    }

    printf( "\n" );

} /* printOneMsgRev2() */


/************************************************************************/
/* Function    : printOneMsgRev3_4					*/
/* Purpose     : Print one received ARGOS message in Rev 3 or 4 format	*/
/* Inputs      : Message Pointer					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printOneMsgRev3_4( ArgosInMsg *msgp, Int32 rev )
{
    Int32	msgType;
    Nat32	i, lval;
    Flt64	flt1, flt2, flt3;
    Int16	co2;
    Nat16	val, msbs, mcp10, mcp30, misc;
    MBool	isGood;

    if ( (rev != 3) && (rev != 4) )
    {
	printf("Internal error - wrong message revision\n");
	return;
    }

    msgType = (msgp->msg.rawb.by_data[1] >> 4) & 0x0f;
    isGood = checkCRC( &msgp->msg.raww );

    if ( isGood )
	goodMsgs += msgp->msg_cnt;
    else
    {
	badMsgs += msgp->msg_cnt;
	if ( !printChksumErrors )
	    return;
    }

    if ( msgType >= ((rev == 3) ? NUM_MSG_TYPES_REV3 : NUM_MSG_TYPES_REV4) )
    {
	printf("Bad Message Type %d received %d times\n",
	       msgType, msgp->msg_cnt);
	return;
    }

    if ( !decode_all && !decode_msg[msgType] )
	return;

    printf("%s Msg rcvd %d times from ", msgNamesRev3_4[msgType], msgp->msg_cnt);

    printTime( msgp->first_msg );
    printf( " to " );
    printTime( msgp->last_msg );
    printf( " GMT\n" );

    if ( !isGood )
	printf("Checksum Error!\n");

    for ( i = 0; i < ARGOS_LEN; i++ )
    {
	printf("%02x ", msgp->msg.rawb.by_data[i]);
	if ( (i % 16) == 15 )
	    printf( "\n" );
    }

    switch( msgType )
    {
      case CHEM:
	if ( rev == 3 )
	{
	    printf("CO2   ");
	    msbs = getIntelword( (Byte *)&msgp->msg.chem.ch_pco2_msbs );
	    for ( i = 0; i < CHEM_SAMPLES; i++ )
	    {
		co2 = 
		    msgp->msg.chem.ch_pco2[i] | (((msbs >> (i + i)) & 3) << 8);
		if ( co2 & 0x200 )
		    co2 |= 0xfc00;			/* Sign extend	*/
		printf( "%03d   ", co2 );
	    }
	    printf( "\n" );
	    printNO3( &msgp->msg.chem.ch_no3 );
	}
	else
	    printChem4Msg( &msgp->msg.chem4 );
	break;

      case SPEC_0M_NOON:
	printSpec( &msgp->msg.spec0m, SPEC_0M_ED, SPEC_0M_LU, 0,
		   SPEC_NOON_LU_CHAN );
	misc = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	printf( "Err cnt %d\n", misc );
	break;

      case SPEC_0M_1230:
	printSpec( &msgp->msg.spec0m_1230, SPEC_0M_ED, SPEC_0M_LU, 0,
		   ((rev == 3) ? SPEC_1230_LU_CHAN : SPEC_1230_LU_CHAN_REV4) );
	misc = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	printf( "Total chans %d\n", misc );
	break;

      case SPEC_20M:
	printSpec( &msgp->msg.spec20m, SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	misc = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	if ( rev == 3 )
	    printf( "Err cnt %d\n", misc );
	else
#if SPEC_VOLTS
	    printf("10m MCP %9.5f\n", decode_prr_chan(misc, NULL));
#else
	    printf("10m MCP %9.5f\n",
		   decode_prr_chan(misc, &spec_cal.spc_cal[2][10]));
#endif
	break;

      case SPEC_20M_1230:
	printSpec( &msgp->msg.spec20m, SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	misc = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	if ( rev == 3 )
	    printf( "Total chans %d\n", misc );
	else
#if SPEC_VOLTS
	    printf("Noon 30m MCP %9.5f\n", decode_prr_chan(misc, NULL));
#else
	    printf("Noon 30m MCP %9.5f\n",
		   decode_prr_chan(misc, &spec_cal.spc_cal[2][14]));
#endif
	break;

      case SPEC_0M_DARK:
	printSpec( &msgp->msg.spec0m, SPEC_0M_ED, SPEC_0M_LU, 0,
		   SPEC_NOON_LU_CHAN );
	printf( "\n" );
	break;

      case SPEC_20M_DARK:
	printSpec( &msgp->msg.spec20m, SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	printf( "\n" );
	break;

      case SAT1:
	printSpec( &msgp->msg.sat1, SAT1_ED, SAT1_LU, 
		  SAT1_ED_CHAN, SAT1_LU_CHAN );
	printf( "\n" );
	break;

      case MISC:
	val = getIntelword( (Byte *)&msgp->msg.misc3.ms_pco2 );
	if ( rev == 3 )
	{
	    co2 = val & 0x3ff;
	    if ( co2 & 0x200 )			/* Sign extend	*/
		co2 |= 0xfc00;
	    misc = (val >> 10) & 7;
	}
	else
	{
	    co2 = val & 0xfff;
	    if ( co2 & 0x800 )			/* Sign extend	*/
		co2 |= 0xf000;
	    misc = (val >> 12) & 7;
	}

	printf( "CO2 Calibr %3d at %02d00 local\n", co2, 3 * misc );
	    
	lval = getIntellong( (Byte *)&msgp->msg.misc3.ms_oasis );
	flt1 = (((analog[TEMP_CHAN].a * (Flt32)(lval & 0x3ff)) +
		  analog[TEMP_CHAN].b) * analog[TEMP_CHAN].c) +
		  analog[TEMP_CHAN].d;
	flt2 = ((analog[OBATT_CHAN].a * (Flt32)((lval >> 10) & 0x3ff)) +
		  analog[OBATT_CHAN].b) * analog[OBATT_CHAN].c;
	flt3 = ((analog[ABATT_CHAN].a * (Flt32)((lval >> 20) & 0x3ff)) +
		  analog[ABATT_CHAN].b) * analog[ABATT_CHAN].c;

	printf("OASIS temp %5.2f C  Battery %5.2f V  Argos Batt %5.2f V\n",
	       flt1, flt2, flt3);

	if ( rev == 3 )
	{
	    mcp10 = getIntelword( (Byte *)&msgp->msg.misc3.ms_mcp10 );
	    mcp30 = getIntelword( (Byte *)&msgp->msg.misc3.ms_mcp30 );
#if SPEC_VOLTS
	    printf("10m MCP %9.5f  30m MCP %9.5f\n",
		   decode_prr_chan(mcp10, NULL), decode_prr_chan(mcp30, NULL));
#else
	    printf("10m MCP %9.5f  30m MCP %9.5f\n",
		   decode_prr_chan(mcp10, &spec_cal.spc_cal[2][10]),
		   decode_prr_chan(mcp30, &spec_cal.spc_cal[2][11]));
#endif
	    printWetstar( &msgp->msg.misc3.ms_wetstar );
	}
	else
	    printWetstarRev4( &msgp->msg.misc4 );
	break;

      case SHUTTR:
	printf("%u total shutter attempts (incl retries), %u opens, %u closes\n",
	       getIntelword((Byte *)&msgp->msg.shutter.sh_attempts),
	       msgp->msg.shutter.sh_opens, msgp->msg.shutter.sh_closes);
	printf("%u shutter errors, %u timeouts, %u retry-outs, %u redundant\n",
	       msgp->msg.shutter.sh_toterrs, msgp->msg.shutter.sh_errs[0],
	       msgp->msg.shutter.sh_errs[1], msgp->msg.shutter.sh_errs[2]);
        printf("Last OASIS error vector %x\n",
	       getIntelword((Byte *)&msgp->msg.shutter.sh_oasisErrs));

        for ( i = 0; i < ASPEC_CHANS; i++ )
	{
	    misc = getIntelword( (Byte *)&msgp->msg.shutter.sh_satDark[i] );
#if SPEC_VOLTS
	    printf("%-5.5s %9.5f  ", 
		   spec_cal.spc_cal[SPEC_0M_LU][i+SPEC_1230_LU_CHAN].name,
		   decode_prr_chan(misc, NULL));
#else
	    printf("%-5.5s %9.5f  ", 
		   spec_cal.spc_cal[SPEC_0M_LU][i+SPEC_1230_LU_CHAN].name,
		   decode_prr_chan(misc, 
			&spec_cal.spc_cal[SPEC_0M_LU][i+SPEC_1230_LU_CHAN]));
#endif
	    if ( (i % 4) == 3 )
		printf("\n");
	}
	printf( "\n" );

	if ( rev == 4 )
	{
	    val  = getIntelword( (Byte *)&msgp->msg.shutter.sh_ctd_temp );
	    misc = getIntelword( (Byte *)&msgp->msg.shutter.sh_ctd_cond );
	    printf("Noon CTD Temp %7.3f  Conductivity %8.4f\n",
		   ((double)val) / 1000., ((double)misc) / 10000.);
	}

	break;

      default:
	printf("Bad Message Type\n");
    }

    printf( "\n" );

} /* printOneMsgRev3_4() */

/************************************************************************/
/* Function    : printOneMsgR5						*/
/* Purpose     : Print one received ARGOS message in Rev 5 format	*/
/* Inputs      : Message Pointer					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printOneMsgR5( ArgosInMsg *msgp, Int32 rev )
{
    Int32	msgType;
    Nat32	i, lval;
    Flt64	flt1, flt2, flt3;
    Int16	co2;
    Nat16	val, misc;
    MBool	isGood;

    if ( (rev != 5) )
    {
	printf("Internal error - wrong message revision\n");
	return;
    }

    msgType = (msgp->msg.rawb.by_data[1] >> 4) & 0x0f;
    isGood = checkCRC( &msgp->msg.raww );

    if ( isGood )
	goodMsgs += msgp->msg_cnt;
    else
    {
	badMsgs += msgp->msg_cnt;
	if ( !printChksumErrors )
	    return;
    }

    if ( msgType >= NUM_MSG_TYPES_REV5 )
    {
	printf("Bad Message Type %d received %d times\n",
	       msgType, msgp->msg_cnt);
	return;
    }

    if ( !decode_all && !decode_msg[msgType] )
	return;

    printf("%s Msg rcvd %d times from ", msgNamesRev5[msgType], msgp->msg_cnt);

    printTime( msgp->first_msg );
    printf( " to " );
    printTime( msgp->last_msg );
    printf( " GMT\n" );

    if ( !isGood )
	printf("Checksum Error!\n");

    for ( i = 0; i < ARGOS_LEN; i++ )
    {
	printf("%02x ", msgp->msg.rawb.by_data[i]);
	if ( (i % 16) == 15 )
	    printf( "\n" );
    }
    mType=msgType;
    switch( msgType )
    {
      case CHEM:
	    printChem5Msg( &msgp->msg.chem5 );
	break;

      case SPEC_0M_NOON:
	printSpecR5( &msgp->msg.spec0m, SPEC_0M_ED, SPEC_0M_LU, 0,
		   SPEC_NOON_LU_CHAN );
	misc = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	printf( "Err cnt %d\n", misc );
	break;

      case SPEC_0M_1000:
	printSpecR5( &msgp->msg.spec0m_1000, SPEC_0M_ED, SPEC_0M_LU, 0,
		   ( SPEC_1000_LU_CHAN_REV5) );
	misc = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	printf( "Total chans %d\n", misc );
	break;

      case SPEC_20M:
	printSpecR5( &msgp->msg.spec20m, SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	printf( "\n" );
	break;

      case SPEC_20M_1000:
	printSpecR5( &msgp->msg.spec20m, SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	printf( "\n" );
	break;

      case SPEC_0M_DARK:
	printSpec( &msgp->msg.spec0m, SPEC_0M_ED, SPEC_0M_LU, 0,
		   SPEC_NOON_LU_CHAN );
	printf( "\n" );
	break;

      case SPEC_20M_DARK:
	printSpec( &msgp->msg.spec20m, SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	printf( "\n" );
	break;

      case SAT1:
	printSpecR5( &msgp->msg.sat1, SAT1_ED, SAT1_LU, 
		  SAT1_ED_CHAN, SAT1_LU_CHAN );
	printf( "\n" );
	break;

      case SHUTTR:
	printf("%u total shutter attempts (incl retries), %u opens, %u closes\n",
	       getIntelword((Byte *)&msgp->msg.shutter.sh_attempts),
	       msgp->msg.shutter.sh_opens, msgp->msg.shutter.sh_closes);
	printf("%u shutter errors, %u timeouts, %u retry-outs, %u redundant\n",
	       msgp->msg.shutter.sh_toterrs, msgp->msg.shutter.sh_errs[0],
	       msgp->msg.shutter.sh_errs[1], msgp->msg.shutter.sh_errs[2]);
        printf("Last OASIS error vector %x\n",
	       getIntelword((Byte *)&msgp->msg.shutter.sh_oasisErrs));

        for ( i = 0; i < ASPEC_CHANS; i++ )
	{
	    misc = getIntelword( (Byte *)&msgp->msg.shutter.sh_satDark[i] );
#if SPEC_VOLTS
	    printf("%-5.5s %9.5f  ", 
		   spec_cal.spc_cal[SPEC_0M_LU][i+SPEC_1000_LU_CHAN].name,
		   decode_prr_chan(misc, NULL));
#else
	    printf("%-5.5s %9.5f  ", 
		   spec_cal.spc_cal[SPEC_0M_LU][i+SPEC_1000_LU_CHAN].name,
		   decode_prr_chan(misc, 
			&spec_cal.spc_cal[SPEC_0M_LU][i+SPEC_1000_LU_CHAN]));
#endif
	    if ( (i % 4) == 3 )
		printf("\n");
	}
	printf( "\n" );

	if ( rev == 5 )
	{
	    val  = getIntelword( (Byte *)&msgp->msg.shutter.sh_ctd_temp );
	    misc = getIntelword( (Byte *)&msgp->msg.shutter.sh_ctd_cond );
	    printf("Noon CTD Temp %7.3f  Conductivity %8.4f\n",
		   ((double)val) / 1000., ((double)misc) / 10000.);
	}

	break;

      case MISC:
	val = getIntelword( (Byte *)&msgp->msg.misc3.ms_pco2 );
	    co2 = val & 0xfff;
	    if ( co2 & 0x800 )			/* Sign extend	*/
		co2 |= 0xf000;
	    misc = (val >> 12) & 7;

	printf( "CO2 Calibr %3d at %02d00 local\n", co2, 3 * misc );
	    
	lval = getIntellong( (Byte *)&msgp->msg.misc3.ms_oasis );
	flt1 = (((analog[TEMP_CHAN].a * (Flt32)(lval & 0x3ff)) +
		  analog[TEMP_CHAN].b) * analog[TEMP_CHAN].c) +
		  analog[TEMP_CHAN].d;
	flt2 = ((analog[OBATT_CHAN].a * (Flt32)((lval >> 10) & 0x3ff)) +
		  analog[OBATT_CHAN].b) * analog[OBATT_CHAN].c;
	flt3 = ((analog[ABATT_CHAN].a * (Flt32)((lval >> 20) & 0x3ff)) +
		  analog[ABATT_CHAN].b) * analog[ABATT_CHAN].c;

	printf("OASIS temp %5.2f C  Battery %5.2f V  Argos Batt %5.2f V\n",
	       flt1, flt2, flt3);

	    printWetstarRev4( &msgp->msg.misc4 );
	break;

      default:
	printf("Bad Message Type\n");
    }

    printf( "\n" );

} /* printOneMsgR5() */

/************************************************************************/
/* Function    : printOneMsgR6						*/
/* Purpose     : Print one received ARGOS message in Rev 6 format	*/
/* Inputs      : Message Pointer					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printOneMsgR6( ArgosInMsg *msgp, Int32 rev )
{
    Int32	msgType;
    Nat32	i,j,k,lval,*pNat32,pixels[3];
	Byte *pIn, *pOut;
    Flt64	flt1, flt2, flt3;
    Int16	co2;
    Nat16	val, misc;
    MBool	isGood;
	Flt32 *pFloat;

    if ( (rev != 6) )
    {
	printf("Internal error - wrong message revision\n");
	return;
    }

    msgType = (msgp->msg.rawb.by_data[1] >> 4) & 0x0f;
    isGood = checkCRC( &msgp->msg.raww );

    if ( isGood )
	goodMsgs += msgp->msg_cnt;
    else
    {
	badMsgs += msgp->msg_cnt;
	if ( !printChksumErrors )
	    return;
    }

    if ( msgType >= NUM_MSG_TYPES_REV5 )
    {
	printf("Bad Message Type %d received %d times\n",
	       msgType, msgp->msg_cnt);
	return;
    }

    if ( !decode_all && !decode_msg[msgType] )
	return;

    printf("%s Msg rcvd %d times from ", msgNamesRev6[msgType], msgp->msg_cnt);

    printTime( msgp->first_msg );
    printf( " to " );
    printTime( msgp->last_msg );
    printf( " GMT\n" );

    if ( !isGood )
	printf("Checksum Error!\n");

    for ( i = 0; i < ARGOS_LEN; i++ )
    {
	printf("%02x ", msgp->msg.rawb.by_data[i]);
	if ( (i % 16) == 15 )
	    printf( "\n" );
    }
    mType=msgType;
    switch( msgType )
    {
      case CHEM:
	    printChem5Msg( &msgp->msg.chem5 );
	break;

      case SPEC_0M_NOON:
	printSpecR5( &msgp->msg.spec0m, SPEC_0M_ED, SPEC_0M_LU, 0,
		   SPEC_NOON_LU_CHAN );
	misc = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	printf( "Err cnt %d\n", misc );
	break;

      case SPEC_0M_1000:
	printSpecR5( &msgp->msg.spec0m_1000, SPEC_0M_ED, SPEC_0M_LU, 0,
		   ( SPEC_1000_LU_CHAN_REV5) );
	misc = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	printf( "Total chans %d\n", misc );
	break;

      case SPEC_20M:
	printSpecR5( &msgp->msg.spec20m, SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	printf( "\n" );
	break;

      case SPEC_20M_1000:
	printSpecR5( &msgp->msg.spec20m, SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	printf( "\n" );
	break;

    case SPEC_0M_DARK:
	printf("HR3 Data:\n");	
	printSpecR6( &msgp->msg.spec0m);
	printf( "\n" );
	break;

      case SPEC_20M_DARK:
	printf("HR3 Data:\n");	
	printSpecR6( &msgp->msg.spec20m);
	printf( "\n" );
	break;

      case SAT1:
	printSpecR5( &msgp->msg.sat1, SAT1_ED, SAT1_LU, 
		  SAT1_ED_CHAN, SAT1_LU_CHAN );
	printf( "\n" );
	break;

      case SHUTTR:
	printf("%u total shutter attempts (incl retries), %u opens, %u closes\n",
	       getIntelword((Byte *)&msgp->msg.shutter.sh_attempts),
	       msgp->msg.shutter.sh_opens, msgp->msg.shutter.sh_closes);
	printf("%u shutter errors, %u timeouts, %u retry-outs, %u redundant\n",
	       msgp->msg.shutter.sh_toterrs, msgp->msg.shutter.sh_errs[0],
	       msgp->msg.shutter.sh_errs[1], msgp->msg.shutter.sh_errs[2]);
        printf("Last OASIS error vector %x\n",
	       getIntelword((Byte *)&msgp->msg.shutter.sh_oasisErrs));

		/* In Rev6, there are 3 4-Byte floats for HR3 412 nm data */
		printf("HR3 Data:\n");
		pNat32 = (Nat32 *) &(msgp->msg.shutter.sh_satDark[0]);

		for(i=0;i<3;i++){
			pixels[i]=0;
			pIn = (Byte *)&pNat32[i];
			pOut = (Byte *)&pixels[i];
#ifdef WIN32
			/* swap byte order for WIN32 */
			for(j=0,k=3;j<4;j++,k--)
				*(pOut+j) = *(pIn+k); 
#else
			for(j=0,k=0;j<4;j++,k++)
				*(pOut+j) = *(pIn+k); 
#endif				
			pFloat = (Flt32 *)&pixels[i];
			/*printf( "\t%d 412 %08X %f\n",i,pixels[i],*pFloat);*/
			printf( "\t%d 412 %f\n",i,*pFloat);
		}
		printf( "\n" );

	if ( rev == 5 || rev == 6)
	{
	    val  = getIntelword( (Byte *)&msgp->msg.shutter.sh_ctd_temp );
	    misc = getIntelword( (Byte *)&msgp->msg.shutter.sh_ctd_cond );
	    printf("Noon CTD Temp %7.3f  Conductivity %8.4f\n",
		   ((double)val) / 1000., ((double)misc) / 10000.);
	}

	break;

      case MISC:
	val = getIntelword( (Byte *)&msgp->msg.misc3.ms_pco2 );
	    co2 = val & 0xfff;
	    if ( co2 & 0x800 )			/* Sign extend	*/
		co2 |= 0xf000;
	    misc = (val >> 12) & 7;

	printf( "CO2 Calibr %3d at %02d00 local\n", co2, 3 * misc );
	    
	lval = getIntellong( (Byte *)&msgp->msg.misc3.ms_oasis );
	flt1 = (((analog[TEMP_CHAN].a * (Flt32)(lval & 0x3ff)) +
		  analog[TEMP_CHAN].b) * analog[TEMP_CHAN].c) +
		  analog[TEMP_CHAN].d;
	flt2 = ((analog[OBATT_CHAN].a * (Flt32)((lval >> 10) & 0x3ff)) +
		  analog[OBATT_CHAN].b) * analog[OBATT_CHAN].c;
	flt3 = ((analog[ABATT_CHAN].a * (Flt32)((lval >> 20) & 0x3ff)) +
		  analog[ABATT_CHAN].b) * analog[ABATT_CHAN].c;

	printf("OASIS temp %5.2f C  Battery %5.2f V  Argos Batt %5.2f V\n",
	       flt1, flt2, flt3);

	    printWetstarRev4( &msgp->msg.misc4 );
	break;

      default:
	printf("Bad Message Type\n");
    }

    printf( "\n" );

} /* printOneMsgR6() */

/************************************************************************/
/* Function    : printOneMsg						*/
/* Purpose     : Print one received ARGOS message			*/
/* Inputs      : Message Pointer					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printOneMsg( ArgosInMsg *msgp )
{
    switch( revision )
    {
      case 1:
      case 2:
	printOneMsgRev2( msgp );
	break;

      case 3:
	printOneMsgRev3_4( msgp, 3 );
	break;

      case 4:
	printOneMsgRev3_4( msgp, 4 );
	break;

      case 5:
	printOneMsgR5( msgp, 5 );
	break;
      case 6:
	printOneMsgR6( msgp, 6 );
	break;

      default:
	printf( "Bad software revision number %d\n", revision );
    }

} /* printOneMsg() */


/************************************************************************/
/* Function    : decodeFile						*/
/* Purpose     : Decode one ARGOS message file				*/
/* Inputs      : File name						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
decodeFile( char *filename )
{
    Reg Nat32	i;
    Reg FILE	*fp;
 
    if ( (fp = fopen(filename, "rb")) == (FILE *)NULL )
	printf("Cannot open %s\n", filename);
    else
    {
	getMessages( fp, buffer );
	
	for ( i = 0; i < msgCnt; i++ )
	    printOneMsg( &msgs[i] );

	printf(" %d valid messages,  %d checksum errors\n\n",
	         goodMsgs, badMsgs );
    }
    
} /* printMessages() */
