/****************************************************************************/
/* Copyright 1992 MBARI                                                     */
/****************************************************************************/
/* $Header: /home/cvs/oasis3/src/operations/src/drifter.c,v 1.1 2003/08/20 19:39:43 headley Exp $		    */
/* Summary  : Program to print position and voltage from OASIS on drifter   */
/* Filename : drifter.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 1.1 $							    */
/* Created  : 03/22/93 from m1.c					    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* $Log: drifter.c,v $
/* Revision 1.1  2003/08/20 19:39:43  headley
/* no message
/*
 * Revision 1.1  2001/06/19  13:02:10  13:02:10  oasisa (Oasis users)
 * Initial revision
 * 
*/
/****************************************************************************/

#include <stdio.h>			/* Standard I/O			    */
#include <stdlib.h>			/* For exit()			    */
#include <mbari/types.h>		/* MBARI type definitions	    */
#include <mbari/const.h>		/* MBARI constants		    */
#include <decode.h>			/* OASIS controller definitions	    */
#include <time.h>			/* Time				    */
#include <math.h>			/* For fmod()			    */
#include <ctype.h>

#define NAMESIZE	256		/* Space allocated for file names   */
#define GLINELEN	 48		/* Approx length of one GPS record  */
#define OLINELEN	 44		/* Approx length of one OASIS record*/
#define LINEBUFLEN	512		/* Size of line buffer		    */
#define NRECS		150		/* Max # OASIS records displayed    */
#define DFLT_RECS	20		/* Default # OASIS records displayed*/
#define YEAR0		70		/* Unix Epoch year (1970)	    */
#define SECS_PER_DAY	86400		/* Seconds per day		    */

typedef struct
{
    double	otime;
    double	otemp;
    double	ovolts;
    double	ocurrent;
    double	opressure;
} OASISDec;

typedef struct
{
    double	gps_time;
    double	gps_lat;
    double	gps_long;
    double	gps_std;
    int		gps_samples;
    int		gps_status;
} GPSDec;


/********************************/
/*	Global Data		*/
/********************************/

Global OASISDec oasisrecs[NRECS];	/* OASIS records		    */
Global GPSDec	gpsrecs[NRECS];		/* GPS records			    */
Global char	oasis_file[NAMESIZE];	/* Name of OASIS data file	    */
Global char	gps_file[NAMESIZE];	/* Name of GPS data file	    */
Global char	*oasis_dir = "/usr/local/oasis/drifter";
Global char	buffer[LINEBUFLEN];

Global char	*months[12] =		/* Month names			    */
{"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"};

Global Nat16	jdays[12] =		/* Days since Jan. 1 in non-leap yr */
{ 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 };


/************************************************************************/
/* Function    : julday							*/
/* Purpose     : Calculate Julian Day given year, month, day		*/
/* Inputs      : year (00 - 99), month (0 - 11), day (1 - 31)		*/
/* Outputs     : Julian day						*/
/* Comment     : Returns Julian day in range 1:366			*/
/************************************************************************/
	Int32
julday( Int32 yr, Int32 mo, Int32 day )
{
    Int32		leap;

    leap = (((yr % 4) == 0) && (mo > 1)) ? 1 : 0;
    return( jdays[mo] + day + leap );

} /* julday() */


/************************************************************************/
/* Function    : ep_days						*/
/* Purpose     : Calculate number days since Unix epoch			*/
/* Inputs      : years since 1970, Unix-type julian day (0:365)		*/
/* Outputs     : Days since 1/1/1970					*/
/* Comment     : The (year + 1)/4 term accounts for leap years, the	*/
/*		 first of which was 1972 & should be added starting '73	*/
/************************************************************************/
	Int32
ep_days( year, yday )
     int	year, yday;
{
    return( (365 * year) + ((year + 1)/4) + yday );

} /* ep_days() */


/************************************************************************/
/* function    : main							*/
/* purpose     : Main routine						*/
/* inputs      : argc, argv						*/
/* outputs     : none							*/
/************************************************************************/
	Void
main( int argc, char **argv )
{
    time_t	curtime, basetime, otm, gtm;
    struct tm	*tp;
    Int		i, j, itime, orecs, grecs, n, year, mo, yday, mday;
    FILE	*ofp, *gfp;
    Int		lat, lng;

    time( &curtime );
    tp = gmtime( &curtime );
    year = tp->tm_year;
    itime = (1000 * year) + tp->tm_yday + 1;
    grecs = DFLT_RECS;
    ofp = gfp = NULLF;

    for ( i = 1; i < argc; i++ )
    {
	if ( *argv[i] == '-' )
	    switch( argv[i][1] )
	    {
	      case 'd':
		if ( isdigit(argv[i][2]) )
		    itime = atoi( &argv[i][2] );
		else
		    itime = atoi( argv[++i] );
		if ( itime < 1000 )
		    itime += year * 1000;
		break;
	    
	      case 'f':
		i++;
		oasis_dir = argv[i];
		break;

	      case 'n':
		if ( isdigit(argv[i][2]) )
		    grecs = atoi( &argv[i][2] );
		else
		    grecs = atoi( argv[++i] );
		if ( grecs > NRECS )
		    grecs = NRECS;
		break;

	      default:
		printf("Unknown command line switch \"%s\"\n", argv[i]);
	    }
    }

    for ( i = itime; itime > i - 10; itime-- )
    {
	sprintf( oasis_file, "%s/oasis/%05d", oasis_dir, itime );
	sprintf( gps_file, "%s/gps/%05d", oasis_dir, itime );
	ofp = fopen( oasis_file, "r" );
	gfp = fopen( gps_file, "r" );
	if ( ofp != NULLF )
	    break;
    }

    orecs = grecs / 2;

    if ( ofp == NULLF )
    {
	printf( "Could not open OASIS file %s\n", oasis_file );
	orecs = 0;
    }

    if ( gfp == NULLF )
    {
	printf( "Could not open GPS file %s\n", gps_file );
	grecs = 0;
    }

    if ( ofp != NULLF )
    {
	fseek( ofp, (long)orecs * OLINELEN * (-1), SEEK_END );
	for ( i = 0; fgets(buffer, LINEBUFLEN, ofp) != NULL; )
	{
	    if ( buffer[0] == '#' )
		continue;
	    if ( sscanf(buffer, " %lg %lg %lg %lg %lg",
		   &oasisrecs[i].otime, &oasisrecs[i].otemp,
		   &oasisrecs[i].ovolts, &oasisrecs[i].ocurrent,
		   &oasisrecs[i].opressure) >= 5 )
		i++;
	}
	fclose( ofp );
	orecs = i;
    }

    if ( orecs <= 0 )
	printf( "No OASIS data in file %s\n", oasis_file );

    if ( gfp != NULLF )
    {
	fseek( gfp, (long)grecs * GLINELEN * (-1), SEEK_END );
	for ( i = 0; fgets(buffer, LINEBUFLEN, gfp) != NULL; )
	{
	    if ( buffer[0] == '#' )
		continue;
	    if ( sscanf(buffer, " %lg %lg %lg %lg %d %d", 
			&gpsrecs[i].gps_time, &gpsrecs[i].gps_lat, 
			&gpsrecs[i].gps_long, &gpsrecs[i].gps_std,
			&gpsrecs[i].gps_samples, &gpsrecs[i].gps_status) == 6 )
		i++;
	}
	fclose( gfp );
	grecs = i;
    }

    if ( grecs <= 0 )
	printf( "No GPS data in file %s\n", gps_file );

    yday = itime % 1000;
    for ( mo = 0; (yday > julday(year,mo+1,0)) && (mo < 11); mo++ )
	;				/* Loop to find month		*/

    mday = yday - julday(year,mo,0);	/* Find day of month		*/
		/* Get time_t for 00:00.  The +30 is for rounding minutes*/
    basetime = ep_days(year - YEAR0, yday) * SECS_PER_DAY + 30;

    printf("\nLatest Drifter data from %s %d, %d (Julian day %d)\n", 
	   months[mo], mday, year + 1900, yday );
    printf("Local  Batt   Can     Can     Position\n");
    printf("Time   Volts  Temp   Press\n");
    for ( i = j = 0; (i < orecs) && (j < grecs); )
    {
	otm = basetime + (SECS_PER_DAY * fmod(oasisrecs[i].otime, 1.0));
	if ( j < grecs )
	    gtm = basetime + (SECS_PER_DAY * fmod(gpsrecs[j].gps_time, 1.0));
	else
	    gtm = (time_t)0x7fffffff;

	tp = localtime(&gtm);

	if ( abs(gtm - otm) <= 300 )
	{
	    printf("%2d:%02d %6.2f %6.2f %6.2f", tp->tm_hour, tp->tm_min,
		   oasisrecs[i].ovolts, oasisrecs[i].otemp, 
		   oasisrecs[i].opressure);
	    i++;
	    lat = (Int)gpsrecs[j].gps_lat;
	    lng = (Int)gpsrecs[j].gps_long;
	    printf( "   %d:%06.3fN %d:%06.3fW", 
		   lat, (gpsrecs[j].gps_lat - (double)lat) * 60.0,
		   lng, (gpsrecs[j].gps_long - (double)lng) * 60.0 );
	    j++;
	}
	else if ( otm <= gtm )
	{
	    tp = localtime(&otm);

	    printf("%2d:%02d %6.2f %6.2f %6.2f", tp->tm_hour, tp->tm_min,
		   oasisrecs[i].ovolts, oasisrecs[i].otemp, 
		   oasisrecs[i].opressure);
	    i++;
	}
	else
	{
	    printf("%2d:%02d                     ", 
		   tp->tm_hour, tp->tm_min);
	    lat = (Int)gpsrecs[j].gps_lat;
	    lng = (Int)gpsrecs[j].gps_long;
	    printf( "   %d:%06.3fN %d:%06.3fW", 
		   lat, (gpsrecs[j].gps_lat - (double)lat) * 60.0,
		   lng, (gpsrecs[j].gps_long - (double)lng) * 60.0 );
	    j++;
	}
	printf("\n");
    }

    exit( 0 );

} /* main() */
