/******************************************************************************\
**	taskman.c
**	
**	Release:		2002/08/23
**	Modified:		2002/10/10
*****************************************************************************
**	
**	COPYRIGHT (C) 2002 PERSISTOR INSTRUMENTS INC., ALL RIGHTS RESERVED
**	
**	Developed by: Thomas P. Sullivan for Persistor Instruments Inc.
**	254-J Shore Road, Bourne, MA 02532  USA
**	tpsully@persistor.com - http://www.persistor.com
**	
**	Copyright (C) 2002 Persistor Instruments Inc.
**	All rights reserved.
**	
*****************************************************************************
**	
**	Copyright and License Information
**	
**	Persistor Instruments Inc. (hereafter, PII) grants you (hereafter,
**	Licensee) a non-exclusive, non-transferable license to use the software
**	source code contained in this single source file with hardware products
**	sold by PII. Licensee may distribute binary derivative works using this
**	software and running on PII hardware products to third parties without
**	fee or other restrictions.
**	
**	PII MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
**	SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
**	IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
**	OR NON-INFRINGEMENT. PII SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
**	LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING THE SOFTWARE OR
**	ITS DERIVATIVES.
**	
**	By using or copying this Software, Licensee agrees to abide by the
**	copyright law and all other applicable laws of the U.S. including, but
**	not limited to, export control laws, and the terms of this license. PII
**	shall have the right to terminate this license immediately by written
**	notice upon Licensee's breach of, or non-compliance with, any of its
**	terms. Licensee may be held legally responsible for any copyright
**	infringement or damages resulting from Licensee's failure to abide by
**	the terms of this license. 
**	
*****************************************************************************
**	
**	This simple application demonstrates using a CF2 to sleep until one or more
**	tasks need to be serviced.  This application is meant for engineers who need
**	to sleep for a long period and then perform some measurement.  Two approaches
**	are possible.  One uses Suspend Mode and consumes the least amount of power.
**	The other method uses the PIT and LPSTOP.  When using Suspend, power is removed
**	from the CF2 resulting in the lowest power consumption.  The MSP430 supervisor
**	turns the power back on when its internal alarm goes off.  The time to wake the
**	CF2 (maybe a few seconds) is too long for some applications.  In those situations
**	using the PIT method will provide a faster response time at the expense of more power.
**
**	To use Suspend mode, remove the comment in front of the line #define USE_SUSPEND
**	and re-compile. 
**	
**	When using either method the program will enter into a faster
**	wake mode when time is within the THRESHHOLD time. This is done so that
**	the target time for the task is not missed.  More power is consumed during
**	this period.
**	
**	When using Suspend mode the CF2 will consume 10 microamperes or less while 
**	not using Suspendmode it will consume about 500 microamperes.  This figure
**	will vary somewhat from CF2 to CF2 and of course we are not considering any
**	other hardware you may be using in the application.
**	
**	
\******************************************************************************/

#include	<cfxbios.h>		// Persistor BIOS and I/O Definitions
#include	<cfxpico.h>		// Persistor PicoDOS Definitions

#include	<assert.h>
#include	<ctype.h>
#include	<errno.h>
#include	<float.h>
#include	<limits.h>
#include	<locale.h>
#include	<math.h>
#include	<setjmp.h>
#include	<signal.h>
#include	<stdarg.h>
#include	<stddef.h>
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<time.h>

#include	<dirent.h>		// PicoDOS POSIX-like Directory Access Defines
#include	<dosdrive.h>	// PicoDOS DOS Drive and Directory Definitions
#include	<fcntl.h>		// PicoDOS POSIX-like File Access Definitions
#include	<stat.h>		// PicoDOS POSIX-like File Status Definitions
#include	<termios.h>		// PicoDOS POSIX-like Terminal I/O Definitions
#include	<unistd.h>		// PicoDOS POSIX-like UNIX Function Definitions

//#define DEBUG		//Comment this line out to prevent all the extra messages
#ifdef DEBUG
  #define	DBG(X)	X	// template:	DBG( cprintf("\n"); )
  #pragma message	("!!! "__FILE__ ": Don't ship with DEBUG compile flag set!")
#else
  #define	DBG(X)		// nothing
#endif

#define	LPMODE		FullStop	// choose: FullStop or FastStop or CPUStop

#define	LOW_CLOCK_SPEED		640L	//Lower power from a slower clock
#define	HIGH_CLOCK_SPEED	16000L	//This is the clock speed when we are awake...lowering it will save power.


typedef struct {		//A simple structure to hold info about tasks (add things like task type or priority)
	ulong taskid;		//So we can identify the task
	ulong tasktime;		//The system time in seconds when we will wake to process this task
	ulong taskperiod;	//The period until the next tasktime.  This will be added to tasktime after the task is processed.
} TASK;

TASK	tasks[3] = {0};	//I use one in this example but you could add more
short	numtasks = 1;	//Change this if there is more than one task
ulong	current_secs;
ushort	current_ticks;

// Function declarations
void LongSleep(void);
void ShortSleep(void);
bool CleanupAfterSuspend(void);

// The watcher function will be called by the PIT.  If you need to have to PIT
// do anything periodically then stick it in here, otherwise, it doesn't need
// to do anything but exist!
void watcher(void);
void watcher(void)
{
}

//#define USE_SUSPEND		//  *******  REMOVE THE COMMENT TO BUILD A REALLY LOW POWER VERSION OF THIS SOFTWARE  *******
#ifdef USE_SUSPEND
#define THRESHHOLD	30
#else
#define THRESHHOLD	13L
#endif

// ************************************************************************************************************
// This is the Suspend time in seconds...change if this is not suitable for your application.
// ************************************************************************************************************
static ulong	SuspendSecs = 20L;	
//NOTE:	SuspendSecs is the basic sleep period when using Suspend Mode. In this example we are processing 
//		tasks seperated by a minute or two.  In a real long-term logging application the task period might
//		be an hour or more.  If one hour was required you could make SuspendSecs equal to exactly one
//		hour and you would be sure to wake up when needed.  You could also play games like make SuspendSecs 
//		equal to 59 minutes worth of seconds and make THRESHHOLD 2 minutes.  This would mean that you would
//		stay awake before the event because you would easily be within the THRESHHOLD value.  But remember that
//		there is more than one task so decide on values for THRESHHOLD and SuspendSecs that work for your 
//		application.
//
//		Another point to make is that you could easily modify this code to Suspend for the exact number of
//		seconds until the next event.  The philosophy behind this application though is LOW POWER consumption
//		BUT we wake up at a rate that allows us to watch for other polled events,
// ************************************************************************************************************
enum {SleepUndefined = 0, SleepShort, SleepLong};

/******************************************************************************\
**	main
\******************************************************************************/
int main(int argc, char **argv)
	{
	short	i;
	short	result = 0;		// no errors so far
	short	NextSleep;

	// Identify the progam and build
	printf("\nProgram: %s: %s %s \n", __FILE__, __DATE__, __TIME__);
	// Identify the device and its firmware
	printf("Persistor CF%d SN:%ld   BIOS:%d.%d   PicoDOS:%d.%d\n", CFX,
		BIOSGVT.CFxSerNum, BIOSGVT.BIOSVersion, BIOSGVT.BIOSRelease, 
		BIOSGVT.PICOVersion, BIOSGVT.PICORelease);
	// Identify the arguments
	printf("\n%d Arguments:\n", argc);
	for (i = 0; i < argc; i++)
		printf("  argv[%d] = \"%s\"\n", i, argv[i]);

	TMGSetSpeed(HIGH_CLOCK_SPEED);

	//Get the current time and set the task time to be the current time + taskperiod 
	//(make it just equal to the current time to process the task right away)
	RTCGetTime(&current_secs,&current_ticks);
	//This is the number of seconds before the next task...change if this is not suitable for your application
	tasks[0].taskid = 1L;
	tasks[0].taskperiod = 60L;			//1 minute
	tasks[0].tasktime = current_secs + tasks[0].taskperiod;
	tasks[1].taskid = 2L;
	tasks[1].taskperiod = 120L;			//2 minutes
	tasks[1].tasktime = current_secs + tasks[1].taskperiod;
	tasks[2].taskid = 3L;
	tasks[2].taskperiod = 240L;			//4 minutes
	tasks[2].tasktime = current_secs + tasks[2].taskperiod;
	numtasks = 3;

//	The long PIT period
	PITSet51msPeriod(PITOff);
	PITAddChore(watcher,3);

#ifndef USE_SUSPEND
	PITSet51msPeriod(255);	//255*.051 = 13.005 seconds
#endif

	PIOClear(28);	//Shut down the Burr-Brown A/D
	PIOClear(29);	//For R216AU drive -OFF low  to turn off the MAX3222
	PIOSet(30);		//For R216AU drive -EN  high to turn off the MAX3222

	PIOMirror(22);
	PIOMirror(23);
	PIOMirror(24);
	PIOMirror(25);
	PIOMirror(26);
	PIOMirror(27);
	PIOMirror(31);
	PIOMirror(32);
	PIOMirror(33);
	PIOMirror(34);
	PIOMirror(35);
	PIOMirror(37);

	NextSleep = SleepLong;

	for(;;)
	{
		//Check tasks
		RTCGetTime(&current_secs,&current_ticks);

		for(i=0;i<numtasks;i++)
		{
			if(current_secs>=tasks[i].tasktime)
			{
				NextSleep = SleepUndefined;		//We force NextSleep to be undefined so we will re-define it!!!
				cprintf("\nProcessing Task #%lu\n",tasks[i].taskid);
				cprintf("The current seconds are: %lu\n",current_secs);
				cprintf("The date/time NOW is        : %s",asctime(RTClocaltime(&current_secs)));
				cprintf("The date/time of the TASK is: %s",asctime(RTClocaltime(&tasks[i].tasktime)));
				PITSet51msPeriod(PITOff);
#ifndef USE_SUSPEND
				PITSet51msPeriod(255);	//Back to the long value (if NOT using Suspend)
#endif
				//*************************
				//PROCESS YOUR TASK HERE!!!
				//*************************
				switch(tasks[i].taskid)	//Selection based on YOUR TASKID!!!
				{
					case 1:
						//Your code here!!!
					break;
					case 2:
						//Your code here!!!
					break;
					case 3:
						//Your code here!!!
					break;
					default:
						//Unknown ID
					break;
				}

				//Set time for next event
				tasks[i].tasktime = current_secs + tasks[i].taskperiod;
				cprintf("Task managed\n\n");

				if(NextSleep==SleepUndefined)
					NextSleep = SleepLong;

			}
			else if((tasks[i].tasktime - current_secs)< THRESHHOLD)
			{
				NextSleep = SleepShort;	//Short Sleeps win out over Long Sleeps
			}
			else
			{
				if(NextSleep==SleepUndefined)
					NextSleep = SleepLong;
			}
		}


		//This is where we decide how we should be sleeping
		switch(NextSleep)
		{
			default:
			case SleepShort:
				cprintf("Calling Short Sleep!\n");
				SCITxWaitCompletion();
				PITSet51msPeriod(PITOff);
				PITSet51msPeriod(19);	//About one second
				ShortSleep();
			break;
			case SleepLong:
				cprintf("Calling Long Sleep!\n");
				SCITxWaitCompletion();
				LongSleep();
			break;
			case SleepUndefined:
				cprintf("Sleep is undefined!!!\n");
				SCITxWaitCompletion();
			break;
		}
	}
	
// Various exit strategies
//	BIOSReset();			// full hardware reset
//	BIOSResetToPBM();		// full reset, but stay in Pesistor Boot Monitor
//	BIOSResetToPicoDOS();	// full reset, but jump to 0xE10000 (PicoDOS)
	return result;

	}	//____ main() ____//

//	****************
//	** LongSleep  **
//	****************
void LongSleep(void)
{
#ifdef USE_SUSPEND
	ushort ret;

	DBG(cprintf("We will now Suspend....");)
	SCITxWaitCompletion();

	ret = PWRSuspendTicks(SuspendSecs, true, WakeTmtOrWAKEFall);
	CleanupAfterSuspend();	//This is needed for PicoDOS 2.27 to fix a possible Suspend issue and using Wake

	if(ret != WokeFromTimeout)
	{
		// NOTE: 	We are RESETING when we wake early as a way of getting out of the program but you
		//			could also use this same feature to allow an early wake to service an event (as long as
		//			the WAKE pin specs are not exceeded).
		PITSet51msPeriod(PITOff);
		cprintf(" You made we wake...I'll RESET!!!\n");
		SCITxWaitCompletion();
		BIOSReset();			// full hardware reset
	}
	else
	{
		//Do anything we need here
		DBG(cprintf(" Awake\n");)		// tell 'em we're back
		SCITxWaitCompletion();
	}
#else	//We are not using PWRSuspendTicks in this section
	//Stick ourselves in low power mode
	DBG(cprintf("Executing LPStopCSE...");)	// tell 'em we're back
	SCITxWaitCompletion();		// let any pending UART data finish
	EIAForceOff(true);			// turn off the RS232 driver
	QSMStop();					// shut down the QSM
	CFEnable(false);			// turn off the CompactFlash card
	TPURun(false);

	TMGSetSpeed(LOW_CLOCK_SPEED);

	LPStopCSE(LPMODE);			// we're here until interrupted

	TMGSetSpeed(HIGH_CLOCK_SPEED);

	EIAForceOff(false);			// turn on the RS232 driver
	TPURun(true);
	QSMRun();					// bring back the QSM
	CFEnable(true);				// turn on the CompactFlash card
	ciflush();					// discard any garbage characters
	DBG(cprintf("Awake\n");)	// tell 'em we're back
#endif
}

//	****************
//	** ShortSleep  **
//	****************
void ShortSleep(void)
{
	DBG(cprintf("Executing LPStopCSE...");)	// tell 'em we're back
	TMGSetSpeed(LOW_CLOCK_SPEED);

	LPStopCSE(LPMODE);			// we're here until interrupted

	TMGSetSpeed(HIGH_CLOCK_SPEED);

	DBG(cprintf(" Awake (short)\n");)	// tell 'em we're back
}