/****************************************************************************/
/* Copyright 1992 MBARI                                                     */
/****************************************************************************/
/* $Header: /home/cvs/oasis3/src/operations/src/port.c,v 1.1 2003/08/20 19:39:44 headley Exp $			    */
/* Summary  : Device port control (locking, etc) for getoasis.c, gps.c	    */
/* Filename : port.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 1.1 $							    */
/* Created  : 02/25/92							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 25feb92 rah - created, from pcomm stuff				    */
/* $Log: port.c,v $
/* Revision 1.1  2003/08/20 19:39:44  headley
/* no message
/*
 * Revision 3.1  2001/06/19  13:17:32  13:17:32  oasisa (Oasis users)
 * Periodic Update 6/19/2001 (klh)
 * 
 * Revision 3.0  99/05/12  10:11:28  10:11:28  bobh (Bob Herlien)
 * Added tstring, misc changes
 * 
 * Revision 2.9  98/08/24  13:45:56  13:45:56  bobh (Bob Herlien)
 * Archiving sources after M2/M3 & Eqpac deployments of 1998
 * 
 * Revision 2.8  98/03/17  11:11:42  11:11:42  bobh (Bob Herlien)
 * Archiving sources prior to porting to DOS/Windows
 * 
 * Revision 2.5  94/12/15  10:59:45  10:59:45  hebo (Bob Herlien)
 * Accumulated minor changes, mainly due to move to tsunami
 * 
 * Revision 1.1  92/05/26  11:32:27  11:32:27  hebo (Bob Herlien)
 * Deleted #define _HPUX_SOURCE.  It's now defined in Makefile.
 * 
 * Revision 1.0  92/02/26  10:15:10  10:15:10  hebo (Bob Herlien)
 * Initial revision
 * 
*/
/****************************************************************************/

#include <stdio.h>
#include <mbariTypes.h>			/* MBARI type definitions	    */
#include <mbariConst.h>			/* MBARI constants		    */
#include <sys/types.h>			/* For umask()			    */
#include <sys/ioctl.h>			/* For ioctl()			    */
#include <termio.h>			/* For terminal I/O stuff	    */
#include <fcntl.h>			/* For open() stuff		    */
#include <errno.h>
#include <limits.h>
#include <unistd.h>

typedef int		Status;		/* Status return, normally OK or ERROR*/

#define LOCK_PATH	"/usr/spool/uucp" /* Where to put lock files	    */
#define BUFSIZE		512		/* Size of name buffer		    */


/********************************/
/*	External Functions	*/
/********************************/

Extern unsigned int	sleep();


/********************************/
/*	External Data		*/
/********************************/

Extern int	errno;			/* errno variable from errno.h	    */


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal char	nbuf[BUFSIZE];		/* Misc buffer for manip. names	    */

MLocal struct baudstruct
{
    Int		ibaud;
    Nat16	bflag;
} bauds[] =
{ {50, B50}, {75, B75}, {110, B110}, {134, B134}, {150, B150}, {200, B200},
  {300, B300}, {600, B600}, {900, _B900}, {1200, B1200}, {1800, B1800},
  {2400, B2400}, {3600, _B3600}, {4800, B4800}, {7200, _B7200}, {9600, B9600},
  {19200, B19200}, {38400, B38400}
};


/************************************************************************/
/* Function : set_baud							*/
/* Purpose  : Set baud rate into termio structure			*/
/* Inputs   : termio pointer, baud rate					*/
/* Outputs  : TRUE if OK, else FALSE					*/
/************************************************************************/
	MBool
set_baud( struct termio *tp, Int baud )
{
    Nat32	i;

    tp->c_cflag &= ~_CBAUD;

    for ( i = 0; i < (sizeof(bauds)/sizeof(struct baudstruct)); i++ )
	if ( baud == bauds[i].ibaud )
	{
	    tp->c_cflag |= bauds[i].bflag;
	    return( TRUE );
	}

    fprintf( stderr, "Invalid baud rate: %d\n", baud );
    return( FALSE );

} /* set_baud() */


/************************************************************************/
/* Function : is_locked							*/
/* Purpose  : Check lock file for a valid pid value			*/
/* Inputs   : Name of lock file						*/
/* Outputs  : TRUE if locked, FALSE if not				*/
/* Comments : Error conditions such as not being able to open the lock	*/
/*	      file or not being able to interpret the contents of the	*/
/*	      lock file cause the code to assume that the lock file is	*/
/*	      valid.							*/
/************************************************************************/
	MBool
is_locked( char *lockfile )
{
    Int		lfd, n;
    pid_t	lckpid;

    if ( access(lockfile, F_OK) )
	return( FALSE );		/* No lock file			*/

    if ( (lfd = open(lockfile, 0)) < 0 )
	return( TRUE );			/* can't open the lock file	*/

    n = read( lfd, (char *) &lckpid, sizeof(pid_t) );
    close( lfd );

    if ( n != sizeof(pid_t) )		/* If error readin lock file, 	*/
	return( TRUE );			/*  assume it's valid		*/

    if (lckpid <= 0 || lckpid > MAXPID)	/* If bad PID in lock file, 	*/
	return( TRUE );			/*   assume it's valid		*/

    if ( (kill(lckpid, 0) == ERROR) && (errno == ESRCH) )
    {		/* If the kill was unsuccessful due to an ESRCH error,  */
		/* that means the process is no longer active and the   */
		/* lock file can be safely removed.			*/
	unlink(lockfile);
	sleep(1);
	return( FALSE );
    }

    return( TRUE );		/* If kill() was successful, the process */
				/* must still be active			 */
} /* is_locked() */


/************************************************************************/
/* Function : release_port						*/
/* Purpose  : Release serial port.  Close device file and remove lock file*/
/* Inputs   : Name of device port, file descriptor, termio parameters	*/
/* Outputs  : OK or ERROR						*/
/************************************************************************/
	Status
release_port( char *name, Int fd, struct termio *parms )
{
    Int		lfd;				/* Lock file descriptor	*/
    pid_t	lockpid;			/* Lockfile PID		*/

    if ( fd != ERROR )				/* close the port 	*/
    {
	ioctl( fd, TCFLSH, 2 );			/* Flush serial port	*/
	if ( parms != (struct termio *)NULL )
	    ioctl( fd, TCSETA, parms );		/* Restore device	*/
	close( fd );				/* Close device		*/
    }

    sprintf( nbuf, "%s/LCK..%s", LOCK_PATH, name );

    if ( (lfd = open(nbuf, O_RDONLY, 0666)) < 0)
	return( ERROR );

    if ( read(lfd, (char *)&lockpid, sizeof(pid_t)) != sizeof(pid_t) )
	return( ERROR );

    close( lfd );

    if ( lockpid == getpid() )
	if ( unlink(nbuf) != ERROR )		/* remove the lock	*/
	    return( OK );

    return( ERROR );

} /* release_port() */


/************************************************************************/
/* Function : get_port							*/
/* Purpose  : Get a serial port, create a lock file, open port, set	*/
/*	      terminal characteristics					*/
/* Inputs      : Device name, open mode, new termio characteristics,	*/
/*		 place to save old termio characteristics		*/
/* Outputs  : File descriptor						*/
/* Comments : Locks and opens named serial port.			*/
/************************************************************************/
	Int
get_port( char *name, Int mode, struct termio *setup, struct termio *save )
{
    Int		lfd, fd, cmask;
    pid_t	progpid;

    sprintf( nbuf, "%s/LCK..%s", LOCK_PATH, name );
					/* create a lock file name */
    if ( is_locked(nbuf) )
    {
	fprintf( stderr, "/dev/%s is already locked\n", name );
	return( ERROR );
    }

    cmask = umask(0);
    if ( (lfd = open(nbuf, O_CREAT|O_EXCL|O_WRONLY, 0666)) < 0)
    {
	fprintf( stderr, "Can't create lockfile %s\n", nbuf );
	return( ERROR );
    }
    umask(cmask);

    progpid = getpid();
    write(lfd, (char *) &progpid, sizeof(pid_t));
    close( lfd );

    sprintf( nbuf, "/dev/%s", name );		/* open the device	*/

    if ( (fd = open(nbuf, mode)) < 0 )
    {
	fprintf( stderr, "Could not open %s\n", nbuf );
	return( ERROR );
    }

    if ( ioctl(fd, TCGETA, save) < 0 )
    {
	fprintf( stderr, "TCGETA failed on %s\n", nbuf );
	return( ERROR );
    }

    if ( ioctl(fd, TCSETA, setup) < 0 )
    {
	fprintf( stderr, "TCSETA failed on %s\n", nbuf );
	return( ERROR );
    }

    return( fd );

} /* get_port() */
